 /** 
 *  A single thread that reads data 
 **/ 
import java.io.IOException;

class NiceExptReaderThread implements Runnable{

 /** 
 *  Constructor 
 **/ 
  public
  NiceExptReaderThread(ProdConsQueue newPendingReads,
                       long newStartTimeMS,
                       LocalInterface newLocalInterface,
		       OutputLogger newLogger,
                       ReadSerializer newReadSerializer){
    this.pendingReads = newPendingReads;
    this.startTimeMS = newStartTimeMS;
    this.localInterface = newLocalInterface;
    this.logger = newLogger;
    NiceExptReaderThread.numThreads++;
    this.threadNum = NiceExptReaderThread.numThreads;
    this.readSerializer = newReadSerializer;
  }

 /** 
 *  Main thread method 
 **/ 
  public
  void run(){
    TraceEntry te = null;
    OutputLogEntry ole = null;
    long scheduledTimeMS = 0;

    te = (TraceEntry)this.pendingReads.dequeue();
    while(te != null){
      try{
        scheduledTimeMS = System.currentTimeMillis() - this.startTimeMS;
        ole = new OutputLogEntry(te.getTimeMS(),   // Trace time (ms)
                                 scheduledTimeMS,  // Scheduled time (ms)
                                 OutputLogEntry.READ_START, // Operation
                                 this.threadNum,   // Thread number
                                 te.getObjId(),    // Object ID
                                 null);            // Accept stamp
        this.logger.addEntry(ole);
        ole = this.performSingleRead(te.getEventNum(),
                                     te.getTimeMS(),
                                     te.getObjId(),
                                     te.getSize());
        this.logger.addEntry(ole);
        te = (TraceEntry)this.pendingReads.dequeue();
      }catch(IOException e){
        e.printStackTrace();
        System.err.println("" + e);
        assert(false);
      }
    }
    System.err.println("Thread " + this.threadNum + " is done!");
  }

 /** 
 *  Read the specified object 
 **/ 
  private OutputLogEntry
  performSingleRead(int eventNum,
                    long traceTimeMS,
                    ObjId objId,
                    long size) throws IOException{
    long currentTimeMS = 0;
    BodyMsg bodyMsg = null;
    OutputLogEntry ole = null;
    ImmutableBytes ib = null;
    ExptFileBytes exptFileBytes = null;
    LocalInterface li = null;

    currentTimeMS = System.currentTimeMillis() - this.startTimeMS;
    NiceExptReaderThread.indicateNewReadStart(currentTimeMS);
    bodyMsg = this.readSerializer.readObjBlock(objId, size);
    ib = bodyMsg.getBody();
    try{
      exptFileBytes = new ExptFileBytes(ib);
    }catch(InvalidExptFileBytesException e){
      e.printStackTrace();
      System.err.println("" + e);
      assert(false);
    }

    currentTimeMS = System.currentTimeMillis() - this.startTimeMS;
    NiceExptReaderThread.indicateNewReadEnd(currentTimeMS);
    ole = new OutputLogEntry(traceTimeMS,
                             currentTimeMS,
                             OutputLogEntry.READ_FINISH,
                             this.threadNum,
                             objId,
                             bodyMsg.getAcceptStamp());
    return(ole);
  }

 /** 
 *  Record read start time stats 
 **/ 
  private static synchronized void indicateNewReadStart(long timeMS){
    numReadsStarted++;
    lastReadStartTimeMS = timeMS;
  }

 /** 
 *  Record read end time stats 
 **/ 
  private static synchronized void indicateNewReadEnd(long timeMS){
    numReadsFinished++;
    lastReadFinishTimeMS = timeMS;
  }

 /** 
 *  Return the time the last read started 
 **/ 
  public static synchronized long getLastReadStartTimeMS(){
    return(lastReadStartTimeMS);
  }
  
 /** 
 *  Return the time the last read finished 
 **/ 
  public static synchronized long getLastReadFinishTimeMS(){
    return(lastReadFinishTimeMS);
  }

 /** 
 *  Return the number of reads started 
 **/ 
  public static synchronized long getNumReadsStarted(){
    return(numReadsStarted);
  }

 /** 
 *  Return the number of reads finished 
 **/ 
  public static synchronized long getNumReadsFinished(){
    return(numReadsFinished);
  }

 /** 
 *  Data members 
 **/ 
  private ProdConsQueue pendingReads;
  private long startTimeMS;
  private LocalInterface localInterface;
  private OutputLogger logger;
  private int threadNum;
  private ReadSerializer readSerializer;

  private static int numThreads = 0;
  private static long lastReadFinishTimeMS = 0;
  private static long lastReadStartTimeMS = 0;
  private static int numReadsStarted = 0;
  private static int numReadsFinished = 0;
}
