 /** 
 *  Emulate the receiver for the Nice experiments 
 **/ 
import java.io.IOException;
import java.io.EOFException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.InputStreamReader;
import java.io.FileOutputStream;
import java.io.BufferedReader;
import java.util.LinkedList;
import java.util.List;
import java.util.Collections;

public class NiceExptReceiver{

 /** 
 *  Local constants 
 **/ 
  private static final int BARRIER_SERVER_PORT = 4790;
  private static final int NUM_ENTRIES_LOGGER_BUFFERS = 10000;
  private static final int NUM_READER_THREADS = 10;
  private static final long MIN_SPACING_MS = 10;

 /** 
 *  Program staring point 
 **/ 
  public static void
  main(String[] argv){
    int nodeIdInt = 0;
    int algorithm = 0;
    String logFileName = null;
    NodeId senderNodeId = null;
    NodeId receiverNodeId = null;
    NodeId myNodeId = null;
    BufferedReader br = null;
    NiceExptReceiver niceExptReceiver = null;
    FileOutputStream logFileOS = null;
    String barrierServerName = null;
    CounterVV newStartVV = null;

    if(argv.length == 6){
      senderNodeId = new NodeId(Integer.parseInt(argv[1]));
      myNodeId = new NodeId(Integer.parseInt(argv[2]));
      algorithm = Integer.parseInt(argv[3]);
      logFileName = argv[4];
      barrierServerName = argv[5];

      try{
        logFileOS = new FileOutputStream(logFileName);
        niceExptReceiver = new NiceExptReceiver(argv[0],
                                                senderNodeId,
                                                myNodeId,
                                                algorithm,
                                                logFileOS,
                                                barrierServerName);
        System.err.println("Algorithm = ");
        Constants.printAlgorithm(algorithm);

        // Initialize the sender
        br = new BufferedReader(new InputStreamReader(System.in));
        // Step 1: Write to all objects
        newStartVV = niceExptReceiver.performInitialWrites(br, myNodeId);

        // Step 2: Set up subscriptions
        niceExptReceiver.setupInitialSubscription(senderNodeId,
                                                  myNodeId,
                                                  newStartVV);
        if(algorithm == Constants.PREFETCH){
          niceExptReceiver.setupBodySubscription(senderNodeId, myNodeId);
        }
        niceExptReceiver.allowSenderToStart();

        // Step 3: Play read trace
        niceExptReceiver.playTraceEntries(br);

        /*
          System.err.println("Experiment finished");
        */
      }catch(IOException e){
        System.err.println("" + e);
      }catch(RMINetworkException e){
        e.printStackTrace();
        System.err.println("" + e);
      }catch(RMIApplicationException e){
        e.printStackTrace();
        System.err.println("" + e);
      }finally{
        if(logFileOS != null){
          try{
            logFileOS.close();
          }catch(IOException e){
            // Can't do much here; ignore
          }
        }
      }
    }else{
      System.err.println("Usage: java NiceExptReceiver " +
                         "<configFileName> <sender node id> <my node id> " +
                         "<algorithm> <logFileName>");
    }
    // Finished the experiment
    System.exit(0);
  }

 /** 
 *  Constructor 
 **/ 
  private
  NiceExptReceiver(String configFile,
                   NodeId senderNodeId,
                   NodeId myNodeId,
                   int newAlgorithm,
                   OutputStream os,
                   String barrierServerName) throws IOException{
    Core core = null;

    this.rmiClient = new RMIClient();
    Config.readConfig(configFile);
    core = new Core(rmiClient, true, true, myNodeId);
    this.logger = new OutputLogger(os, NUM_ENTRIES_LOGGER_BUFFERS);
    this.exptController = new ReceiverController(this.rmiClient,
                                                 senderNodeId,
                                                 myNodeId,
                                                 Config.getDNS(myNodeId),
                                                 Config.getPortBody(myNodeId),
                                                 this.logger);
    this.uraNode = new URANode(core, this.exptController, this.rmiClient);
    this.startTimeMS = 0;
    this.lastReadTimeMS = -1 * MIN_SPACING_MS;
    this.algorithm = newAlgorithm;
    this.barrierClient = new BarrierClient(barrierServerName,
                                           BARRIER_SERVER_PORT,
                                           1);
  }

 /** 
 *  Write to all files that we will touch throughout the experiment 
 **/ 
  private CounterVV
  performInitialWrites(BufferedReader br, NodeId myNodeId) throws IOException{
    LocalInterface li = null;
    TraceEntry traceEntry = null;
    ExptFileBytes exptFileBytes = null;
    //OutputLogEntry ole = null;
    CounterVV newStartVV = null;
    long localClock = 0;
    AcceptStamp myWriteStamps[] = null;

    li = this.uraNode.getLocalInterface();
    newStartVV = new CounterVV(CounterVV.makeVVAllNegatives());
    myWriteStamps = new AcceptStamp[1];
    localClock = -1;
    do{
      try{
        traceEntry = new TraceEntry(br);
        if(traceEntry.getOperation() == TraceEntry.WRITE){
          /*
            // NOTE: For now, don't log this. Logging it is useful for
            // visual debugging but interferes with the log analyzer
            // Log the operation
            ole = new OutputLogEntry(Constants.BEFORE_TIME_MS,  // trace time
            Constants.BEFORE_TIME_MS,  // scheduled time
            OutputLogEntry.WRITE_START, // operation
            traceEntry.getObjId(),  // object id
            null);                  // data accept stamp
            this.logger.addEntry(ole);
          */

          // Perform the write operation
          exptFileBytes = new ExptFileBytes(0, traceEntry.getSize());
          li.write(traceEntry.getObjId(),
                   0, // Offset
                   traceEntry.getSize(),
                   traceEntry.getPriority(),
                   exptFileBytes.makeImmutableBytes(),
                   false);
          localClock++;
          myWriteStamps[0] = new AcceptStamp(localClock, myNodeId);
          newStartVV.advanceTimestamps(new AcceptVV(myWriteStamps));
        }else{
          assert(traceEntry.getOperation() == TraceEntry.SYNC);
        }
      }catch(InvalidExptFileBytesException e){
        e.printStackTrace();
        System.err.println("" + e);
        assert(false);
      }catch(InvalidTraceEntryException e){
        e.printStackTrace();
        System.err.println("" + e);
        assert(false);
      }
    }while(traceEntry.getOperation() != TraceEntry.SYNC);
    this.barrierClient.sendBarrierRequest(-1, -1);
    System.out.println("Finished stage 1");
    System.out.println("newStartVV = " + newStartVV);
    return(newStartVV);
  }

 /** 
 *  Subscribe to listen to intialization invalidates from the sender 
 **/ 
  private void
  setupInitialSubscription(NodeId senderNodeId,
                           NodeId myNodeId,
                           CounterVV startVV)
    throws IOException, RMINetworkException, RMIApplicationException{
    SubscriptionSet ss = null;

    ss = SubscriptionSet.makeSubscriptionSet(":/*");

    this.rmiClient.subscribe(ss,
                             startVV,
                             senderNodeId,
                             myNodeId,
                             Config.getDNS(myNodeId),
                             Config.getPortInval(myNodeId),
                             86400000, // 1 day
                             0,
                             0);
  }

  //-------------------------------------------------------------------------
  // Subscribe to listen to updates from the sender
  //-------------------------------------------------------------------------
  private void
  setupBodySubscription(NodeId senderNodeId, NodeId myNodeId)
    throws IOException, RMINetworkException, RMIApplicationException{
    SubscriptionSet ss = null;
    CounterVV startVV = null;

    ss = SubscriptionSet.makeSubscriptionSet(":/*");
    startVV = new CounterVV(CounterVV.makeVVAllNegatives());
    this.rmiClient.subscribeBody(ss,
                                 startVV,
                                 senderNodeId,
                                 myNodeId,
                                 Config.getDNS(myNodeId),
                                 Config.getPortNice(myNodeId),
                                 86400000); // 1 day
  }

  //-------------------------------------------------------------------------
  // Signal to the barrier server that the waiting sender may start
  //-------------------------------------------------------------------------
  private void
  allowSenderToStart(){
    this.barrierClient.sendBarrierRequest(-1, -1);
    System.out.println("Finished stage 2");
  }

  //-------------------------------------------------------------------------
  // Play a trace and perform actions accordingly
  //-------------------------------------------------------------------------
  private void
  playTraceEntries(BufferedReader br) throws IOException{
    boolean done = false;
    long exptTimeMS = 0;
    TraceEntry te = null;
    ProdConsQueue pendingReads = null;
    Thread threads[] = null;
    ReadSerializer readSerializer = null;

    this.startTimeMS = System.currentTimeMillis();
    this.exptController.startClock();
    System.err.println("startTimeMS = " + this.startTimeMS);
    pendingReads = new ProdConsQueue();
    readSerializer = new ReadSerializer(this.uraNode.getLocalInterface());

    threads = new Thread[NUM_READER_THREADS];
    for(int i = 0; i < NUM_READER_THREADS; i++){
      // Note: Thread will automatically start itself
      NiceExptReaderThread nert = null;
      nert = new NiceExptReaderThread(pendingReads,
                                      this.startTimeMS,
                                      this.uraNode.getLocalInterface(),
				      this.logger,
                                      readSerializer);
      threads[i] = new Thread(nert);
      threads[i].start();
    }

    while(!done){
      try{
        te = new TraceEntry(br);
        te = this.getSpacedOutTraceEntry(te);
        this.sleepUntilExptTimeMS(te.getTimeMS());

        exptTimeMS = System.currentTimeMillis() - this.startTimeMS;
        if(te.getOperation() == TraceEntry.READ){
          // Queue the operation so that next free thread will pick it up
          pendingReads.enqueue(te);
        }else{
          // Do nothing
          assert((te.getOperation() == TraceEntry.SYNC) ||
                 (te.getOperation() == TraceEntry.WRITE));
        }
      }catch(EOFException e){
        done = true;
        pendingReads.setDone();
        pendingReads.waitQueueEmpty();
        this.logger.flush();
	for(int i = 0; i < NUM_READER_THREADS; i++){
	    threads[i].interrupt();
	}
	System.err.println("lastReadStartTimeMS = " +
			   NiceExptReaderThread.getLastReadStartTimeMS());
	System.err.println("lastReadFinishTimeMS = " +
			   NiceExptReaderThread.getLastReadFinishTimeMS());
	System.err.println("numReadsStarted = " +
			   NiceExptReaderThread.getNumReadsStarted());
	System.err.println("numReadsFinished = " +
			   NiceExptReaderThread.getNumReadsFinished());
      }catch(InvalidTraceEntryException e){
        e.printStackTrace();
        System.out.println("" + e);
        assert(false);
      }
    }

    this.barrierClient.sendBarrierRequest(-1, -1);
    System.err.println("Finished stage 3");
  }

  //-------------------------------------------------------------------------
  // Generate an entry that is not too close to the previous in time
  //-------------------------------------------------------------------------
  private TraceEntry
  getSpacedOutTraceEntry(TraceEntry entry){
    TraceEntry newEntry = null;

    if(entry.getOperation() == TraceEntry.READ){
      if(entry.getTimeMS() >= (this.lastReadTimeMS + MIN_SPACING_MS)){
        this.lastReadTimeMS = entry.getTimeMS();
        newEntry = entry;
      }else{
        this.lastReadTimeMS += MIN_SPACING_MS;
        newEntry = new TraceEntry(entry.getEventNum(),
                                  this.lastReadTimeMS,
                                  entry.getOperation(),
                                  entry.getPriority(),
                                  entry.getSize(),
                                  entry.getObjId(),
                                  entry.getIsBound());
      }
    }else{
      newEntry = entry;
    }
    return(newEntry);
  }

  //-------------------------------------------------------------------------
  // Sleep until the specified time
  //-------------------------------------------------------------------------
  private void
  sleepUntilExptTimeMS(long eventTimeMS){
    long exptTimeMS = 0;

    exptTimeMS = System.currentTimeMillis() - this.startTimeMS;
    while(exptTimeMS < eventTimeMS){
      try{
        Thread.sleep(eventTimeMS - exptTimeMS);
      }catch(InterruptedException e){
        // This shouldn't happen; no thread calls interrupt()
        e.printStackTrace();
        System.err.println("" + e);
        assert(false);
      }
      exptTimeMS = System.currentTimeMillis() - this.startTimeMS;
    }
  }

  //-------------------------------------------------------------------------
  // Data members
  //-------------------------------------------------------------------------
  private ReceiverController exptController;
  private RMIClient rmiClient;
  private URANode uraNode;
  private long startTimeMS;
  private long lastReadTimeMS;
  private int algorithm;
  private OutputLogger logger;
  private BarrierClient barrierClient;
}