 /** 
 *  Peform the Nice experiments 
 **/ 
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.BufferedReader;
import java.io.InputStream;
import java.io.EOFException;
import java.io.FileOutputStream;
import java.io.OutputStream;

public class NiceExptSender{

 /** 
 *  Private data 
 **/ 
  private static final int BARRIER_SERVER_PORT = 4790;
  private static final int NUM_ENTRIES_LOGGER_BUFFERS = 10000;

 /** 
 *  Program staring point 
 **/ 
  public static void
  main(String[] argv){
    int algorithm = 0;
    NodeId myNodeId = null;
    NodeId receiverNodeId = null;
    BufferedReader br = null;
    String configFileName = null;
    String logFileName = null;
    String barrierServerName = null;
    FileOutputStream logFileOS = null;
    NiceExptSender niceExptSender = null;

    if(argv.length == 6){
      configFileName = argv[0];
      myNodeId = new NodeId(Integer.parseInt(argv[1]));
      receiverNodeId = new NodeId(Integer.parseInt(argv[2]));
      algorithm = Integer.parseInt(argv[3]);
      logFileName = argv[4];
      barrierServerName = argv[5];
      try{
        logFileOS = new FileOutputStream(logFileName);
        niceExptSender = new NiceExptSender(configFileName,
                                            myNodeId,
                                            receiverNodeId,
                                            algorithm,
                                            logFileOS,
                                            barrierServerName);

        System.err.println("Algorithm = ");
        Constants.printAlgorithm(algorithm);

        // Step 1: Wait for the receiver to finish writing all data
        niceExptSender.waitReceiverFinishDataWrites();

        // Step 2: Request a checkpoint and wait for its arrival
        niceExptSender.makeCheckpointRequest();
        niceExptSender.waitCheckpointArrival();

        // Step 3: Perform writes and wait for receiver to finish
        br = new BufferedReader(new InputStreamReader(System.in));
        niceExptSender.skipInitialWrites(br);
        niceExptSender.performWrites(br, algorithm);
      }catch(IOException e){
        e.printStackTrace();
        System.err.println("" + e);
        assert(false);
      }catch(RMINetworkException e){
        e.printStackTrace();
        System.err.println("" + e);
        assert(false);
      }catch(RMIApplicationException e){
        e.printStackTrace();
        System.err.println("" + e);
        assert(false);
      }finally{
        if(logFileOS != null){
          try{
            logFileOS.close();
          }catch(IOException e){
            // Can't do anything
          }
        }
      }
    }else{
      System.err.println("java NiceExptSender <configFileName> " +
                         "<my node id> <algorithm> <logFileName>");
    }
    // Complete the experiment
    System.exit(0);
  }

 /** 
 *  Constructor 
 **/ 
  private
  NiceExptSender(String configFileName,
                 NodeId newMyNodeId,
                 NodeId newReceiverNodeId,
                 int algorithm,
                 OutputStream logFileOS,
                 String barrierServerName){
    Core core = null;

    this.rmiClient = new RMIClient();
    Config.readConfig(configFileName);
    this.myNodeId = newMyNodeId;
    core = new Core(this.rmiClient, true, true, this.myNodeId);
    this.logger = new OutputLogger(logFileOS, NUM_ENTRIES_LOGGER_BUFFERS);
    this.senderController = new SenderController(logger);
    this.uraNode = new URANode(core, this.senderController, this.rmiClient);
    this.barrierClient = new BarrierClient(barrierServerName,
                                           BARRIER_SERVER_PORT,
                                           0);
    this.receiverNodeId = newReceiverNodeId;
  }

 /** 
 *  Wait for the receiver to finish writing all objects 
 **/ 
  private void
  waitReceiverFinishDataWrites(){
    this.barrierClient.sendBarrierRequest(-1, -1);
    System.out.println("Finished stage 1");
  }

 /** 
 *  Request the receiver to send a checkpoint 
 **/ 
  private void
  makeCheckpointRequest()
    throws IOException, RMINetworkException, RMIApplicationException{
    CounterVV startVV = null;

    startVV = new CounterVV(CounterVV.makeVVAllNegatives());
    this.rmiClient.subscribeCheckpoint("/",
                                       new String[0],
                                       startVV,
                                       this.receiverNodeId,
                                       this.myNodeId,
                                       Config.getDNS(this.myNodeId),
                                       Config.getPortCP(this.myNodeId),
                                       false);
  }

 /** 
 *  Wait for the first checkpoint to arrive from the receiver 
 **/ 
  private void
  waitCheckpointArrival(){
    // Wait for the controller to be notified about having received a CP
    // Tell the barrier server
    this.senderController.waitAppliedCheckpoint();
    assert(this.senderController.checkpointSuccessful());
    this.barrierClient.sendBarrierRequest(-1, -1);
    System.out.println("Finished stage 2");
  }

 /** 
 *  Skip over the initial writes in the trace 
 *  
 *  The initial writes in the trace - until the SYNC - are intended 
 *  to be done by the receiver to populate its own cache. We should 
 *  ignore those by skipping over them. 
 **/ 
  private void
  skipInitialWrites(BufferedReader br) throws IOException{
    TraceEntry traceEntry = null;

    try{
      do{
        traceEntry = new TraceEntry(br);
        assert((traceEntry.getOperation() == TraceEntry.WRITE) ||
               (traceEntry.getOperation() == TraceEntry.SYNC));
      }while(traceEntry.getOperation() != TraceEntry.SYNC);
    }catch(InvalidTraceEntryException e){
      e.printStackTrace();
      System.err.println("" + e);
      assert(false);
    }
  }

 /** 
 *  Read a trace file and generate events accordingly 
 **/ 
  private void
  performWrites(BufferedReader br, int algorithm) throws IOException{
    LocalInterface li = null;
    boolean done = false;
    TraceEntry traceEntry = null;
    long startTimeMS = 0;
    long exptTimeMS = 0;
    ExptFileBytes exptFileBytes = null;
    boolean useBoundInvals = false;
    OutputLogEntry ole = null;

    useBoundInvals = (algorithm == Constants.PUSH_ALL);
    li = this.uraNode.getLocalInterface();
    startTimeMS = System.currentTimeMillis();
    this.senderController.startClock();
    while(!done){
      try{
        do{
          traceEntry = new TraceEntry(br);
        }while(traceEntry.getOperation() != TraceEntry.WRITE);

        // Found a write operation; wait until we can perform it
        exptTimeMS = System.currentTimeMillis() - startTimeMS;
        while(exptTimeMS < traceEntry.getTimeMS()){
          try{
            Thread.sleep(traceEntry.getTimeMS() - exptTimeMS);
          }catch(InterruptedException e){
            e.printStackTrace();
            System.err.println("" + e);
            assert(false);
          }
          exptTimeMS = System.currentTimeMillis() - startTimeMS;
        }

        // Time to write
        exptFileBytes = new ExptFileBytes(0, traceEntry.getSize());
        ole = new OutputLogEntry(traceEntry.getTimeMS(),  // trace time (ms)
                                 exptTimeMS,              // scheduled time(ms)
                                 OutputLogEntry.WRITE_START, // operation
                                 0,                       // thread number
                                 traceEntry.getObjId(),   // objId
                                 null);                   // data accept stamp
        this.logger.addEntry(ole);
        li.write(traceEntry.getObjId(),
                 0,
                 traceEntry.getSize(),
                 traceEntry.getPriority(),
                 exptFileBytes.makeImmutableBytes(),
                 useBoundInvals);
      }catch(EOFException e){
        done = true;
        this.logger.flush();
      }catch(InvalidExptFileBytesException e){
        e.printStackTrace();
        System.err.println("" + e);
        assert(false);
      }catch(InvalidTraceEntryException e){
        e.printStackTrace();
        System.err.println("" + e);
        assert(false);
      }
    }
    this.barrierClient.sendBarrierRequest(-1, -1);
    System.out.println("Finished stage 3");
  }

 /** 
 *  Private data 
 **/ 
  private SenderController senderController;
  private URANode uraNode;
  private OutputLogger logger;
  private BarrierClient barrierClient;
  private RMIClient rmiClient;
  private NodeId myNodeId;
  private NodeId receiverNodeId;
}
