 /** 
 *  Code to parse/store an entry in log files printed by the Nice experiments. 
 **/ 
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.BufferedReader;
import java.io.Reader;
import java.io.EOFException;
import java.io.PrintStream;
import java.io.OutputStream;
import java.util.StringTokenizer;

// Used for testing
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;

public class OutputLogEntry implements Immutable, Cloneable{

 /** 
 *  Constants 
 **/ 
  public static final int READ_START = 1;
  public static final int READ_FINISH = 2;
  public static final int INV_ARRIV = 3;
  public static final int BND_ARRIV = 4;
  public static final int DEMAND_REQ = 5;
  public static final int DEMAND_ARRIV = 6;
  public static final int PUSH_ARRIV = 7;
  public static final int WRITE_START = 8;
  public static final int WRITE_FINISH = 9;

 /** 
 *  Constructor 
 **/ 
  public
  OutputLogEntry(long newTraceTimeMS,
                 long newScheduledTimeMS,
                 int newOperation,
                 int newThreadNum,
                 ObjId newObjId,
                 AcceptStamp newDataAcceptStamp){
    assert((newOperation >= READ_START) || (newOperation <= WRITE_FINISH));
    this.traceTimeMS = newTraceTimeMS;
    this.scheduledTimeMS = newScheduledTimeMS;
    this.operation = newOperation;
    this.threadNum = newThreadNum;
    this.objId = (ObjId)newObjId.clone();
    if(newDataAcceptStamp != null){
      this.dataAcceptStamp = (AcceptStamp)newDataAcceptStamp.clone();
    }else{
      this.dataAcceptStamp = null;
    }
  }

 /** 
 *  Constructor 
 **/ 
  public
  OutputLogEntry(BufferedReader br)
    throws IOException, InvalidOutputLogEntryException{
    String line = null;
    StringTokenizer st = null;

    line = br.readLine();
    if(line == null){
      throw(new EOFException());
    }
    st = new StringTokenizer(line);
    this.traceTimeMS = OutputLogEntry.parseTraceTimeMS(st);
    this.scheduledTimeMS = OutputLogEntry.parseScheduledTimeMS(st);
    this.operation = OutputLogEntry.parseOperationType(st);
    this.threadNum = OutputLogEntry.parseThreadNum(st);
    this.objId = OutputLogEntry.parseObjId(st);
    this.dataAcceptStamp = OutputLogEntry.parseDataAcceptStamp(st);
  }

 /** 
 *  Write this object to an output stream 
 **/ 
  public void
  writeToOS(OutputStream os) throws IOException{
    PrintStream ps = null;

    assert((this.operation >= OutputLogEntry.READ_START) ||
           (this.operation <= OutputLogEntry.WRITE_FINISH));
    ps = new PrintStream(os);
    ps.print("" + this.traceTimeMS + "\t");
    ps.print("" + this.scheduledTimeMS + "\t");
    this.printOperation(ps);
    ps.print("\t");
    ps.print("" + this.threadNum + "\t");
    ps.print("\"" + this.objId.getPath() + "\"\t");
    this.printDataAcceptStamp(ps);
    ps.println();
    ps.flush();
  }

 /** 
 *  Return the trace time of the event 
 **/ 
  public long
  getTraceTimeMS(){
    return(this.traceTimeMS);
  }

 /** 
 *  Return the scheduled time of the event 
 **/ 
  public long
  getScheduledTimeMS(){
    return(this.scheduledTimeMS);
  }

 /** 
 *  Return the operation type 
 **/ 
  public int
  getOperation(){
    return(this.operation);
  }

 /** 
 *  Return the operation type 
 **/ 
  public int
  getThreadNum(){
    return(this.threadNum);
  }

 /** 
 *  Return the time of the event 
 **/ 
  public ObjId
  getObjId(){
    return(this.objId);
  }

 /** 
 *  Return the data event number 
 **/ 
  public AcceptStamp
  getDataAcceptStamp(){
    return(this.dataAcceptStamp);
  }

 /** 
 *  Return true if "o" equals this 
 **/ 
  public boolean
  equals(Object o){
    OutputLogEntry ole = null;
    boolean result = false;
    boolean dataStampsEqual = false;

    // 9 constants, 1 java-provided fields, 6 declared fields
    assert(this.getClass().getDeclaredFields().length == 16);

    if(o instanceof OutputLogEntry){
      ole = (OutputLogEntry)o;
      // Either both are null, or equal. The top line checks for nullness.
      dataStampsEqual = ((this.dataAcceptStamp == ole.dataAcceptStamp) ||
                         (this.dataAcceptStamp.equals(ole.dataAcceptStamp)));
      result = ((this.traceTimeMS == ole.traceTimeMS) &&
                (this.scheduledTimeMS == ole.scheduledTimeMS) &&
                (this.operation == ole.operation) &&
                (this.threadNum == ole.threadNum) &&
                (this.objId.equals(ole.objId)) &&
                dataStampsEqual);
    }else{
      result = false;
    }
    return(result);
  }

 /** 
 *  Clone this object 
 **/ 
  public Object
  clone(){
    // Because this object is immutable, we do not need to do anything
    return(this);
  }

 /** 
 *  Return a hashCode (note: doesn't currently do anything) 
 **/ 
  public int
  hashCode(){
    assert(false);
    return(0);
  }

 /** 
 *  Used for testing 
 **/ 
  public static void
  main(String[] argv){
    if(argv.length < 1){
      Env.verifyAssertEnabled();
      System.out.println("Testing OutputLogEntry.java...");
      OutputLogEntry.test1();
      OutputLogEntry.test2();
      OutputLogEntry.test3();
      System.out.println("...Finished");
    }else{
      OutputLogEntry.readLoopTest();
    }
  }

 /** 
 *  Test OutputLogEntry 
 **/ 
  private static void
  test1(){
    OutputLogEntry ole1 = null;
    OutputLogEntry ole2 = null;
    ByteArrayOutputStream baos = null;
    ByteArrayInputStream bais = null;
    BufferedReader br = null;
    long timeMS = 0;
    AcceptStamp as = null;

    try{
      baos = new ByteArrayOutputStream();
      timeMS = System.currentTimeMillis();
      as = new AcceptStamp(200, new NodeId(10));
      ole1 = new OutputLogEntry(timeMS,           // trace time (ms)
                                timeMS - 100,     // scheduled time (ms)
                                READ_START,       // event operation
                                1,                // thread num
                                new ObjId("/0"),  // object id
                                as);              // data accept stamp
      ole1.writeToOS(baos);
      assert(ole1.equals(ole1));
      //ole1.writeToOS(System.out); // Already tested

      bais = new ByteArrayInputStream(baos.toByteArray());
      br = new BufferedReader(new InputStreamReader(bais));
      ole2 = new OutputLogEntry(br);
      //ole2.writeToOS(System.out); // Already tested
      try{
        ole2 = new OutputLogEntry(br);
        assert(false);
      }catch(EOFException e){
        // Do nothing; this was expected.
      }
      assert(ole1 != ole2);
      assert(ole1.equals(ole2));
    }catch(IOException e){
      assert(false);
    }catch(InvalidOutputLogEntryException e){
      System.err.println("" + e);
      assert(false);
    }
  }

 /** 
 *  Test OutputLogEntry 2 
 **/ 
  private static void
  test2(){
    OutputLogEntry ole = null;
    long timeMS = 0;
    AcceptStamp as = null;

    timeMS = System.currentTimeMillis();
    as = new AcceptStamp(101, new NodeId(500));
    ole = new OutputLogEntry(timeMS,            // trace time (ms)
                             timeMS - 100,      // scheduled time (ms)
                             WRITE_FINISH,      // operation type
                             2,                 // thread num
                             new ObjId("/0"),   // obj id
                             as);               // data accept stamp
    assert(ole.getTraceTimeMS() == ole.traceTimeMS);
    assert(ole.getScheduledTimeMS() == ole.scheduledTimeMS);
    assert(ole.getOperation() == ole.operation);
    assert(ole.getThreadNum() == ole.threadNum);
    assert(ole.getObjId().equals(ole.objId));
    assert(ole.getDataAcceptStamp().equals(ole.dataAcceptStamp));
  }

 /** 
 *  Test OutputLogEntry 3 
 **/ 
  private static void
  test3(){
    OutputLogEntry ole1 = null;
    OutputLogEntry ole2 = null;
    ByteArrayOutputStream baos = null;
    ByteArrayInputStream bais = null;
    BufferedReader br = null;
    long timeMS = 0;

    try{
      baos = new ByteArrayOutputStream();
      timeMS = System.currentTimeMillis();
      ole1 = new OutputLogEntry(timeMS,           // trace time (ms)
                                timeMS - 700,     // scheduled time (ms)
                                WRITE_FINISH,     // event operation
                                3,                // thread num
                                new ObjId("/0"),  // object id
                                null);            // data accept stamp
      ole1.writeToOS(baos);
      assert(ole1.equals(ole1));
      //ole1.writeToOS(System.out); // Already tested

      bais = new ByteArrayInputStream(baos.toByteArray());
      br = new BufferedReader(new InputStreamReader(bais));
      ole2 = new OutputLogEntry(br);
      //ole2.writeToOS(System.out); // Already tested
      try{
        ole2 = new OutputLogEntry(br);
        assert(false);
      }catch(EOFException e){
        // Do nothing; this was expected.
      }
      assert(ole1 != ole2);
      assert(ole1.equals(ole2));
    }catch(IOException e){
      assert(false);
    }catch(InvalidOutputLogEntryException e){
      System.err.println("" + e);
      assert(false);
    }
  }

 /** 
 *  Try reading a file from stdin 
 **/ 
  private static void
  readLoopTest(){
    OutputLogEntry ole = null;
    BufferedReader reader = null;

    try{
      reader = new BufferedReader(new InputStreamReader(System.in));
      while(true){
        ole = new OutputLogEntry(reader);
        ole.writeToOS(System.out);
      }
    }catch(EOFException e){
      System.out.println("Done!");
    }catch(InvalidOutputLogEntryException e){
      System.out.println("" + e);
      assert(false);
    }catch(IOException e){
      System.out.println("" + e);
      assert(false);
    }
  }

 /** 
 *  Write the operation to the given PrintStream 
 **/ 
  private void
  printOperation(PrintStream ps) throws IOException{
    switch(this.operation){
    case READ_START:
      ps.print("READ_START");
      break;
    case READ_FINISH:
      ps.print("READ_FINISH");
      break;
    case INV_ARRIV:
      ps.print("INV_ARRIV");
      break;
    case BND_ARRIV:
      ps.print("BND_ARRIV");
      break;
    case DEMAND_REQ:
      ps.print("DEMAND_REQ");
      break;
    case DEMAND_ARRIV:
      ps.print("DEMAND_ARRIV");
      break;
    case PUSH_ARRIV:
      ps.print("PUSH_ARRIV");
      break;
    case WRITE_START:
      ps.print("WRITE_START");
      break;
    case WRITE_FINISH:
      ps.print("WRITE_FINISH");
      break;
    default:
      assert(false);
      break;
    }
  }

 /** 
 *  Write the accept stamp to the given PrintStream 
 **/ 
  private void
  printDataAcceptStamp(PrintStream ps) throws IOException{
    String nodeIdString = null;
    long localClock = 0;

    if(this.dataAcceptStamp != null){
      nodeIdString = "" + this.dataAcceptStamp.getNodeId();
      localClock = this.dataAcceptStamp.getLocalClock();
      ps.print("(" + nodeIdString + ":" + localClock + ")");
    }else{
      ps.print("null");
    }
  }

 /** 
 *  Parse the trace time in milliseconds 
 **/ 
  private static long
  parseTraceTimeMS(StringTokenizer st)
    throws IOException, InvalidOutputLogEntryException{
    String token = null;
    long readTraceTimeMS = 0;
    boolean badTimeMS = false;

    // Read the trace time in ms
    if(st.hasMoreTokens()){
      token = st.nextToken();
      try{
        readTraceTimeMS = Long.parseLong(token);
      }catch(NumberFormatException nfe){
        badTimeMS = true;
      }
    }else{
      badTimeMS = true;
    }
    if(badTimeMS){
      throw(new InvalidOutputLogEntryException("The first field should be " +
                                               "the trace time in ms!"));
    }
    return(readTraceTimeMS);
  }

 /** 
 *  Parse the scheduled time in milliseconds 
 **/ 
  private static long
  parseScheduledTimeMS(StringTokenizer st)
    throws IOException, InvalidOutputLogEntryException{
    String token = null;
    long readScheduledTimeMS = 0;
    boolean badTimeMS = false;

    // Read the event time in ms
    if(st.hasMoreTokens()){
      token = st.nextToken();
      try{
        readScheduledTimeMS = Long.parseLong(token);
      }catch(NumberFormatException nfe){
        badTimeMS = true;
      }
    }else{
      badTimeMS = true;
    }
    if(badTimeMS){
      throw(new InvalidOutputLogEntryException("The second field should be " +
                                               "the scheduled time in ms!"));
    }
    return(readScheduledTimeMS);
  }

 /** 
 *  Parse the operation type (mostly copied from TraceEntry.java) 
 **/ 
  private static int
  parseOperationType(StringTokenizer st)
    throws IOException, InvalidOutputLogEntryException{
    String token = null;
    int operationType = READ_START;
    boolean badOperation = false;

    // Read the operation type
    if(st.hasMoreTokens()){
      token = st.nextToken();
      if(token.equalsIgnoreCase("read_start")){
        operationType = OutputLogEntry.READ_START;
      }else if(token.equalsIgnoreCase("read_finish")){
        operationType = OutputLogEntry.READ_FINISH;
      }else if(token.equalsIgnoreCase("inv_arriv")){
        operationType = OutputLogEntry.INV_ARRIV;
      }else if(token.equalsIgnoreCase("bnd_arriv")){
        operationType = OutputLogEntry.BND_ARRIV;
      }else if(token.equalsIgnoreCase("demand_req")){
        operationType = OutputLogEntry.DEMAND_REQ;
      }else if(token.equalsIgnoreCase("demand_arriv")){
        operationType = OutputLogEntry.DEMAND_ARRIV;
      }else if(token.equalsIgnoreCase("push_arriv")){
        operationType = OutputLogEntry.PUSH_ARRIV;
      }else if(token.equalsIgnoreCase("write_start")){
        operationType = OutputLogEntry.WRITE_START;
      }else if(token.equalsIgnoreCase("write_finish")){
        operationType = OutputLogEntry.WRITE_FINISH;
      }else{
        badOperation = true;
      }
    }else{
      badOperation = true;
    }
    if(badOperation){
      throw(new InvalidOutputLogEntryException("The third field should " +
                                               "the operation type!"));
    }
    return(operationType);
  }

 /** 
 *  Parse the thread number 
 **/ 
  private static int
  parseThreadNum(StringTokenizer st)
    throws IOException, InvalidOutputLogEntryException{
    String token = null;
    int readThreadNum = 0;
    boolean badThreadNum = false;

    // Read the thread number
    if(st.hasMoreTokens()){
      token = st.nextToken();
      try{
        readThreadNum = Integer.parseInt(token);
      }catch(NumberFormatException nfe){
        badThreadNum = true;
      }
    }else{
      badThreadNum = true;
    }
    if(badThreadNum){
      throw(new InvalidOutputLogEntryException("The fourth field should be " +
                                               "the thread number!"));
    }
    return(readThreadNum);
  }

 /** 
 *  Parse the object ID 
 **/ 
  private static ObjId
  parseObjId(StringTokenizer st)
    throws IOException, InvalidOutputLogEntryException{
    String token = null;
    String objIdPath = null;
    ObjId readObjId = null;
    boolean badObjId = false;

    // Read the event time in ms
    if(st.hasMoreTokens()){
      token = st.nextToken();
      if((token.startsWith("\"")) && (token.endsWith("\""))){
        objIdPath = token.substring(1, token.length() - 1);
        readObjId = new ObjId(objIdPath);
      }else{
        badObjId = true;
      }
    }else{
      badObjId = true;
    }
    if(badObjId){
      throw(new InvalidOutputLogEntryException("The fifth field should be " +
                                               "the object ID"));
    }
    return(readObjId);
  }

 /** 
 *  Parse the data AcceptStamp 
 **/ 
  private static AcceptStamp
  parseDataAcceptStamp(StringTokenizer st)
    throws IOException, InvalidOutputLogEntryException{
    String token = null;
    String nodeIdString = null;
    String clockString = null;
    int colonIndex = 0;
    NodeId nodeId = null;
    long localClock = 0;
    long nodeIdLong = 0;
    boolean badAcceptStamp = false;
    AcceptStamp readDataAcceptStamp = null;

    // Read the accept stamp
    if(st.hasMoreTokens()){
      token = st.nextToken();
      if(!token.equalsIgnoreCase("null")){
        colonIndex = token.indexOf(':');
        try{
          if((token.startsWith("(")) &&
             (token.endsWith(")")) &&
             (colonIndex >= 0)){
            nodeIdString = token.substring(1, colonIndex);
            clockString = token.substring(colonIndex + 1, token.length() - 1);
            nodeIdLong = Long.parseLong(nodeIdString);
            nodeId = new NodeId(nodeIdLong);
          localClock = Long.parseLong(clockString);
          readDataAcceptStamp = new AcceptStamp(localClock, nodeId);
          }else{
            badAcceptStamp = true;
          }
        }catch(NumberFormatException nfe){
          badAcceptStamp = true;
        }
      }else{
        readDataAcceptStamp = null;
      }
    }else{
      badAcceptStamp = true;
    }
    if(badAcceptStamp){
      throw(new InvalidOutputLogEntryException("The sixth field should be " +
                                               "an accept stamp!"));
    }
    return(readDataAcceptStamp);
  }

 /** 
 *  Data members 
 **/ 
  private long traceTimeMS;
  private long scheduledTimeMS;
  private int operation;
  private int threadNum;
  private ObjId objId;
  private AcceptStamp dataAcceptStamp;
}

 /** 
 *  Exception that is thrown when we read an invalid entry in our trace file 
 **/ 
class InvalidOutputLogEntryException extends Exception{

 /** 
 *  Constructor 
 **/ 
  public
  InvalidOutputLogEntryException(String newMsg){
    this.msg = newMsg;
  }

 /** 
 *  Convert this exception to a string 
 **/ 
  public String
  toString(){
    return(this.msg);
  }

 /** 
 *  Data members 
 **/ 
  private String msg;
}
