 /** 
 *  Code to parse/store an entry in trace files used for the Nice experiments. 
 **/ 
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.BufferedReader;
import java.io.Reader;
import java.io.EOFException;
import java.io.PrintStream;
import java.io.OutputStream;
import java.util.StringTokenizer;

// Used for testing
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;

public class TraceEntry implements Immutable, Cloneable{

 /** 
 *  Constants 
 **/ 
  public static final int READ = 1;
  public static final int WRITE = 2;
  public static final int SYNC = 3;
  public static final String BOUND_STRING_LOWERCASE = "bound";
  public static final String UNBOUND_STRING_LOWERCASE = "unbound";

 /** 
 *  Constructor 
 **/ 
  public
  TraceEntry(int newEventNum,
             long newTimeMS,
             int newOperation,
             double newPriority,
             long newSize,
             ObjId newObjId,
             boolean newIsBound){
    assert((newOperation == READ) ||
           (newOperation == WRITE) ||
           (newOperation == SYNC));
    this.eventNum = newEventNum;
    this.timeMS = newTimeMS;
    this.operation = newOperation;
    this.priority = newPriority;
    this.size = newSize;
    this.objId = (ObjId)newObjId.clone();
    this.isBound = newIsBound;
  }

 /** 
 *  Constructor 
 **/ 
  public
  TraceEntry(BufferedReader br) throws IOException, InvalidTraceEntryException{
    String line = null;
    StringTokenizer st = null;

    line = br.readLine();
    if(line == null){
      throw(new EOFException());
    }
    st = new StringTokenizer(line);
    this.eventNum = TraceEntry.parseEventNum(st);
    this.timeMS = TraceEntry.parseEventTimeMS(st);
    this.operation = TraceEntry.parseOperationType(st);
    this.priority = TraceEntry.parsePriority(st);
    this.size = TraceEntry.parseSize(st);
    this.objId = TraceEntry.parseObjId(st);
    this.isBound = TraceEntry.parseIsBound(st);
  }

 /** 
 *  Write this object to an output stream 
 **/ 
  public void
  writeToOS(OutputStream os) throws IOException{
    PrintStream ps = null;

    assert((this.operation == TraceEntry.READ) ||
           (this.operation == TraceEntry.WRITE) ||
           (this.operation == TraceEntry.SYNC));
    ps = new PrintStream(os);
    ps.print("" + this.eventNum);
    ps.print("\t" + this.timeMS);
    if(this.operation == TraceEntry.READ){
      ps.print("\tREAD");
    }else if(this.operation == TraceEntry.WRITE){
      ps.print("\tWRITE");
    }else{
      assert(this.operation == TraceEntry.SYNC);
      ps.print("\tSYNC");
    }
    ps.print("\t" + this.priority);
    ps.print("\t" + this.size);
    ps.print("\t\"" + this.objId.getPath() + "\"");
    if(this.isBound){
      ps.print("\t" + TraceEntry.BOUND_STRING_LOWERCASE);
    }else{
      ps.print("\t" + TraceEntry.UNBOUND_STRING_LOWERCASE);
    }
    ps.println();
    ps.flush();
  }

 /** 
 *  Return the event number 
 **/ 
  public int
  getEventNum(){
    return(this.eventNum);
  }

 /** 
 *  Return the operation type 
 **/ 
  public int
  getOperation(){
    return(this.operation);
  }

 /** 
 *  Return the time of the event 
 **/ 
  public long
  getTimeMS(){
    return(this.timeMS);
  }

 /** 
 *  Return the priority of the event (used only for writes) 
 **/ 
  public double
  getPriority(){
    return(this.priority);
  }

 /** 
 *  Return the size of the event 
 **/ 
  public long
  getSize(){
    return(this.size);
  }

 /** 
 *  Return the object Id from the event 
 **/ 
  public ObjId
  getObjId(){
    return(this.objId);
  }


 /** 
 *  Return true if this write is bound 
 **/ 
  public boolean
  getIsBound(){
    return(this.isBound);
  }

 /** 
 *  Return true if "o" equals this 
 **/ 
  public boolean
  equals(Object o){
    TraceEntry te = null;
    boolean result = false;

    // 5 constants, 2 java-provided fields, 7 declared fields
    assert(this.getClass().getDeclaredFields().length == 14);

    if(o instanceof TraceEntry){
      te = (TraceEntry)o;
      result = ((this.eventNum == te.eventNum) &&
                (this.operation == te.operation) &&
                (this.timeMS == te.timeMS) &&
                (this.priority == te.priority) &&
                (this.size == te.size) &&
                (this.objId.equals(te.objId)) &&
                (this.isBound == te.isBound));
    }else{
      result = false;
    }
    return(result);
  }

 /** 
 *  Clone this object 
 **/ 
  public Object
  clone(){
    // Because this object is immutable, we do not need to do anything
    return(this);
  }

 /** 
 *  Return a hashCode (note: doesn't currently do anything) 
 **/ 
  public int
  hashCode(){
    assert(false);
    return(0);
  }

 /** 
 *  Used for testing 
 **/ 
  public static void
  main(String[] argv){
    if(argv.length < 1){
      //Env.verifyAssertEnabled();
      System.out.println("Testing TraceEntry.java...");
      TraceEntry.test1();
      TraceEntry.test2();
      System.out.println("...Finished");
    }else{
      TraceEntry.readLoopTest();
    }
  }

 /** 
 *  Test TraceEntry (NOTE: Currently never called) 
 **/ 
  private static void
  test0(){
    String line = null;
    StringTokenizer tokenizer = null;
    TraceEntry te = null;
    BufferedReader reader = null;

    try{
      reader = new BufferedReader(new InputStreamReader(System.in));
      while(true){
        line = reader.readLine();
        if(line == null){
          break;
        }
        tokenizer = new StringTokenizer(line);
        while(tokenizer.hasMoreTokens()){
          System.out.println("" + tokenizer.nextToken());
        }
      }
    }catch(IOException e){
      System.out.println("" + e);
      assert(false);
    }
  }

 /** 
 *  Test TraceEntry 
 **/ 
  private static void
  test1(){
    TraceEntry te1 = null;
    TraceEntry te2 = null;
    ByteArrayOutputStream baos = null;
    ByteArrayInputStream bais = null;
    BufferedReader br = null;
    long timeMS = 0;

    try{
      baos = new ByteArrayOutputStream();
      timeMS = System.currentTimeMillis();
      te1 = new TraceEntry(1, timeMS, READ, 0.1, 100, new ObjId("/0"), true);
      te1.writeToOS(baos);
      assert(te1.equals(te1));
      //te1.writeToOS(System.out); // Already tested

      bais = new ByteArrayInputStream(baos.toByteArray());
      br = new BufferedReader(new InputStreamReader(bais));
      te2 = new TraceEntry(br);
      //te2.writeToOS(System.out); // Already tested
      try{
          te2 = new TraceEntry(br);
          assert(false);
      }catch(EOFException e){
        // Do nothing; this was expected.
      }
      assert(te1 != te2);
      assert(te1.equals(te2));
    }catch(IOException e){
      assert(false);
    }catch(InvalidTraceEntryException e){
      System.err.println("" + e);
      assert(false);
    }
  }

 /** 
 *  Test TraceEntry 2 
 **/ 
  private static void
  test2(){
    TraceEntry te = null;
    long timeMS = 0;

    timeMS = System.currentTimeMillis();
    te = new TraceEntry(1, timeMS, READ, 0.1, 200, new ObjId("/0"), false);
    assert(te.getEventNum() == te.eventNum);
    assert(te.getTimeMS() == te.timeMS);
    assert(te.getOperation() == te.operation);
    assert(te.getPriority() == te.priority);
    assert(te.getSize() == te.size);
    assert(te.getObjId().equals(te.objId));
    assert(te.getIsBound() == te.isBound);
  }

 /** 
 *  Try reading a file from stdin 
 **/ 
  private static void
  readLoopTest(){
    TraceEntry te = null;
    BufferedReader reader = null;

    try{
      reader = new BufferedReader(new InputStreamReader(System.in));
      while(true){
        te = new TraceEntry(reader);
        te.writeToOS(System.out);
      }
    }catch(EOFException e){
      System.out.println("Done!");
    }catch(InvalidTraceEntryException e){
      System.out.println("" + e);
      assert(false);
    }catch(IOException e){
      System.out.println("" + e);
      assert(false);
    }
  }

 /** 
 *  Parse the event number 
 **/ 
  private static int
  parseEventNum(StringTokenizer st)
    throws IOException, InvalidTraceEntryException{
    String token = null;
    int readEventNum = 0;
    boolean badEventNum = false;

    // Read the event number
    if(st.hasMoreTokens()){
      token = st.nextToken();
      try{
        readEventNum = Integer.parseInt(token);
      }catch(NumberFormatException nfe){
        badEventNum = true;
      }
    }else{
      badEventNum = true;
    }

    if(badEventNum){
      throw(new InvalidTraceEntryException("First column should be " +
                                           "an event number!"));
    }
    return(readEventNum);
  }

 /** 
 *  Parse the event time (in milliseconds) 
 **/ 
  private static long
  parseEventTimeMS(StringTokenizer st)
    throws IOException, InvalidTraceEntryException{
    String token = null;
    long readEventTimeMS = 0;
    boolean badTimeMS = false;

    // Read the event time in ms
    if(st.hasMoreTokens()){
      token = st.nextToken();
      try{
        readEventTimeMS = Long.parseLong(token);
      }catch(NumberFormatException nfe){
        badTimeMS = true;
      }
    }else{
      badTimeMS = true;
    }
    if(badTimeMS){
      throw(new InvalidTraceEntryException("Second column should be " +
                                           "the event time in ms!"));
    }
    return(readEventTimeMS);
  }

 /** 
 *  Parse the operation type 
 **/ 
  private static int
  parseOperationType(StringTokenizer st)
    throws IOException, InvalidTraceEntryException{
    String token = null;
    int operationType = READ;
    boolean badOperation = false;

    // Read the operation type
    if(st.hasMoreTokens()){
      token = st.nextToken();
      if(token.equalsIgnoreCase("read")){
        operationType = TraceEntry.READ;
      }else if(token.equalsIgnoreCase("write")){
        operationType = TraceEntry.WRITE;
      }else if(token.equalsIgnoreCase("sync")){
        operationType = TraceEntry.SYNC;
      }else{
        badOperation = true;
      }
    }else{
      badOperation = true;
    }
    if(badOperation){
      throw(new InvalidTraceEntryException("Third column should either be " +
                                           "READ or WRITE!"));
    }
    return(operationType);
  }

 /** 
 *  Parse the priority 
 **/ 
  private static double
  parsePriority(StringTokenizer st)
    throws IOException, InvalidTraceEntryException{
    String token = null;
    double priority = 0;
    boolean badPriority = false;

    // Read the event time in ms
    if(st.hasMoreTokens()){
      token = st.nextToken();
      try{
        priority = Double.parseDouble(token);
      }catch(NumberFormatException e){
        badPriority = true;
      }
    }else{
      badPriority = true;
    }
    if(badPriority){
      throw(new InvalidTraceEntryException("Fourth column should be " +
                                           "the priority!"));
    }
    return(priority);
  }

 /** 
 *  Parse the size 
 **/ 
  private static long
  parseSize(StringTokenizer st)
    throws IOException, InvalidTraceEntryException{
    String token = null;
    long size = 0;
    boolean badSize = false;

    // Read the event time in ms
    if(st.hasMoreTokens()){
      token = st.nextToken();
      try{
        size = Long.parseLong(token);
      }catch(NumberFormatException e){
        badSize = true;
      }
    }else{
      badSize = true;
    }
    if(badSize){
      throw(new InvalidTraceEntryException("Fifth column should be " +
                                           "the size!"));
    }
    return(size);
  }

 /** 
 *  Parse the object ID 
 **/ 
  private static ObjId
  parseObjId(StringTokenizer st)
    throws IOException, InvalidTraceEntryException{
    String token = null;
    String objIdPath = null;
    ObjId readObjId = null;
    boolean badObjId = false;

    // Read the event time in ms
    if(st.hasMoreTokens()){
      token = st.nextToken();
      if((token.startsWith("\"")) && (token.endsWith("\""))){
        objIdPath = token.substring(1, token.length() - 1);
        readObjId = new ObjId(objIdPath);
      }else{
        badObjId = true;
      }
    }else{
      badObjId = true;
    }
    if(badObjId){
      throw(new InvalidTraceEntryException("Sixth column should be " +
                                           "the object ID"));
    }
    return(readObjId);
  }

 /** 
 *  Parse the isBound field 
 **/ 
  private static boolean
  parseIsBound(StringTokenizer st)
    throws IOException, InvalidTraceEntryException{
    String token = null;
    String isBoundString = null;
    boolean isBound = false;
    boolean badIsBound = false;

    // Read the event time in ms
    if(st.hasMoreTokens()){
      token = st.nextToken();
      if(token.equalsIgnoreCase(TraceEntry.BOUND_STRING_LOWERCASE)){
        isBound = true;
        badIsBound = false;
      }else if(token.equalsIgnoreCase(TraceEntry.UNBOUND_STRING_LOWERCASE)){
        isBound = false;
        badIsBound = false;
      }else{
        badIsBound = true;
      }
    }else{
      badIsBound = true;
    }
    if(badIsBound){
      throw(new InvalidTraceEntryException("Seventh column may only be " +
                                           "\"bound\" or \"unbound\""));
    }
    return(isBound);
  }

 /** 
 *  Data members 
 **/ 
  private int eventNum;
  private int operation;
  private long timeMS;
  private double priority;
  private long size;
  private ObjId objId;
  private boolean isBound;
}

 /** 
 *  Exception that is thrown when we read an inval entry in our trace file 
 **/ 
class InvalidTraceEntryException extends Exception{

 /** 
 *  Constructor 
 **/ 
  public
  InvalidTraceEntryException(String newMsg){
    this.msg = newMsg;
  }

 /** 
 *  Convert this exception to a string 
 **/ 
  public String
  toString(){
    return(this.msg);
  }

 /** 
 *  Data members 
 **/ 
  private String msg;
}
