 /** 
 *  AccessLogRecord: Parse access log records 
 **/ 
import java.util.Date;
import java.util.Vector;
import java.util.Hashtable;
import java.text.ParsePosition;
import java.text.SimpleDateFormat;

// For testing
import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.io.IOException;

public class AccessLogRecord{

 /** 
 *  Constants 
 **/ 
  public static final int DATE_STRING_LEN = 28;

 /** 
 *  Nested exception class 
 **/ 

  public static class UnparseableException extends Exception{
 /** 
 *  Constructor 
 **/ 
    public UnparseableException(){
      this.message = "Encountered unparseable entry";
    }

 /** 
 *  Constructor 2 
 **/ 
    public UnparseableException(String s){
      this.message = s;
    }

 /** 
 *  Convert to a string 
 **/ 
    public String toString(){
      return(this.message);
    }

 /** 
 *  Private data 
 **/ 
    private String message;
  }

 /** 
 *  Constructor 
 **/ 
  public AccessLogRecord(String line) throws UnparseableException{
    int offset = 0;

    offset = this.parseRemoteHost(line, 0);
    offset = this.parseRemoteUserName(line, offset);
    offset = this.parseAuthUserName(line, offset);
    offset = this.parseTime(line, offset);
    offset = this.parseRequest(line, offset);
    offset = this.parseStatus(line, offset);
    offset = this.parseNumBytes(line, offset);
    offset = this.parseReferer(line, offset);
    offset = this.parseUserSoftware(line, offset);
    offset = this.parseUnknownField(line, offset);
  }

 /** 
 *  Extract the names of remote hosts from the line 
 **/ 
  protected int parseRemoteHost(String line, int offset)
    throws UnparseableException{
    int tokenEnd = 0;
    String host = null;
    int tokenStart = 0;
    boolean done = false;
    Vector hostVector = null;

    hostVector = new Vector();
    tokenStart = 0;
    while(!done){
      tokenEnd = line.indexOf(' ', tokenStart);
      if(tokenEnd < 0){
        throw(new UnparseableException());
      }
      host = line.substring(tokenStart, tokenEnd);
      if(line.charAt(tokenEnd - 1) == ','){
        host = line.substring(tokenStart, tokenEnd - 1);
        done = false;
      }else{
        done = true;
      }
      hostVector.add(host);
      tokenStart = tokenEnd + 1;
    }
    this.remoteHosts = new String[hostVector.size()];
    for(int i = 0; i < hostVector.size(); i++){
      host = (String)hostVector.get(i);
      this.remoteHosts[i] = host;
    }
    return(tokenEnd + 1);
  }

 /** 
 *  Extract the remote user name from the line 
 **/ 
  protected int parseRemoteUserName(String line, int offset)
    throws UnparseableException{
    int tokenEnd = 0;

    tokenEnd = line.indexOf(' ', offset);
    if(tokenEnd <= 0){
      throw(new UnparseableException());
    }
    this.remoteUserName = line.substring(offset, tokenEnd);
    return(tokenEnd + 1);
  }

 /** 
 *  Extract the authorized user name from the line 
 **/ 
  protected int parseAuthUserName(String line, int offset)
    throws UnparseableException{
    int tokenEnd = 0;

    tokenEnd = line.indexOf(' ', offset);
    if(tokenEnd <= 0){
      throw(new UnparseableException());
    }
    this.authUserName = line.substring(offset, tokenEnd);
    return(tokenEnd + 1);
  }

 /** 
 *  Extract the date and time from the line 
 **/ 
  protected int parseTime(String line, int offset)
    throws UnparseableException{
    Date date = null;
    String format = null;
    String dateString = null;
    ParsePosition parsePosition = null;
    SimpleDateFormat dateFormatter = null;

    if((line.charAt(offset) != '[') ||
       (line.charAt(offset + AccessLogRecord.DATE_STRING_LEN) != ' ')){
      throw(new UnparseableException());
    }

    dateString = line.substring(offset);
    format = "'['dd/MMM/yyy:HH:mm:ss '+0000]'";
    dateFormatter = new SimpleDateFormat(format);
    parsePosition = new ParsePosition(0);
    date = dateFormatter.parse(dateString, parsePosition);
    if(date == null){
      throw(new UnparseableException("Bad date: " + dateString));
    }
    this.time = date.getTime();
    return(offset + AccessLogRecord.DATE_STRING_LEN + 1);
  }

 /** 
 *  Extract the request from the line, removing both quotes 
 **/ 
  protected int parseRequest(String line, int offset)
    throws UnparseableException{
    int tokenEnd = 0;
    String requestString = null;

    if(line.charAt(offset) != '\"'){
      throw(new UnparseableException());
    }
    tokenEnd = line.indexOf('\"', offset + 1);
    while(line.charAt(tokenEnd + 1) != ' '){
      tokenEnd = line.indexOf('\"', tokenEnd + 1);
      if(tokenEnd < 0){
        throw(new UnparseableException());
      }
    }
    if(line.charAt(tokenEnd + 1) != ' '){
      throw(new UnparseableException());
    }
    requestString = line.substring(offset + 1, tokenEnd);
    this.request = new WebLogRequest(requestString);
    return(tokenEnd + 2); // End quote and whitespace
  }

 /** 
 *  Extract the server status code from the line 
 **/ 
  protected int parseStatus(String line, int offset)
    throws UnparseableException{
    int tokenEnd = 0;
    String statusStr = null;

    tokenEnd = line.indexOf(' ', offset);
    if(tokenEnd <= 0){
      throw(new UnparseableException());
    }
    statusStr = line.substring(offset, tokenEnd);
    try{
      this.status = Integer.parseInt(statusStr);
    }catch(NumberFormatException e){
      throw(new UnparseableException("Bad status code"));
    }
    return(tokenEnd + 1);
  }

 /** 
 *  Extract the returned number of bytes from the line 
 **/ 
  protected int parseNumBytes(String line, int offset)
    throws UnparseableException{
    int tokenEnd = 0;
    String numBytesStr = null;

    tokenEnd = line.indexOf(' ', offset);
    if(tokenEnd <= 0){
      throw(new UnparseableException());
    }
    numBytesStr = line.substring(offset, tokenEnd);
    if(numBytesStr.equals("-")){
      // Num bytes does not apply or is unknown
      this.numBytes = -1;
    }else{
      try{
        this.numBytes = Integer.parseInt(numBytesStr);
      }catch(NumberFormatException e){
        throw(new UnparseableException("Bad number of bytes"));
      }
    }
    return(tokenEnd + 1);
  }

 /** 
 *  Extract the referer from the line, removing both quotes 
 **/ 
  protected int parseReferer(String line, int offset)
    throws UnparseableException{
    int tokenEnd = 0;

    if(line.charAt(offset) != '\"'){
      throw(new UnparseableException());
    }
    tokenEnd = line.indexOf('\"', offset + 1);
    if(tokenEnd < 0){
      throw(new UnparseableException());
    }
    while(line.charAt(tokenEnd + 1) != ' '){
      tokenEnd = line.indexOf('\"', tokenEnd + 1);
      if(tokenEnd < 0){
        throw(new UnparseableException());
      }
    }
    this.referer = line.substring(offset + 1, tokenEnd);
    return(tokenEnd + 2); // End quote and whitespace
  }

 /** 
 *  Extract the user browser name from the line, removing both quotes 
 **/ 
  protected int parseUserSoftware(String line, int offset)
    throws UnparseableException{
    int tokenEnd = 0;

    if(line.charAt(offset) != '\"'){
      throw(new UnparseableException());
    }
    tokenEnd = line.indexOf('\"', offset + 1);
    if((tokenEnd < 0) || (line.charAt(tokenEnd + 1) != ' ')){
      throw(new UnparseableException());
    }
    this.userSoftware = line.substring(offset + 1, tokenEnd);
    return(tokenEnd + 2); // End quote and whitespace
  }

 /** 
 *  Extract the unknown field from the line, removing both quotes 
 **/ 
  protected int parseUnknownField(String line, int offset)
    throws UnparseableException{
    int tokenEnd = 0;

    if(line.charAt(offset) != '\"'){
      throw(new UnparseableException());
    }
    tokenEnd = line.indexOf('\"', offset + 1);
    if(tokenEnd < 0){
      throw(new UnparseableException());
    }
    this.unknownField = line.substring(offset + 1, tokenEnd);
    return(tokenEnd + 2); // End quote and whitespace
  }

 /** 
 *  Return the array of remote hosts 
 **/ 
  public String[] getRemoteHosts(){
    return(this.remoteHosts);
  }

 /** 
 *  Return the remote user name 
 **/ 
  public String getRemoteUserName(){
    return(this.remoteUserName);
  }

 /** 
 *  Return the authorized user name 
 **/ 
  public String getAuthUserName(){
    return(this.authUserName);
  }

 /** 
 *  Return the time (in MS from epoch) 
 **/ 
  public long getTimeMS(){
    return(this.time);
  }

 /** 
 *  Return the request string 
 **/ 
  public WebLogRequest getRequest(){
    return(this.request);
  }

 /** 
 *  Return the status code 
 **/ 
  public int getStatus(){
    return(this.status);
  }

 /** 
 *  Return the number of the bytes 
 **/ 
  public int getNumBytes(){
    return(this.numBytes);
  }

 /** 
 *  Return the referer string 
 **/ 
  public String getReferer(){
    return(this.referer);
  }

 /** 
 *  Return the user software string 
 **/ 
  public String getUserSoftware(){
    return(this.userSoftware);
  }

 /** 
 *  Return the unknown field 
 **/ 
  public String getUnknownField(){
    return(this.unknownField);
  }

 /** 
 *  Convert to a string 
 **/ 
  public String toString(){
    String str = null;

    str = remoteHosts[0];
    for(int i = 1; i < remoteHosts.length; i++){
      str += ", " + remoteHosts[i];
    }
    str += " " + this.remoteUserName;
    str += " " + this.authUserName;
    str += " " + this.time;
    str += " \"" + this.request + "\"";
    str += " " + this.status;
    str += " " + this.numBytes;
    str += " \"" + this.referer + "\"";
    str += " \"" + this.userSoftware + "\"";
    str += " \"" + this.unknownField + "\"";
    return(str);
  }

 /** 
 *  Used for testing 
 **/ 
  public static void main(String[] argv){
    int badCount = 0;
    int lineCount = 0;
    String line = null;
    boolean print = false;
    AccessLogRecord wlr = null;
    BufferedReader br = null;
    InputStreamReader isr = null;

    try{
      isr = new InputStreamReader(System.in);
      br = new BufferedReader(isr);
      line = br.readLine();
      print = (argv.length == 1) && (argv[0].equals("print"));
      while(line != null){
        try{
          wlr = new AccessLogRecord(line);
          if(print){
            System.out.println("" + wlr);
            System.out.println("Time = " +
                               new Date(wlr.getTimeMS()));
            System.out.println("Request = \"" +
                               wlr.getRequest() + "\"");
            System.out.println("Status = " + wlr.getStatus());
            System.out.println("Num Bytes = " + wlr.getNumBytes());
            System.out.println("--------------------------------");
          }
        }catch(AccessLogRecord.UnparseableException e){
          System.out.println("" + e + ", " + lineCount);
          badCount++;
        }
        lineCount++;
        line = br.readLine();
      }
    }catch(IOException e){
      System.err.println("" + e);
    }catch(RuntimeException e){
      System.err.println("Line count = " + lineCount);
      System.err.println("" + e);
      e.printStackTrace();
    }catch(Exception e){
      System.err.println("Line count = " + lineCount);
      System.err.println("" + e);
      e.printStackTrace();
    }
  }

 /** 
 *  Private data 
 **/ 
  protected String[] remoteHosts;
  protected String remoteUserName;
  protected String authUserName;
  protected long time;
  protected WebLogRequest request;
  protected int status;
  protected int numBytes;
  protected String referer;
  protected String userSoftware;
  protected String unknownField;
}
