 /** 
 *  Code to generate a trace file for the Nice experiments in a useful format 
 **/ 
import java.io.IOException;
import java.io.OutputStream;
import java.util.HashMap;
import java.util.Set;
import java.util.Iterator;
import java.util.Map;

public class GenMergedTraceFile{

 /** 
 *  Constructor 
 **/ 
  public
  GenMergedTraceFile(long newSkewMS,
                     String updateTraceFileName,
                     String allAccessedFilesListName,
                     String[] accessTraceFileNames,
                     String priorityFileName) throws IOException{
    this.skewMS = newSkewMS;
    this.updateLogReader = new UpdateLogReader(updateTraceFileName);
    this.updateLogRecord = this.updateLogReader.getNextValidRecord();
    this.priorityTracker = new PriorityTracker(priorityFileName);
    this.accessLogReaders = new AccessLogReader[accessTraceFileNames.length];
    this.accessLogRecords = new AccessLogRecord[accessTraceFileNames.length];
    for(int i = 0; i < accessTraceFileNames.length; i++){
      this.accessLogReaders[i] = new AccessLogReader(accessTraceFileNames[i]);
      this.accessLogRecords[i] =
        this.accessLogReaders[i].getNextStaticRecord();
    }
    this.eventNum = 1;
    this.allFilesTable =
      GenMergedTraceFile.readAllFilesTable(allAccessedFilesListName);
  }

 /** 
 *  Generate the output stream to os 
 **/ 
  public void
  genToOS(OutputStream os) throws IOException{
    TraceEntry te = null;

    this.genAllFileWritesToOS(os);
    te = new TraceEntry(this.eventNum,
                        0,                    // Time in ms
                        TraceEntry.SYNC,
                        1.0,                  // Priority (ignored for sync)
                        0,                    // Size (ignored for sync)
                        new ObjId("null"),    // Object ID (ignored for sync)
                        false);               // isBound (ignored for sync)
    this.eventNum++;
    te.writeToOS(os);
    te = this.getNextTraceEntry();
    while(te != null){
      te.writeToOS(os);
      te = this.getNextTraceEntry();
    }
  }

 /** 
 *  Program starting point 
 **/ 
  public static void
  main(String[] argv){
    long startTimeMS = 0;
    String[] accessTraceNames = null;
    GenMergedTraceFile gmtf = null;

    // Usage: java GenMergedTraceFile startTimeMS <updateLog.gz>
    // <allAccessedFilesListName> <traceFile.gz>* <priorityFile>
    try{
      accessTraceNames = new String[argv.length - 4];
      for(int i = 0; i < accessTraceNames.length; i++){
        accessTraceNames[i] = argv[i + 3];
      }
      assert(argv.length >= 2);
      startTimeMS = Long.parseLong(argv[0]);
      gmtf = new GenMergedTraceFile(startTimeMS,
                                    argv[1],
                                    argv[2],
                                    accessTraceNames,
                                    argv[argv.length - 1]);
      gmtf.genToOS(System.out);
    }catch(IOException e){
      System.err.println("" + e);
    }
  }

 /** 
 *  Read from disk the list of all files accessed across all traces 
 **/ 
  private static HashMap
  readAllFilesTable(String allAccessedFilesListName) throws IOException{
    String objId = null;
    long size = 0;
    HashMap allFilesTable = null;
    AllAccessedFilesListReader allFilesReader = null;

    allFilesTable = new HashMap();
    allFilesReader = new AllAccessedFilesListReader(allAccessedFilesListName);
    while(allFilesReader.hasMoreEntries()){
      objId = allFilesReader.getObjectId();
      size = allFilesReader.getSize();
      assert(!allFilesTable.containsKey(objId));
      allFilesTable.put(objId, new Long(size));
      allFilesReader.moveToNextEntry();
    }
    return(allFilesTable);
  }

 /** 
 *  Generate write entries for all objects in the system 
 **/ 
  private void
  genAllFileWritesToOS(OutputStream os) throws IOException{
    Set allFilesEntrySet = null;
    Map.Entry entry = null;
    long size = 0;
    TraceEntry te = null;
    String objId = null;

    allFilesEntrySet = this.allFilesTable.entrySet();
    for(Iterator i = allFilesEntrySet.iterator(); i.hasNext();){
      entry = (Map.Entry)i.next();
      objId = (String)entry.getKey();
      size = ((Long)entry.getValue()).longValue();
      te = new TraceEntry(this.eventNum,
                          0,       // Time in ms
                          TraceEntry.WRITE,
                          1.0,     // Priority (ignored for bound invalidate)
                          size,
                          new ObjId(objId),
                          true);   // Bound invalidate
      te.writeToOS(os);
      this.eventNum++;
    }
  }

 /** 
 *  Get the next trace entry (the one with the lowest time from all traces) 
 **/ 
  private TraceEntry
  getNextTraceEntry() throws IOException{
    int index = 0;
    long size = 0;
    long updateLogTimeMS = -1;
    long minAccessLogTimeMS = -1;
    boolean isValidAccessLogTime = true;
    String objIdString = null;
    TraceEntry te = null;
    WebLogRequest request = null;

    if(this.updateLogRecord != null){
      updateLogTimeMS = this.updateLogRecord.getTimeMS();
    }

    index = this.getNextAccessRecordIndex();
    if(index >= 0){
      minAccessLogTimeMS = this.accessLogRecords[index].getTimeMS();
      isValidAccessLogTime = true;
    }else{
      isValidAccessLogTime = false;
    }

    if(isValidAccessLogTime){
      if((this.updateLogRecord != null) &&
         (updateLogTimeMS < minAccessLogTimeMS)){
        objIdString = this.updateLogRecord.getObjectID();
        te = new TraceEntry(this.eventNum,
                            this.updateLogRecord.getTimeMS() - this.skewMS,
                            TraceEntry.WRITE,
                            this.priorityTracker.getPriority(objIdString),
                            this.updateLogRecord.getObjectSize(),
                            new ObjId(objIdString),
                            false);
        this.allFilesTable.put(objIdString,
                               new Long(this.updateLogRecord.getObjectSize()));
        this.updateLogRecord = this.updateLogReader.getNextValidRecord();
      }else{
        request = this.accessLogRecords[index].getRequest();
        objIdString = request.getFixedObjectID();
        size = ((Long)this.allFilesTable.get(objIdString)).longValue();
        te = new TraceEntry(this.eventNum,
                            this.accessLogRecords[index].getTimeMS() - skewMS,
                            TraceEntry.READ,
                            0.0,    // Priority (not needed for reads)
                            size,   // Size
                            new ObjId(objIdString),
                            false);
        this.accessLogRecords[index] =
          this.accessLogReaders[index].getNextStaticRecord();
      }
    }else if(this.updateLogRecord != null){
      objIdString = this.updateLogRecord.getObjectID();
      te = new TraceEntry(this.eventNum,
                          this.updateLogRecord.getTimeMS() - this.skewMS,
                          TraceEntry.WRITE,
                          this.priorityTracker.getPriority(objIdString),
                          this.updateLogRecord.getObjectSize(),
                          new ObjId(objIdString),
                          false);
      this.allFilesTable.put(objIdString,
                             new Long(this.updateLogRecord.getObjectSize()));
      this.updateLogRecord = this.updateLogReader.getNextValidRecord();
    }
    this.eventNum++;
    return(te);
  }

 /** 
 *  Get the index of the next access record 
 **/ 
  private int
  getNextAccessRecordIndex(){
    int minAccessLogTimeIndex = -1;
    long minAccessLogTimeMS = -1;
    long accessLogTimeMS = 0;

    for(int i = 0; i < this.accessLogRecords.length; i++){
      if(this.accessLogRecords[i] != null){
        accessLogTimeMS = this.accessLogRecords[i].getTimeMS();
        assert(accessLogTimeMS >= 0);
        if((minAccessLogTimeMS > accessLogTimeMS) || (minAccessLogTimeMS < 0)){
          minAccessLogTimeIndex = i;
          minAccessLogTimeMS = accessLogTimeMS;
        }
      }
    }
    return(minAccessLogTimeIndex);
  }

 /** 
 *  Private data 
 **/ 
  private PriorityTracker priorityTracker;
  private UpdateLogReader updateLogReader;
  private AccessLogReader[] accessLogReaders;
  private UpdateTraceRecord updateLogRecord;
  private AccessLogRecord[] accessLogRecords;
  private int eventNum;
  private HashMap allFilesTable;
  private long skewMS;
}