 /** 
 *  PriorityTracker: Keep track of object priorities 
 **/ 
import java.io.IOException;
import java.io.FileReader;
import java.io.BufferedReader;
import java.util.Hashtable;

public class PriorityTracker{

 /** 
 *  Constant 
 **/ 
  public static final double DEFAULT_PRIORITY = 0.0;

 /** 
 *  Constructor 
 **/ 
  public PriorityTracker(String fileName)
    throws IOException{
    this.priorityTable = this.parseTable(fileName);
  }

 /** 
 *  Return the priority of a given object 
 **/ 
  public double getPriority(String objectID){
    Double p = null;
    double priority = 0;

    p = (Double)this.priorityTable.get(objectID);
    if(p == null){
      priority = DEFAULT_PRIORITY;
    }else{
      priority = p.doubleValue();
    }
    return(priority);
  }

 /** 
 *  Used for testing 
 **/ 
  public static void main(String[] argv){
    String objectID = null;
    PriorityTracker pt = null;

    try{
      if(argv.length == 1){
        pt = new PriorityTracker(argv[0]);
        objectID = "Something.else";
        System.out.println("Priority of " + objectID + " = " +
                           pt.getPriority(objectID));
        objectID = "/u/nayate/research/Something";
        System.out.println("Priority of " + objectID + " = " +
                           pt.getPriority(objectID));
      }else{
        System.err.println("Usage: PriorityTracker <filename>");
      }
    }catch(IOException e){
      System.err.println("PriorityTracker (main 1): " + e);
    }
  }

 /** 
 *  Parse the table 
 **/ 
  private Hashtable parseTable(String fileName){
    long delay = 0L;
    String line = null;
    int numEntries = 0;
    String objectID = null;
    double priority = 0.0;
    BufferedReader reader = null;
    Hashtable table = null;

    table = new Hashtable();
    try{
      // NOTE: Ignore the first term; it should be 0
      reader = new BufferedReader(new FileReader(fileName));
      delay = this.readDelay(reader);
      assert(delay == 0L);
      numEntries = this.readNumEntries(reader);
      for(int i = 0; i < numEntries; i++){
        line = reader.readLine();
        assert(line != null);
        objectID = this.parseObjectID(line);
        priority = this.parsePriority(line);
        table.put(objectID, new Double(priority));
      }
    }catch(IOException e){
      assert(false);
    }catch(Throwable t){
      t.printStackTrace();
      assert(false);
    }finally{
      if(reader != null){
        try{
          reader.close();
        }catch(IOException e){
          // Do nothing
        }
      }
    }
    return(table);
  }

 /** 
 *  Read the delay from the file 
 **/ 
  private long
  readDelay(BufferedReader reader) throws IOException{
    long delay = 0;
    String line = null;

    delay = -1;
    line = reader.readLine();
    if(line != null){
      delay = Long.parseLong(line);
    }
    return(delay);
  }

 /** 
 *  Read the number of entries before the next delay 
 **/ 
  private int readNumEntries(BufferedReader reader) throws IOException{
    int numEntries = -1;
    String line = null;

    line = reader.readLine();
    if(line != null){
      numEntries = Integer.parseInt(line);
    }
    return(numEntries);
  }

 /** 
 *  Extract the object ID from the line 
 **/ 
  private String parseObjectID(String line){
    int index = 0;

    index = line.indexOf(' ');
    return(line.substring(0, index));
  }

 /** 
 *  Extract the priority from the line 
 **/ 
  private double parsePriority(String line){
    int index = 0;

    index = line.indexOf(' ');
    return(Double.parseDouble(line.substring(index + 1)));
  }

 /** 
 *  Protected data 
 **/ 
  protected Hashtable priorityTable;
}
