 /** 
 *  WebLogRecord: Parse web log requests 
 **/ 

// Used for testing
import java.io.IOException;
import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.PrintStream;

public class WebLogRequest{

 /** 
 *  Constructor 
 **/ 
  public WebLogRequest(String requestString)
    throws AccessLogRecord.UnparseableException{
    int versionIndex = 0;
    int lastSpaceIndex = 0;
    int firstSpaceIndex = 0;
    String httpString = null;

    firstSpaceIndex = requestString.indexOf(' ');
    if(firstSpaceIndex < 0){
      throw(new AccessLogRecord.UnparseableException("Bad request method"));
    }
    this.method = requestString.substring(0, firstSpaceIndex);

    lastSpaceIndex = requestString.lastIndexOf(' ');
    if(lastSpaceIndex <= firstSpaceIndex){
      throw(new AccessLogRecord.UnparseableException("Bad object ID"));
    }
    this.objectID = requestString.substring(firstSpaceIndex + 1,
                                            lastSpaceIndex);

    httpString = requestString.substring(lastSpaceIndex + 1);
    if(httpString.indexOf("HTTP/") < 0){
      throw(new AccessLogRecord.UnparseableException("Bad HTTP field"));
    }
    try{
      this.httpVersion = Double.parseDouble(httpString.substring(5));
    }catch(NumberFormatException e){
      throw(new AccessLogRecord.UnparseableException("Bad HTTP version"));
    }
  }

 /** 
 *  Return the request method 
 **/ 
  public String getMethod(){
    return(this.method);
  }

 /** 
 *  Return the requested object ID 
 **/ 
  public String
  getObjectID(){
    return(this.objectID);
  }

 /** 
 *  Return the requested object ID with appended information removed 
 **/ 
  public String
  getFixedObjectID(){
    String newObjID = null;
    int questionIndex = 0;

    questionIndex = this.objectID.indexOf('?');
    if(questionIndex >= 0){
      newObjID = this.objectID.substring(0, questionIndex);
    }else{
      newObjID = this.objectID;
    }
    if(newObjID.endsWith("/")){
      newObjID += "index.html";
    }
    return(newObjID);
  }

 /** 
 *  Return true if this request is for dynamic data 
 **/ 
  public boolean
  isDynamic(){
    String fixedObjId = null;
    boolean dynamic = false;

    if(this.objectID.indexOf('?') >= 0){
      fixedObjId = this.getFixedObjectID();
      if(fixedObjId.endsWith(".cgi") || (fixedObjId.indexOf("cgi-bin") >= 0)){
        dynamic = true;
      }
    }
    return(dynamic);
  }

 /** 
 *  Return the request HTTP version (1.0 or 1.1) 
 **/ 
  public double getHTTPVersion(){
    return(this.httpVersion);
  }

 /** 
 *  Convert this request to a string 
 **/ 
  public String toString(){
    String str = null;

    str = "(" + this.method + ", " +
      this.objectID + ", HTTP/" + this.httpVersion + ")";
    return(str);
  }

 /** 
 *  Used for testing 
 **/ 
  public static void main(String[] argv){
    String line = null;
    String line1 = null;
    String line2 = null;
    String line3 = null;
    String line4 = null;
    WebLogRequest wlr = null;
    BufferedReader br = null;
    InputStreamReader isr = null;
    ByteArrayInputStream bais = null;

    try{
      System.out.println("Testing WebLogRequest...");

      line1 = "GET objId1?asdf HTTP/1.1";
      bais = WebLogRequest.makeByteArrayWithData(line1);
      isr = new InputStreamReader(bais);
      br = new BufferedReader(isr);
      line = br.readLine();
      wlr = new WebLogRequest(line);
      assert(!wlr.isDynamic());
      assert(wlr.getFixedObjectID().equals("objId1"));

      line2 = "GET objId1.cgi?asdf HTTP/1.1";
      bais = WebLogRequest.makeByteArrayWithData(line2);
      isr = new InputStreamReader(bais);
      br = new BufferedReader(isr);
      line = br.readLine();
      wlr = new WebLogRequest(line);
      assert(wlr.isDynamic());
      assert(wlr.getFixedObjectID().equals("objId1.cgi"));

      line3 = "GET /something/uc.GIF?/something/referring_page HTTP/1.1";
      bais = WebLogRequest.makeByteArrayWithData(line3);
      isr = new InputStreamReader(bais);
      br = new BufferedReader(isr);
      line = br.readLine();
      wlr = new WebLogRequest(line);
      assert(!wlr.isDynamic());
      assert(wlr.getFixedObjectID().equals("/something/uc.GIF"));

      line4 = "GET /cgi-bin/date? HTTP/1.0";
      bais = WebLogRequest.makeByteArrayWithData(line4);
      isr = new InputStreamReader(bais);
      br = new BufferedReader(isr);
      line = br.readLine();
      wlr = new WebLogRequest(line);
      assert(wlr.isDynamic());
      assert(wlr.getFixedObjectID().equals("/cgi-bin/date"));

      System.out.println("...Finished");

      /* Already tested
      while(line != null){
        try{
          wlr = new WebLogRequest(line);
          System.out.println("" + wlr);
          System.out.println("Method = " + wlr.getMethod());
          System.out.println("ObjectID = " + wlr.getObjectID());
          System.out.println("HTTP version = " +
                             wlr.getHTTPVersion());
          System.out.println("");
        }catch(AccessLogRecord.UnparseableException e){
          System.err.println("" + e);
          System.err.println("");
        }
        line = br.readLine();
      }
      */
    }catch(IOException e){
      System.err.println("" + e);
    }catch(AccessLogRecord.UnparseableException e){
      System.err.println("" + e);
    }
  }

 /** 
 *  Fill a byte array input stream with data 
 **/ 
  private static ByteArrayInputStream
  makeByteArrayWithData(String line) throws IOException{
    ByteArrayOutputStream baos = null;
    PrintStream ps = null;
    ByteArrayInputStream bais = null;
    byte[] b = null;

    try{
      baos = new ByteArrayOutputStream();
      ps = new PrintStream(baos);
      ps.println(line);
      ps.flush();
      baos.flush();
      b = baos.toByteArray();
      bais = new ByteArrayInputStream(b);
    }finally{
      if(ps != null){
        ps.close();
      }
      if(baos != null){
        try{
          baos.close();
        }catch(IOException e){
          System.err.println("" + e);
        }
      }
    }
    return(bais);
  }

 /** 
 *  Protected data 
 **/ 
  protected String method;
  protected String objectID;
  protected double httpVersion;
}
