 /** 
 *  CoordScriptGen: Generate files used by our experiments 
 **/ 
import java.util.Vector;
import java.util.LinkedList;
import java.util.ListIterator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Random;
import java.util.Collection;

public class CoordScriptGen
{
    public static final int BRANCHING_FACTOR = 10;
    public static final int RANDOM_SEED = 9877;
    private HashMap interestSet;

 /** 
 *  Constructor 
 **/ 
    public CoordScriptGen()
    {
	this.interestSet = new HashMap();
    }

 /** 
 *  Make directories 
 **/ 
    private void makeDirectories(LinkedList queue, int numDirs, String prefix)
    {
	int[] a = null;
	int capacity = 0;
	String root = null;
	String dir = null;
	String child = null;
	boolean result = false;

	assert(prefix.endsWith("/"));
	root = prefix;
	queue.addLast(root);
	capacity = 1;
	while(capacity < numDirs){
	    assert(!queue.isEmpty());
	    dir = (String)queue.removeFirst();
	    capacity = capacity - 1;
	    for(int i = 0;
		(i < BRANCHING_FACTOR) && (capacity < numDirs);
		i++){
		assert(dir.endsWith("/"));
		child = dir + (i + 1) + "/";
		queue.addLast(child);
		capacity = capacity + 1;
	    }
	}
    }

 /** 
 *  Add a child to the tree 
 **/ 
    public void makeTree(int numNodes, String prefix, HashMap allObjects)
    {
	LinkedList queue = null;
	LinkedList dirFiles = null;
	int numDirs = 0;
	String currentDirName = null;
	String fileName = null;

	numDirs = (numNodes + BRANCHING_FACTOR - 1) / BRANCHING_FACTOR;
	queue = new LinkedList();
	this.makeDirectories(queue, numDirs, prefix);

	// Now store the actual files
	assert(!queue.isEmpty());
	while(numNodes > 0){
	    currentDirName = (String)queue.removeFirst();
	    assert(currentDirName.endsWith("/"));
	    for(int i = 0; (i < BRANCHING_FACTOR) && (numNodes > 0); i++){
		fileName = currentDirName + (i + 1);
		dirFiles = (LinkedList)allObjects.get(currentDirName);
		if(dirFiles == null){
		    dirFiles = new LinkedList();
		    allObjects.put(currentDirName, dirFiles);
		}
		dirFiles.add(fileName);
		numNodes--;
	    }
	}
	assert(queue.isEmpty());
    }

 /** 
 *  Fill in our interest set by choosing a subset of all objects 
 **/ 
    public void populateInterestSet(int dirInterestPct,
				    int fileInterestPct,
				    HashMap allObjects)
    {
	// Iterate through directories and file, mark interest with certain
	// probabilities
	Map.Entry entry = null;
	String dirName = null;
	String fileName = null;
	LinkedList files = null;
	int dirPickCoin = 0;
	int filePickCoin = 0;
	LinkedList interestFiles = null;
	LinkedList allFiles = null;
	Random rand = null;

	rand = new Random(RANDOM_SEED);
	for(Iterator i = allObjects.entrySet().iterator(); i.hasNext();){
	    // See if this entry belongs in the set
	    entry = (Map.Entry)i.next();
	    dirName = (String)entry.getKey();
	    dirPickCoin = rand.nextInt(100);
	    if(dirPickCoin <= dirInterestPct){
		assert(!this.interestSet.containsKey(dirName));
		this.interestSet.put(dirName, new LinkedList());
	    }
	}
	for(Iterator i = this.interestSet.entrySet().iterator(); i.hasNext();){
	    // Choose which files we are interested in
	    entry = (Map.Entry)i.next();
	    dirName = (String)entry.getKey();
	    allFiles = (LinkedList)allObjects.get(dirName);
	    interestFiles = (LinkedList)entry.getValue();
	    assert((allFiles != null) && (interestFiles != null));
	    for(ListIterator j = allFiles.listIterator(); j.hasNext();){
		fileName = (String)j.next();
		filePickCoin = rand.nextInt(100);
		if(filePickCoin <= fileInterestPct){
		    assert(!interestFiles.contains(fileName));
		    interestFiles.add(fileName);
		}
	    }
	}
    }

 /** 
 *  Print all objects 
 **/ 
    public void printAllObjects(HashMap allObjects)
    {
	Map.Entry entry = null;
	String dirName = null;
	String fileName = null;
	LinkedList files = null;

	for(Iterator i = allObjects.entrySet().iterator(); i.hasNext();){
	    entry = (Map.Entry)i.next();
	    dirName = (String)entry.getKey();
	    files = (LinkedList)entry.getValue();
	    for(ListIterator j = files.listIterator(); j.hasNext();){
		fileName = (String)j.next();
		System.out.println(fileName);
	    }
	}
    }

 /** 
 *  Print our interest set 
 **/ 
    public void printInterestSet()
    {
	Map.Entry entry = null;
	String dirName = null;
	String fileName = null;
	LinkedList files = null;

	for(Iterator i = this.interestSet.entrySet().iterator(); i.hasNext();){
	    entry = (Map.Entry)i.next();
	    dirName = (String)entry.getKey();
	    files = (LinkedList)entry.getValue();
	    for(ListIterator j = files.listIterator(); j.hasNext();){
		fileName = (String)j.next();
		System.out.println(fileName);
	    }
	}
    }

 /** 
 *  Print demand read requests 
 **/ 
    public void printDemandReadRequests(int fileSize, int numReads)
    {
	Vector isVector = null;
	Random rand = null;
	int start = 0;
	int len = 0;
	int fileIndex = 0;

	isVector = new Vector();
	rand = new Random(RANDOM_SEED);
	for(Iterator i = this.interestSet.values().iterator(); i.hasNext();){
	    isVector.addAll((Collection)i.next());
	}
	for(int i = 0; i < numReads; i++){
	    fileIndex = rand.nextInt(isVector.size());
	    start = rand.nextInt(fileSize);
	    len = rand.nextInt(fileSize - start + 1);
	    assert((start + len) <= fileSize);
	    System.out.println("d " + isVector.get(fileIndex) + " " +
			       start + " " +
			       len + " " +
			       i + " 9 10");
	}
    }

 /** 
 *  Print a (1) sleep request, (2) subscribe request 
 **/ 
    public static void printScriptPreamble()
    {
	System.out.println("c Start of script");
	System.out.println("c Wait for others to wake up");
	System.out.println("l 4000");
	System.out.println("c Send a subscribe RMI message");
	System.out.println("s /* 10:-1,9:-1 9 10 18000");
	System.out.println("c Wait for invalidates to get transferred");
	System.out.println("l 21000");
	System.out.println("c Print out stats (to subtract out later)");
	System.out.println("p");
    }

    //-----------------------------------------------------------------------
    // Print ending messages to the script
    //-----------------------------------------------------------------------
    public static void printScriptPostamble()
    {
	System.out.println("c Wait for demand fetches to finish");
	System.out.println("l 11000");
	System.out.println("c Print stats and then end the experiment");
	System.out.println("p");
    }

    //-----------------------------------------------------------------------
    // Program staring point
    //-----------------------------------------------------------------------
    public static void main(String[] argv)
    {
	int numFiles = 0;
	int fileSize = 0;
	int numReads = 0;
	String startDirName = null;
	CoordScriptGen csg = null;
	int dirInterestPct = 0;
	int fileInterestPct = 0;
	HashMap allObjects = null;

	if(argv.length == 6){
	    allObjects = new HashMap();
	    csg = new CoordScriptGen();
	    numFiles = Integer.parseInt(argv[0]);
	    startDirName = argv[1];
	    fileSize = Integer.parseInt(argv[2]);
	    numReads = Integer.parseInt(argv[3]);
	    dirInterestPct = Integer.parseInt(argv[4]);
	    fileInterestPct = Integer.parseInt(argv[5]);

	    csg.makeTree(numFiles, startDirName, allObjects);
	    csg.populateInterestSet(dirInterestPct,
				    fileInterestPct,
				    allObjects);
	    CoordScriptGen.printScriptPreamble();
	    csg.printDemandReadRequests(fileSize, numReads);
	    CoordScriptGen.printScriptPostamble();
	    //csg.printCoordScript(fileSize, numReads);
	}else{
	    System.err.println("Usage: java CoordScriptGen <numfiles> " +
			       "<startDirName> <fileSize> <numReads> " +
			       "<dirIntersectPct> <fileInterestPct>");
	}
    }
}

//---------------------------------------------------------------------------
// $Log: CoordScriptGen.java,v $
// Revision 1.4  2004/05/19 01:46:36  arun
// *** empty log message ***
//
// Revision 1.3  2004/05/18 19:51:43  nayate
// Changed a sleep delay
//
// Revision 1.2  2004/05/18 08:56:50  nayate
// Fixed the coordinator script generator
//
// Revision 1.1  2004/05/17 21:58:17  nayate
// Program that generates a script for the coordinator
//
//---------------------------------------------------------------------------
