 /** 
 *  ExptFileGen: Generate files used by our experiments 
 **/ 
import java.util.LinkedList;
import java.util.ListIterator;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;

/*
  (1) Assume that we have a script reader that can read a file and send it.
  (2) Actually make these files
 */

public class ExptFileGen
{
 /** 
 *  Represent a single node in the directory tree 
 **/ 
    private static class Node
    {
	public File dirFile;

 /** 
 *  Constructor 
 **/ 
	public Node(String name, LinkedList children)
	{
	    boolean result = false;

	    this.dirFile = new File(name);
	    if(!dirFile.exists()){
		// Create it as a directory
		result = dirFile.mkdir();
		assert(result);
	    }
	}
    }

    public static final int BRANCHING_FACTOR = 10;
    private ExptFileGen.Node tree;

 /** 
 *  Make directories 
 **/ 
    private void makeDirectories(LinkedList queue, int numDirs, String prefix)
    {
	int capacity = 0;
	String root = null;
	String dir = null;
	String child = null;
	File dirFile = null;
	boolean result = false;

	assert(prefix.endsWith("/"));
	root = prefix;
	queue.addLast(root);
	capacity = 1;
	while(capacity < numDirs){
	    assert(!queue.isEmpty());
	    dir = (String)queue.removeFirst();
	    capacity = capacity - 1;
	    for(int i = 0;
		(i < BRANCHING_FACTOR) && (capacity < numDirs);
		i++){
		assert(dir.endsWith("/"));
		child = dir + (i + 1) + "/";
		dirFile = new File(child);
		result = dirFile.mkdir();
		assert(result);
		queue.addLast(child);
		capacity = capacity + 1;
	    }
	}
    }

 /** 
 *  Populate the given file with the given size 
 **/ 
    public void populateFile(String fileName, long fileSize) throws IOException
    {
	FileOutputStream fos = null;
	String preamble = null;
	byte[] preambleBytes = null;
	byte[] buffer = null;
	int index = 0;
	byte letter = (byte)'a';
	int numToSend = 0;
	long numTotalSent = 0;

	preamble = "For your universal replication needs: URS";
	fos = new FileOutputStream(fileName);
	preambleBytes = preamble.getBytes();
	buffer = new byte[Math.min((int)fileSize, 1024)];
	index = 0;
	while(index < Math.min(buffer.length, preambleBytes.length)){
	    buffer[index] = preambleBytes[index];
	    index++;
	    numTotalSent++;
	}
	fos.write(buffer, 0, index);
	letter = 'a';
	while(numTotalSent < fileSize){
	    numToSend = Math.min((int)(fileSize - numTotalSent),
				 buffer.length);
	    for(int i = 0; i < numToSend; i++){
		buffer[i] = letter;
		letter++;
		if(letter > 'z'){
		    letter = 'a';
		}
	    }
	    fos.write(buffer, 0, numToSend);
	    numTotalSent += numToSend;
	}
    }

 /** 
 *  Add a child to the tree 
 **/ 
    public void makeTree(int numNodes, String prefix, long fileSize)
    throws IOException
    {
	LinkedList queue = null;
	int numDirs = 0;
	String currentDirName = null;
	String fileName = null;

	numDirs = (numNodes + BRANCHING_FACTOR - 1) / BRANCHING_FACTOR;
	queue = new LinkedList();
	this.makeDirectories(queue, numDirs, prefix);

	// Now make the actual files
	assert(!queue.isEmpty());
	while(numNodes > 0){
	    currentDirName = (String)queue.removeFirst();
	    assert(currentDirName.endsWith("/"));
	    for(int i = 0; (i < BRANCHING_FACTOR) && (numNodes > 0); i++){
		fileName = currentDirName + (i + 1);
		this.populateFile(fileName, fileSize);
		numNodes--;
	    }
	}
	assert(queue.isEmpty());
    }

 /** 
 *  Program staring point 
 **/ 
    public static void main(String[] argv)
    {
	int numFiles = 0;
	int fileSize = 0;
	String startDirName = null;
	ExptFileGen tsg = null;

	if(argv.length == 3){
	    try{
		tsg = new ExptFileGen();
		numFiles = Integer.parseInt(argv[0]);
		startDirName = argv[1];
		fileSize = Integer.parseInt(argv[2]);
		tsg.makeTree(numFiles, startDirName, fileSize);
	    }catch(IOException e){
		System.err.println("" + e);
		e.printStackTrace();
	    }
	}else{
	    System.err.println("Usage: java ExptFileGen <numfiles> " +
			       "<startDirName> <fileSize>");
	}
    }
}
