 /** 
 *  SenderScriptGen: Generate files used by our experiments 
 **/ 
import java.util.Vector;
import java.util.LinkedList;
import java.util.ListIterator;
import java.util.Random;

public class SenderScriptGen
{
    public static final int BRANCHING_FACTOR = 10;
    public static final int RANDOM_SEED = 4356;
    private Vector allObjects;

 /** 
 *  Constructor 
 **/ 
    public SenderScriptGen()
    {
	this.allObjects = new Vector();
    }

 /** 
 *  Make directories 
 **/ 
    private void makeDirectories(LinkedList queue, int numDirs, String prefix)
    {
	int[] a = null;
	int capacity = 0;
	String root = null;
	String dir = null;
	String child = null;
	boolean result = false;

	assert(prefix.endsWith("/"));
	root = prefix;
	queue.addLast(root);
	capacity = 1;
	while(capacity < numDirs){
	    assert(!queue.isEmpty());
	    dir = (String)queue.removeFirst();
	    capacity = capacity - 1;
	    for(int i = 0;
		(i < BRANCHING_FACTOR) && (capacity < numDirs);
		i++){
		assert(dir.endsWith("/"));
		child = dir + (i + 1) + "/";
		queue.addLast(child);
		capacity = capacity + 1;
	    }
	}
    }

 /** 
 *  Add a child to the tree 
 **/ 
    public void makeTree(int numNodes, String prefix)
    {
	LinkedList queue = null;
	int numDirs = 0;
	String currentDirName = null;
	String fileName = null;

	numDirs = (numNodes + BRANCHING_FACTOR - 1) / BRANCHING_FACTOR;
	queue = new LinkedList();
	this.makeDirectories(queue, numDirs, prefix);

	// Now make the actual files
	assert(!queue.isEmpty());
	while(numNodes > 0){
	    currentDirName = (String)queue.removeFirst();
	    assert(currentDirName.endsWith("/"));
	    for(int i = 0; (i < BRANCHING_FACTOR) && (numNodes > 0); i++){
		fileName = currentDirName + (i + 1);
		this.allObjects.add(fileName);
		numNodes--;
	    }
	}
	assert(queue.isEmpty());
    }

 /** 
 *  Print write script entries for all writes 
 **/ 
    public void printAllObjectMods(long fileSize, boolean bound)
    {
	int numObjects = 0;

	System.out.println("c Printing all object writes");
	numObjects = this.allObjects.size();
	for(int i = 0; i < numObjects; i++){
	    System.out.println("w " + (String)this.allObjects.get(i) +
			       " 0 " + fileSize + " " +
			       (bound ? 1 : 0));
	}
    }

 /** 
 *  Randomly generate write events to put into the script 
 **/ 
    public void printRandomWrites(long fileSize, boolean bound, int numWrites)
    {
	String objectName = null;
	int objectIndex = 0;
	int start = 0;
	long len = 0;
	Random rand = null;

	rand = new Random(RANDOM_SEED);
	for(int i = 0; i < numWrites; i++){
	    objectIndex = rand.nextInt(this.allObjects.size());
	    start = rand.nextInt((int)fileSize);
	    len = (long)rand.nextInt((int)(fileSize - start + 1));
	    assert((len + start) <= fileSize);
	    objectName = (String)this.allObjects.get(objectIndex);
	    System.out.println("w " + objectName + " " +
			       start + " " +
			       len + " " +
			       (bound ? 1 : 0));
	}
    }

 /** 
 *  Print script starting messages 
 **/ 
    public static void printScriptPreamble()
    {
	System.out.println("c Start of script");
    }

 /** 
 *  Print script intermediate messages 
 **/ 
    public static void printScriptIntermediate(int numWrites)
    {
	System.out.println("c Wait for B to subscribe + receive all invals");
	System.out.println("l 18000");
	System.out.println("c Do " + numWrites + " random writes");
    }

 /** 
 *  Print script ending messages 
 **/ 
    public static void printScriptPostamble()
    {
	System.out.println("c End of script");
	System.out.println("c Coordinator will now send demand RMIs");
    }

    public void createSequentialWrites(long numFiles, long fileSize,  boolean bound){
	// Note: only generate a full balanced tree if the numFiles is in the form of 10^(x).
	for(long i=0; i<numFiles; i++){
	    String s =  createFileName(i, numFiles);
	    System.out.println("w " + s + " " + 0 + "  " + fileSize + " " + (bound ? 1 : 0));
	}
    }

    public void createPercentageWrites(long fileSize, boolean bound, double dRatio, double fRatio, long numFiles){
	Random r = new Random();
	long numWrites = Math.round(numFiles * dRatio);

	int perDirFile = 1;
	if(fRatio == 0.1){ // 10% files in the directory
	    perDirFile = 10;
	}
	System.out.println(perDirFile);
	for(long i=0; i<numWrites; i=i+perDirFile){
	    String s = createFileName(i, numFiles);
	    long off = Math.abs(r.nextLong())%fileSize;
	    long len = Math.abs(r.nextLong())%(fileSize-off);
	    assert(off+len<=fileSize);
	    System.out.println("w " + s + " " + off + "  " + len + " " + (bound ? 1 : 0));
	}
    }

    public void createRandomWrites(long fileSize, boolean bound, long numWrites, long numFiles){
	Random r = new Random();
	for(long i=0; i<numWrites; i++){
	    long baseName = Math.abs(r.nextLong())%numFiles;
	    String s = createFileName(baseName, numFiles);
	    System.out.println("w " + s + " " + 0 + " " + fileSize + " " + (bound ? 1 : 0));
	}
    }

    private String createFileName(long n, long l){
	long numLen = Math.round(Math.log(l)/Math.log(BRANCHING_FACTOR));
	int fileLen = 2 * (int)numLen; // including /'s as seperator
	byte [] bName = new byte[fileLen];
	for(int i=0; i<fileLen; i++){
	    if(i%2 == 0){// even position
		bName[i] = '/';
	    } else {// odd position
		long base = Math.round(Math.pow(BRANCHING_FACTOR, numLen-(i+1)/2));
		bName[i] = (byte) ('0' + n/base);
		n = n%base;
		assert(bName[i]>='0' && bName[i]<='9');
	    }
	}
	return new String(bName);
    }

 /** 
 *  Program staring point 
 **/ 
    public static void main(String[] argv)
    {
	int numFiles = 0;
	int fileSize = 0;
	int numWrites = 0;
	boolean bound = false;
	String startDirName = null;
	SenderScriptGen ssg = null;
	if(argv.length == 4){
	    ssg = new SenderScriptGen();
	    numFiles = Integer.parseInt(argv[0]);
	    fileSize = Integer.parseInt(argv[1]);
	    assert((argv[2].equals("1")) || (argv[2].equals("0")));
	    bound = (argv[3].equals("1"));
	    numWrites = Integer.parseInt(argv[3]);
	    
	    ssg.printScriptPreamble();
	    ssg.createSequentialWrites(numFiles, fileSize, bound);
	    ssg.printScriptIntermediate(numWrites);
//	    ssg.createRandomWrites(fileSize, bound, numWrites, numFiles);
//	    ssg.printScriptIntermediate(numWrites);
	    ssg.createPercentageWrites(fileSize, bound, 0.1, 1, numFiles);
	    ssg.printScriptPostamble();
/*	    ssg.makeTree(numFiles, startDirName);
	    ssg.printScriptPreamble();
	    ssg.printAllObjectMods(fileSize, bound);
	    ssg.printScriptIntermediate(numWrites);
	    ssg.printRandomWrites(fileSize, bound, numWrites)
	    ssg.printScriptPostamble();*/
	}else{
	    System.err.println("Usage: java SenderScriptGen <numfiles> " +
			       "<startDirName> <fileSize> <bound> " +
			       "<numWrites>");
	}
    }
}
 
 /** 
 *  $Log: SenderScriptGen.java,v $ 
 *  Revision 1.6  2004/05/21 08:08:56  lgao 
 *  VS: ---------------------------------------------------------------------- 
 *  
 *  Revision 1.5  2004/05/19 01:46:37  arun 
 *  *** empty log message *** 
 *  
 *  Revision 1.4  2004/05/18 09:23:37  nayate 
 *  Changed SenderScriptGen to use a chosen random seed 
 *  
 *  Revision 1.3  2004/05/18 08:56:10  nayate 
 *  Changed the sender script generator to first generate entries for all 
 *  files 
 *  
 *  Revision 1.2  2004/05/17 22:06:34  nayate 
 *  Added CVS logging 
 *  
 **/ 
