 /** 
 *  Global Coordinator 
 *  Read scripts from standard input 
 *  Currently have 4 kinds of commands: subscribe, subscribeBody, 
 *                                      issueDemandRead, and sleep 
 **/ 
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintStream;
import java.io.BufferedWriter;
import java.io.FileWriter;
import java.net.InetAddress;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.HashMap;
import java.util.Random;
import java.util.Vector;

public class ByteDieYoung extends CoordExptA
{
  public static long SUBSCRIPTION_DURATION_MS = 1000000;  // 1000 seconds
  public static long MAX_DELAY_MS = 1000; // 1 second

 /** 
 *  Constructor 
 **/ 
  public ByteDieYoung(String exptConfigFileName, String shellType_)
    {
      super(exptConfigFileName, shellType_);
    }
  
  public SubscriptionSet getSubscriptionSet(double percentage)
    {
      SubscriptionSet dis = null;
      if (percentage == 1){//100%
        dis = SubscriptionSet.makeSubscriptionSet(":/*");
      } else if(percentage == 0.1){//10%
        dis = SubscriptionSet.makeSubscriptionSet("/0/*");
      } else if(percentage == 0.01){//1%
        dis = SubscriptionSet.makeSubscriptionSet("/0/0/*");
      } else if(percentage == 0.001){//0.1%
        dis = SubscriptionSet.makeSubscriptionSet("/0/0/0/*");
      } else{
        assert false; // not implemented yet.
      }
      return(dis);
    }

  //-----------------------------------------------------------------------
  // Demand Read the required percentage of files one by one and return the
  // last file demandReaded so that for experiments we know when it's over
  //-----------------------------------------------------------------------
  public String demandReadPartialFiles(int supplierNodeId, 
                                       int receiverNodeId, 
                                       long numTotalFiles,
                                       double percentage,
                                       long fileSize){
    String fileName = null;
    long numFiles = Math.round(numTotalFiles * percentage);
    for(long i = 0; i < numFiles; i++){
      fileName = this.senderScriptGen.createFileName(i, numTotalFiles);
      this.issueDemandRead(supplierNodeId,
                           receiverNodeId,
                           fileName,
                           0,
                           fileSize,
                           i);
    }
    return fileName;
  }

  public void sendMultiPopulateFullScript(int nodeId,
                                          boolean bound,
                                          long numFiles,
                                          long fileSize,
                                          int times){
    OutputStream os = null;
    PrintStream ps = null;
	
    os = this.getProcessOutputStream(nodeId);
    for(int i = 0; i<times; i++){
      this.senderScriptGen.createSequentialWrites(numFiles,
                                                  fileSize,
                                                  bound,
                                                  os);
    }
    ps = new PrintStream(os);
    ps.println("g " + nodeId);
    try{
      os.flush();
    }catch(IOException e){
      System.err.println("Failed to send all object mods over stdout" +
                         e);
      e.printStackTrace();
    }
  }
  //-----------------------------------------------------------------------
  // Full Replication
  //-----------------------------------------------------------------------
  public long fullRepl(int initiater,
                       int subscriber,
                       int times)
    {
      CoordCommPacket ccp = null;
      StatsRecord initStats = null;
      StatsRecord finalStats = null;
      StatsRecord diffStats = null;
      SubscriptionSet dis = null;
        
      long numFiles = 100;  
      long fileSize = 1000; //1kB
      long start, end;
      double avgMS;
      AcceptStamp asAfterWrite, asAfterRead;
      int[] nodeList = new int[2];
      nodeList[0] = initiater;
      nodeList[1] = subscriber;
      this.restartRMIRegistrySync();
      
      this.startNode(initiater);
      this.startNode(subscriber);
      this.waitStartNode(initiater);
      this.waitStartNode(subscriber);

      initStats = this.getStats(nodeList);
      
      //----------------------------------------------------
      // case 1. full replication
      //----------------------------------------------------

      //Write all files with bound inval
      this.sendMultiPopulateFullScript(initiater, 
				       true, //bound
				       numFiles, //numFiles
				       fileSize,
				       times);   // file size

      ccp = this.recv(initiater);
      assert(ccp.getSignalType() == CoordCommPacket.ACCEPT_STAMP);
      assert(ccp.getData() instanceof AcceptStamp);
        
      asAfterWrite = (AcceptStamp)ccp.getData();

      dis = SubscriptionSet.makeSubscriptionSet(":/*");

      initStats = this.getAllStats();
      this.issueSubscribe(initiater,
                          subscriber,
                          AcceptVV.makeVVAllNegatives(),
      
                          dis,
                          SUBSCRIPTION_DURATION_MS,
                          MAX_DELAY_MS);
      this.sendSync(subscriber, asAfterWrite);
      ccp = this.recv(subscriber);
      assert(ccp.getSignalType() == CoordCommPacket.SYNC_DONE);
      assert(ccp.getData() == null);
      System.err.println("Received a SYNC_DONE");
      // post condition:
      // Sender and receiver have synchronized for the first initial
      // sequential writes.
        
      finalStats = this.getStats(nodeList);
      diffStats = finalStats.getDiff(initStats);
      System.out.println("-------------CASE 1 Full Replication-----------");
      System.out.println("****x=" + times + " Full:" + diffStats);
      
      System.out.println("Exit value for node " + initiater + " = " + this.killNode(initiater));
      System.out.println("Exit value for node " + subscriber + " = " + this.killNode(subscriber));

      return diffStats.getTotal();
    }
  //----------------------------------------------------------------
  // Case 2 separate data from metadata
  //        do the same thing as above except that now
  //        (1) the sender writes with unbound inval
  //        (2) the receiver subscribes all files for invals and 
  //            gets data by demandread partial files. 
  //----------------------------------------------------------------
  public long partialRepl(int initiater,
                          int subscriber,
                          int times){
    
    CoordCommPacket ccp = null;
    StatsRecord initStats = null;
    StatsRecord finalStats = null;
    StatsRecord diffStats = null;
    SubscriptionSet dis = null;
        
    long numFiles = 100;  
    long fileSize = 1000; //1kB
    long start, end;
    double avgMS;
    AcceptStamp asAfterWrite, asAfterRead;
    int[] nodeList = new int[2];
    nodeList[0] = initiater;
    nodeList[1] = subscriber;
    this.restartRMIRegistrySync();
      
    this.startNode(initiater);
    this.startNode(subscriber);
    this.waitStartNode(initiater);
    this.waitStartNode(subscriber);

    initStats = this.getStats(nodeList);
    //Write all files with unbound inval
    this.sendMultiPopulateFullScript(initiater, 
                                     false, //bound
                                     numFiles, //numFiles
                                     fileSize,
                                     times);   // file size
    ccp = this.recv(initiater);
    assert(ccp.getSignalType() == CoordCommPacket.ACCEPT_STAMP);
    assert(ccp.getData() instanceof AcceptStamp);
        
    asAfterWrite = (AcceptStamp)ccp.getData();

    initStats = this.getAllStats();
    this.issueSubscribe(initiater,
                        subscriber,
                        AcceptVV.makeVVAllNegatives(),
                        
                        SubscriptionSet.makeSubscriptionSet(":/*"),
                        SUBSCRIPTION_DURATION_MS,
                        MAX_DELAY_MS);
    this.sendSync(subscriber, asAfterWrite);
    ccp = this.recv(subscriber);
    assert(ccp.getSignalType() == CoordCommPacket.SYNC_DONE);
    assert(ccp.getData() == null);
    System.err.println("Received a SYNC_DONE");
      
    String lastFile = this.demandReadPartialFiles(initiater,
                                                  subscriber,
                                                  numFiles,
                                                  1,
                                                  fileSize);
      
    this.sendRead(subscriber, lastFile, 0, fileSize);
    //System.err.println("222222222");
    ccp = this.recv(subscriber);
    //System.err.println("333333333333");
    assert(ccp.getSignalType() == CoordCommPacket.READ_DONE);
    assert(ccp.getData() instanceof AcceptStamp);
        	
    finalStats = this.getStats(nodeList);
    diffStats = finalStats.getDiff(initStats);
    System.out.println("---------CASE 2 Subscribe all inval --------------");
    System.out.println("******* x=" + times+" PRACTI: " + diffStats);
      
    System.out.println("Exit value for node " + initiater + " = " + this.killNode(initiater));
    System.out.println("Exit value for node " + subscriber + " = " + this.killNode(subscriber));

    return diffStats.getTotal();
  }

  //-----------------------------------------------------------------------
  // Program starting point
  //-----------------------------------------------------------------------
  public static void main(String[] argv)
    {
      ByteDieYoung coord = null;

      if (argv.length < 3){
        System.out.println("Usage: <exptConfigfile> <shellType> <times>");
        System.exit(-1);
      }
      coord = new ByteDieYoung(argv[0], argv[1]);
      int times = new Integer(argv[2]).intValue();
      long full = coord.fullRepl(9, 10, times);
      long pr = coord.partialRepl(11, 12, times);
      String dataFileName = System.getProperty("user.dir")+"/experiments/sosp/results/BDY.data";
      try{
        BufferedWriter bw = new BufferedWriter(new FileWriter(dataFileName, true));
        assert bw!=null;
        String data = times + " " + full
          +" " + pr
          + "\n"; 
        bw.write(data, 0, data.length());
        bw.flush();
      
      } catch (Exception e){
        e.printStackTrace();
        assert(false);
      }
    }
}

//---------------------------------------------------------------------------
// $Log: ByteDieYoung.java,v $
// Revision 1.3  2005/03/02 21:44:07  zjiandan
// Modified to work with new code
//
// Revision 1.2  2005/02/28 20:26:00  zjiandan
// Added Garbage Collection code and part of Checkpoint exchange protocol code
//
// Revision 1.1  2005/01/13 20:55:41  zjiandan
// Reorganized sosp experiments files into sosp subdirectory under experiments.
//
// Revision 1.1  2005/01/10 03:50:30  zjiandan
// Source files for the first three Experiments.
//
//---------------------------------------------------------------------------
