 /** 
 *  Global Coordinator 
 *  Read scripts from standard input 
 *  Currently have 4 kinds of commands: subscribe, subscribeBody, 
 *                                      issueDemandRead, and sleep 
 **/ 
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintStream;
import java.net.InetAddress;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.HashMap;
import java.util.Vector;

public class CoordExptA
{
  private HashMap processTable;
  private HashMap errorStreamTable;
  private RMIClient rmiClient;
  protected long numFiles ;
  private String shellType;
  protected ExptConfig exptConfig;
  protected SenderScriptGen senderScriptGen;

  public static long SUBSCRIPTION_DURATION_MS = 86400000; // 1 day
  public static long MAX_DELAY_MS = 1000; // 1 second

 /** 
 *  Constructor 
 **/ 
  public CoordExptA(String exptConfigFileName, String shellType_){
    this.exptConfig = new ExptConfig(exptConfigFileName);
    Config.readConfig(exptConfig.getConfigFileName());
    this.processTable = new HashMap();
    this.errorStreamTable = new HashMap();
    this.senderScriptGen = new SenderScriptGen();
    this.numFiles = this.exptConfig.getNumFiles();
    // this.senderScriptGen.makeTree(this.exptConfig.getNumFiles(), "/");
    this.rmiClient = new RMIClient();
    this.shellType = shellType_;
  }

 /** 
 *  Get statistics from all machines 
 **/ 
  public StatsRecord getAllStats(){
    Stats.reset();
    for (Enumeration e = Config.getKnownNodeIds(); e.hasMoreElements();){
      NodeId id = (NodeId)e.nextElement();
      Hashtable h = null;
      try{
        h = this.rmiClient.getStats(id);
      } catch (Exception exc){
        exc.printStackTrace();
        assert(false);
        System.exit(-1);
      }
      if((h == null) || (h.size() == 0)){
        Env.printDebug("Machine " + Config.getRMIUrl(id) +
                       " sent no data");
      } else {
        for(Enumeration entry = h.keys(); entry.hasMoreElements();){
          String s = (String)entry.nextElement();
          Long value = (Long)h.get(s);
          assert(value != null);
          Stats.put(s,value.longValue());
        }
      }
    }
    return Stats.createRecord();
  }

 /** 
 *  Get statistics from two machines 
 **/ 
  public StatsRecord getStats(int[] nodeList){
    Stats.reset();
    for (int i=0; i<nodeList.length; i++){
      NodeId id = (NodeId)new NodeId(nodeList[i]);
      Hashtable h = null;
      try{
        h = this.rmiClient.getStats(id);
      } catch (Exception exc){
        exc.printStackTrace();
        assert(false);
        System.exit(-1);
      }
      if((h == null) || (h.size() == 0)){
        Env.printDebug("Machine " + Config.getRMIUrl(id) +
                       " sent no data");
      } else {
        for(Enumeration entry = h.keys(); entry.hasMoreElements();){
          String s = (String)entry.nextElement();
          Long value = (Long)h.get(s);
          assert(value != null);
          Stats.put(s,value.longValue());
        }
      }
    }
    return Stats.createRecord();
  }
  /*
  //-----------------------------------------------------------------------
  // Kill previous instances
  //-----------------------------------------------------------------------
  public void killAllSync()
  {
  Runtime runtime = null;
  Process process = null;

  try{
  runtime = Runtime.getRuntime();
  process = runtime.exec("./killall.sh");
  process.waitFor();
  }catch(IOException e){
  System.err.println("Failed to killall: " + e);
  e.printStackTrace();
  }catch(InterruptedException e){
  System.err.println("Got interrupted: " + e);
  e.printStackTrace();
  }
  }
  */

 /** 
 *  Restart the RMI registry 
 **/ 
  public void restartRMIRegistrySync(){
    Runtime runtime = null;
    Process process = null;

    try{
      runtime = Runtime.getRuntime();
      process = runtime.exec("./killRMIRegistry.sh");
      process.waitFor();
    }catch(IOException e){
      System.err.println("Failed to kill RMI registry: " + e);
      e.printStackTrace();
    }catch(InterruptedException e){
      System.err.println("Got interrupted: " + e);
      e.printStackTrace();
    }
    try{
      runtime = Runtime.getRuntime();
      process = runtime.exec("rmiregistry");
    }catch(IOException e){
      System.err.println("Failed to start RMI registry: " + e);
      e.printStackTrace();
    }
  }


 /** 
 *  Restart the RMI registry on remote machines 
 **/ 
  public void restartRMIRegistrySync(int nodeId){
    Runtime runtime = null;
    Process process = null;
    String hostName = null;
    String pwd = null;
    String runStr = null;
	
    try{
      hostName = Config.getDNS(new NodeId(nodeId));
      pwd = System.getProperty("user.dir");
      if(this.shellType.equalsIgnoreCase("bash")){
        runStr = "ssh " + hostName + " " +
          "export CLASSPATH=" + pwd +
          ":" + pwd + "/classes:" + pwd +
          "/je-1.5.1/lib/je.jar:" + pwd +
          "/experiments/sosp/classes:" + pwd +
          "/experiments/expt1/classes:" + pwd +
          "/experiments/expt2/classes:" + pwd +
          "/experiments/expt3/classes; " +
          "cd " + pwd + "; " +
          "./killall.sh; ./rmiregistry.sh";
      }else if(this.shellType.equalsIgnoreCase("tcsh")){
        runStr = "ssh " + hostName + " " +
          "setenv PATH /lusr/java2/bin; " +
          "setenv CLASSPATH " + pwd +
          ":" + pwd + "/classes:" + pwd +
          "/je-1.5.1/lib/je.jar:" +
          pwd  +"/experiments/sosp/classes:"+
          pwd  +"/experiments/expt1/classes:"+
          pwd  +"/experiments/expt2/classes:"+
          pwd +"/experiments/expt3/classes; " +
          "cd " + pwd + "; " +
          "./killall.sh; ./rmiregistry.sh";
      }else{
        System.err.println("Unrecognized shell " + this.shellType);
        System.exit(-1);
      }
      System.err.println(runStr);
      runtime = Runtime.getRuntime();
      process = runtime.exec(runStr);
      process.waitFor();
    }catch(IOException e){
      System.err.println("Failed to kill RMI registry: " + e);
      System.err.println("or failed to start RMI registry: " + e);
      e.printStackTrace();
    }catch(InterruptedException e){
      System.err.println("Got interrupted: " + e);
      e.printStackTrace();
    }
  }

 /** 
 *  Start a node 
 **/ 
  public void startNode(int nodeId){
    Runtime runtime = null;
    Process process = null;
    String pwd = null;
    String runStr = null;
    ErrPrintThread errPrintThread = null;
    InetAddress localAddress = null;
	
    String hostName = Config.getDNS(new NodeId(nodeId));

    try{
      localAddress = InetAddress.getLocalHost();
      pwd = System.getProperty("user.dir");
      if(this.shellType.equalsIgnoreCase("bash")){
        runStr = "ssh " + hostName + " " +
          "export CLASSPATH=" + pwd +
          ":" + pwd + "/classes:" + pwd +
          "/je-1.5.1/lib/je.jar:" + pwd +
          "/experiments/sosp/classes:" + pwd +
          "/experiments/expt1/classes:" + pwd +
          "/experiments/expt2/classes:" + pwd +
          "/experiments/expt3/classes; " +
          "cd " + pwd + "; " +
          "java -esa -ea " +
          "-Djava.rmi.server.codebase=\"file://" +
          pwd + "/classes/ " + "file://" + pwd +
          "/experiments/sosp/classes/\" " +
          "-Djava.security.policy=" + pwd +
          "/java.security.SecurityPermission URANode " +
          this.exptConfig.getConfigFileName() + " " + nodeId;
      }else if(this.shellType.equalsIgnoreCase("tcsh")){
        runStr = "ssh " + hostName + " " +
          "setenv PATH /lusr/java2/bin; " +
          "setenv CLASSPATH " + pwd +
          ":" + pwd + "/classes:" + pwd +
          "/je-1.5.1/lib/je.jar:" +
          pwd +"/experiments/sosp/classes:"+
          pwd +"/experiments/expt1/classes:"+
          pwd +"/experiments/expt2/classes:"+
          pwd +"/experiments/expt3/classes; " +
          "cd " + pwd + "; " +
          "java -esa -ea " +
          "-Djava.rmi.server.codebase=\"file://" +
          pwd + "/classes/ " + "file://" + pwd +
          "/experiments/sosp/classes/\" " +
          "-Djava.security.policy=" + pwd +
          "/java.security.SecurityPermission URANode " +
          this.exptConfig.getConfigFileName() + " " + nodeId;
      }else{
        System.err.println("Unrecognized shell " + this.shellType);
        System.exit(-1);
      }
      System.err.println(runStr);
      runtime = Runtime.getRuntime();
      process = runtime.exec(runStr);
      this.processTable.put(new NodeId(nodeId), process);
      errPrintThread = new ErrPrintThread(nodeId,
                                          process.getErrorStream());
      this.errorStreamTable.put(new NodeId(nodeId), errPrintThread);
    }catch(IOException e){
      System.err.println("Failed to start node: " + e);
      e.printStackTrace();
    }
  }

 /** 
 *  Wait for a node to signal that it has started 
 **/ 
  public void waitStartNode(int nodeId){
    CoordCommPacket ccp = null;
    InputStream is = null;
    Process nodeProcess = null;

    nodeProcess = (Process)this.processTable.get(new NodeId(nodeId));
    assert(nodeProcess != null);
    is = nodeProcess.getInputStream();
    try{
      ccp = CoordCommPacket.makePacket(is);
      assert(ccp.getSignalType() == CoordCommPacket.AWAKE_SIGNAL);
    }catch(IOException e){
      System.err.println("Failed to read from standard input: " + e);
      e.printStackTrace();
    }catch(ClassNotFoundException e){
      System.err.println("Failed to read from standard input: " + e);
      e.printStackTrace();
    }
  }

 /** 
 *  Kill a node 
 **/ 
  public int killNode(int nodeId){
    Process process = null;
    OutputStream os = null;
    InputStream err = null;
    PrintStream ps = null;
    ErrPrintThread errThread = null;
    int exitValue = 0;

    process = (Process)this.processTable.get(new NodeId(nodeId));
    errThread =
      (ErrPrintThread)this.errorStreamTable.get(new NodeId(nodeId));
    assert(process != null);
    assert(errThread != null);

    // Send an "e" to the process
    os = this.getProcessOutputStream(nodeId);
    ps = new PrintStream(os);
    ps.println("e");
    ps.flush();
    ps.close();
    errThread.setRunFlag(false);
    try{
      Thread.sleep(2000);
      process.destroy();
    }catch(InterruptedException e){
      System.err.println("Could not wait for " + nodeId +
                         "to finish: " + e);
      e.printStackTrace();
    }
    return(exitValue);
  }

 /** 
 *  Send the populate script to standard output for node nodeId 
 *  -- send the sequential writes to each file of the required number of files 
 *     to the receive node and finally attach a "get acceptStamp request" 
 **/ 
  public void sendPopulateFullScript(int nodeId){
    OutputStream os = null;
    PrintStream ps = null;

    os = this.getProcessOutputStream(nodeId);
    this.senderScriptGen.createSequentialWrites(this.numFiles,
                                                this.exptConfig.getFileSize(),
                                                false,
                                                os);
    ps = new PrintStream(os);
    ps.println("g " + nodeId);
    try{
      os.flush();
    }catch(IOException e){
      System.err.println("Failed to send all object mods over stdout" +
                         e);
      e.printStackTrace();
    }
  }

 /** 
 *  Send the populate script to standard output for node nodeId, but use 
 *  bound invals 
 **/ 
  public void sendPopulateFullScriptBound(int nodeId){
    OutputStream os = null;
    PrintStream ps = null;

    os = this.getProcessOutputStream(nodeId);
    this.senderScriptGen.createSequentialWrites(this.numFiles,
                                                this.exptConfig.getFileSize(),
                                                true,
                                                os);
    ps = new PrintStream(os);
    ps.println("g " + nodeId);
    try{
      os.flush();
    }catch(IOException e){
      System.err.println("Failed to send all object mods over stdout" +
                         e);
      e.printStackTrace();
    }
  }

 /** 
 *  Send the populate script to standard output for node nodeId, but use 
 *  bound invals 
 **/ 
  public void sendPopulateFullScript(int nodeId, 
                                     boolean bound,
                                     long numFiles, 
                                     long fileSize){
    OutputStream os = null;
    PrintStream ps = null;

    os = this.getProcessOutputStream(nodeId);
    this.senderScriptGen.createSequentialWrites(numFiles,
                                                fileSize,
                                                bound,
                                                os);
    ps = new PrintStream(os);
    ps.println("g " + nodeId);
    try{
      os.flush();
    }catch(IOException e){
      System.err.println("Failed to send all object mods over stdout" +
                         e);
      e.printStackTrace();
    }
  }
  
 /** 
 *  Send the sequential read script to standard output for node nodeId,  
 *  and wait for the acceptstamp   "r ... r... r... .... g" 
 **/ 
  public void sendSequentialReadScript(int nodeId, 
                                     long numFiles, 
                                     long fileSize){
    OutputStream os = null;
    PrintStream ps = null;

    os = this.getProcessOutputStream(nodeId);
    this.senderScriptGen.createSequentialNonBlockReads(numFiles,
						       fileSize,
						       os);
    ps = new PrintStream(os);
    ps.println("d ");
    try{
      os.flush();
    }catch(IOException e){
      System.err.println("Failed to send all object mods over stdout" +
                         e);
      e.printStackTrace();
    }
  }
  public void sendPartialWritesScript(int nodeId, double dirPct, double filePct, boolean bound){
    OutputStream os = null;
    PrintStream ps = null;
	
    assert(dirPct>0 && dirPct<=1);
    assert(filePct>0 && filePct<=1);
	
    os = this.getProcessOutputStream(nodeId);
    this.senderScriptGen.createPercentageWrites(this.exptConfig.getFileSize(),
                                                bound,
                                                dirPct,
                                                filePct,
                                                this.numFiles,
                                                os);
    ps = new PrintStream(os);
    ps.println("g " + nodeId);
    try{
      os.flush();
    }catch (IOException e){
      System.err.println("Failed to send partial file  writes " +
                         "over stdout" + e);
      e.printStackTrace();
    }
  }

 /** 
 *  Send the random writes script to standard output for node nodeId 
 **/ 
  public void sendRandomWritesScript(int nodeId){

    boolean boundInvals = (this.exptConfig.getExptNum() == 1);
    sendRandomWritesScript(nodeId,
                           this.exptConfig.getNumWrites(),
                           this.numFiles,
                           1,
                           boundInvals);
  }

 /** 
 *  Send the random writes script to standard output for node nodeId 
 **/ 
  public void sendRandomWritesScript(int nodeId, boolean bound){

    sendRandomWritesScript(nodeId,
                           this.exptConfig.getNumWrites(),
                           this.numFiles,
                           1,
                           bound);
  }

  public void sendRandomWritesScript(int nodeId,
                                     long numWrites,
                                     long numFiles,
                                     double ratio,
                                     boolean bound){
    OutputStream os = null;
    PrintStream ps = null;
    boolean boundInvals = bound;

    os = this.getProcessOutputStream(nodeId);
    this.senderScriptGen.createRandomWrites(this.exptConfig.getFileSize(),
                                            boundInvals,
                                            numWrites,
                                            numFiles,
                                            ratio,
                                            os);
    ps = new PrintStream(os);
    ps.println("g " + nodeId);
    try{
      os.flush();
    }catch(IOException e){
      System.err.println("Failed to send random object writes " +
                         "over stdout" + e);
      e.printStackTrace();
    }
  }

 /** 
 *  Send a sync message to the receiver data store and wait for a reply 
 **/ 
  public void sendSync(int nodeId, AcceptStamp as){
    OutputStream os = null;
    PrintStream ps = null;

    os = this.getProcessOutputStream(nodeId);
    ps = new PrintStream(os);
    ps.println("s " + as.getNodeId() + " " + as.getLocalClock());
    try{
      os.flush();
    }catch(IOException e){
      System.err.println("Failed to send all object mods over stdout" +
                         e);
      e.printStackTrace();
    }
  }

 /** 
 *  Send a sync message to the receiver log and wait for a reply 
 **/ 
  public void sendLogSync(int nodeId, AcceptStamp as){
    OutputStream os = null;
    PrintStream ps = null;

    os = this.getProcessOutputStream(nodeId);
    ps = new PrintStream(os);
    ps.println("o " + as.getNodeId() + " " + as.getLocalClock());
    ps.flush();
  }
    

 /** 
 *  Send a sync message to the receiver and wait for a reply 
 **/ 
  public void sendSyncBody(int[] nodeIds){
    OutputStream os = null;
    PrintStream ps = null;
    boolean allDone = false;
    CoordCommPacket resp_ccp = null;
    Vector waitingList = new Vector();

    while(!allDone){
      // sending out the signal
      for(int i = 0; i < nodeIds.length; i++){
        os = this.getProcessOutputStream(nodeIds[i]);
        ps = new PrintStream(os);
        ps.println("b ");
        try{
          os.flush();
        }catch(IOException e){
          System.err.println("Failed to send all object mods over stdout" +
                             e);
          e.printStackTrace();
        }
      }
	    
      // receiving the signal : false will be returned if the corresponding queue is not empty
      for(int i = 0; i< nodeIds.length; i++){
        resp_ccp = recv(nodeIds[i]);
        assert(resp_ccp.getSignalType() == CoordCommPacket.SYNC_BODY_DONE);
        assert(resp_ccp.getData() != null);
        boolean isDone = ((Boolean)resp_ccp.getData()).booleanValue();
        if(!isDone){
          waitingList.add(new Integer(nodeIds[i]));
        } 
      }

      // wait until the current non-empty queue to clear
      for(Enumeration e = waitingList.elements(); e.hasMoreElements();){
        allDone = false; // if at least one queue is non-empty, retry later.
        int id = ((Integer)e.nextElement()).intValue();
        resp_ccp = recv(id);
        assert(resp_ccp.getSignalType() == CoordCommPacket.SYNC_BODY_RETRY);
        assert(resp_ccp.getData() == null);
        // no need to check the value.  We will check again if needed.
      }
    }
  }

 /** 
 *  Send a read message to the receiver and wait for a reply 
 **/ 
  public void sendRead(int nodeId, String filename, long offset, long length){
    OutputStream os = null;
    PrintStream ps = null;

    os = this.getProcessOutputStream(nodeId);
    ps = new PrintStream(os);
    ps.println("r " + filename + " " + offset + " " + length);
    ps.flush();
  }

 /** 
 *  Send a write message to the receiver and wait for a reply 
 **/ 
  public void sendWrite(int nodeId, String filename, long offset, long length, boolean bound){
    OutputStream os = null;
    PrintStream ps = null;

    os = this.getProcessOutputStream(nodeId);
    ps = new PrintStream(os);
    ps.println("w " + filename + " " + offset + " " + length + " " + (bound ? 1 : 0)
               + " " + 0);
    ps.println("g " + nodeId);
    ps.flush();
  }


 /** 
 *  Receive something from standard input written by nodeId 
 **/ 
  public CoordCommPacket recv(int nodeId){
    InputStream is = null;
    CoordCommPacket ccp = null;

    try{
      is = this.getProcessInputStream(nodeId);
      ccp = CoordCommPacket.makePacket(is);
    }catch(IOException e){
      System.err.println("Could not receive data from stdin of node " +
                         nodeId + ": " + e);
      e.printStackTrace();
    }catch(ClassNotFoundException e){
      System.err.println("Could not receive data from stdin of node " +
                         nodeId + ": " + e);
      e.printStackTrace();
      assert(false);
    }
    return(ccp);
  }

 /** 
 *  Get the interest set of the receiver (depends on the experiment) 
 **/ 
  public SubscriptionSet getInterestSet(){
    SubscriptionSet dis = null;

    switch(exptConfig.getExptNum()){
    case 1:
      dis = SubscriptionSet.makeSubscriptionSet(":/*");
      break;
    case 2:
      dis = SubscriptionSet.makeSubscriptionSet(":/*");
      break;
    default:
      System.err.println("Unimplemented");
      System.exit(-1);
      break;
    }
    return(dis);
  }

  //-----------------------------------------------------------------------
  // Ask a sender to demand fetch an object to a receiver
  //-----------------------------------------------------------------------
  public void issueDemandRead(int supplierNodeId,
                              int receiverNodeId,
                              String objIdStr,
                              long offset,
                              long len,
                              long requestId){
    ObjId objId = null;
    NodeId supplier = null;
    NodeId receiver = null;

    try{
      objId = new ObjId(objIdStr);
      supplier = new NodeId(supplierNodeId);
      receiver = new NodeId(receiverNodeId);
      rmiClient.issueDemandRead(objId,
                                offset,
                                len,
                                requestId,
                                supplier,
                                receiver,
                                Config.getDNS(receiver),
                                Config.getPortBody(receiver));
    }catch(RMINetworkException e){
      System.err.println("Could not tell " + supplierNodeId + " to " +
                         "demand push to " + receiverNodeId + ": " +
                         e);
      e.printStackTrace();
    }catch(RMIApplicationException e){
      e.printStackTrace();
      assert false:("Could not tell " + supplierNodeId + " to " +
                    "demand push to " + receiverNodeId + ": " + e);
    }catch(ObjNotFoundException e){
      e.printStackTrace();
      assert false:("Could not tell " + supplierNodeId + " to " +
                    "demand push to " + receiverNodeId + ": " + e);
    }catch(IOException e){
      e.printStackTrace();
      assert false:("Could not tell " + supplierNodeId + " to " +
                    "demand push to " + receiverNodeId + ": " + e);
    }catch(ReadOfInvalidRangeException e){
      e.printStackTrace();
      assert false;
    }

  }

  //-----------------------------------------------------------------------
  // Ask the receiver to subscribe to the sender to hear invalidates without 
  // any propagation delay
  //-----------------------------------------------------------------------
  public void issueSubscribe(int supplierNodeId,
                             int subscriberNodeId,
                             VV startVV,
                             
                             SubscriptionSet dis,
                             long subDurationMS,
                             long maxForceMS){
    NodeId supplier = null;
    NodeId subscriber = null;

    try{
      supplier = new NodeId(supplierNodeId);
      subscriber = new NodeId(subscriberNodeId);
      rmiClient.subscribe(dis,
                          startVV,
                          
                          supplier,
                          subscriber,
                          Config.getDNS(subscriber),
                          Config.getPortInval(subscriber),
                          subDurationMS,
                          maxForceMS,
                          0);
    }catch(RMINetworkException e){
      System.err.println("Could not tell " + supplierNodeId + " to " +
                         "send invals to " + subscriberNodeId + ": " +
                         e);
      e.printStackTrace();
    }catch(RMIApplicationException e){
      e.printStackTrace();
      assert false:("Could not tell " + supplierNodeId + " to " +
                    "send invals to " + subscriberNodeId + ": " + e);
    }
  }

  //-----------------------------------------------------------------------
  // Ask the receiver to subscribe to the sender to hear invalidates till
  // endVV is reached.
  //-----------------------------------------------------------------------
  public void issueSubscribe(int supplierNodeId,
                             int subscriberNodeId,
                             VV startVV,
                             VV endVV,//different from the above method
                             SubscriptionSet dis,
                             long subDurationMS,
                             long maxForceMS){
    NodeId supplier = null;
    NodeId subscriber = null;

    try{
      supplier = new NodeId(supplierNodeId);
      subscriber = new NodeId(subscriberNodeId);
      rmiClient.subscribe(dis,
                          startVV,
                          endVV,
                          
                          supplier,
                          subscriber,
                          Config.getDNS(subscriber),
                          Config.getPortInval(subscriber),
                          subDurationMS,
                          maxForceMS, 
                          0);
    }catch(RMINetworkException e){
      System.err.println("Could not tell " + supplierNodeId + " to " +
                         "send invals to " + subscriberNodeId + ": " +
                         e);
      e.printStackTrace();
    }catch(RMIApplicationException e){
      e.printStackTrace();
      assert false:("Could not tell " + supplierNodeId + " to " +
                    "send invals to " + subscriberNodeId + ": " + e);
    }
  }

  //-----------------------------------------------------------------------
  // Ask the receiver to subscribe to the sender to hear invalidates till
  // endVV is reached.
  //-----------------------------------------------------------------------
  public void issueSubscribe(int supplierNodeId,
                             int subscriberNodeId,
                             VV startVV,
                             VV endVV,//different from the above method
                             SubscriptionSet dis,
                             long subDurationMS,
                             long maxForceMS,
                             long maxDelayMS){
    NodeId supplier = null;
    NodeId subscriber = null;

    try{
      supplier = new NodeId(supplierNodeId);
      subscriber = new NodeId(subscriberNodeId);
      rmiClient.subscribe(dis,
                          startVV,
                          endVV,
                          
                          supplier,
                          subscriber,
                          Config.getDNS(subscriber),
                          Config.getPortInval(subscriber),
                          subDurationMS,
                          maxForceMS, 
                          maxDelayMS);
    }catch(RMINetworkException e){
      System.err.println("Could not tell " + supplierNodeId + " to " +
                         "send invals to " + subscriberNodeId + ": " +
                         e);
      e.printStackTrace();
    }catch(RMIApplicationException e){
      e.printStackTrace();
      assert false:("Could not tell " + supplierNodeId + " to " +
                    "send invals to " + subscriberNodeId + ": " + e);
    }
  }


  //-----------------------------------------------------------------------
  // Ask the receiver to subscribe to the sender to hear invalidates without 
  // any propagation delay
  //-----------------------------------------------------------------------
  public void issueSubscribe(int supplierNodeId,
                             int subscriberNodeId,
                             VV startVV,
                             SubscriptionSet dis,
                             long subDurationMS,
                             long maxForceMS,
                             long maxDelayMS){
    NodeId supplier = null;
    NodeId subscriber = null;

    try{
      supplier = new NodeId(supplierNodeId);
      subscriber = new NodeId(subscriberNodeId);
      rmiClient.subscribe(dis,
                          startVV,
                          supplier,
                          subscriber,
                          Config.getDNS(subscriber),
                          Config.getPortInval(subscriber),
                          subDurationMS,
                          maxForceMS,
                          maxDelayMS);
    }catch(RMINetworkException e){
      System.err.println("Could not tell " + supplierNodeId + " to " +
                         "send invals to " + subscriberNodeId + ": " +
                         e);
      e.printStackTrace();
    }catch(RMIApplicationException e){
      e.printStackTrace();
      assert false:("Could not tell " + supplierNodeId + " to " +
                    "send invals to " + subscriberNodeId + ": " + e);
    }
  }

  //-----------------------------------------------------------------------
  // Ask the receiver to subscribe to the sender to hear updates
  //-----------------------------------------------------------------------
  public void issueSubscribeBody(int supplierNodeId,
                                 int subscriberNodeId,
                                 VV startVV,
                                 SubscriptionSet dis,
                                 long subDurationMS){
    NodeId supplier = null;
    NodeId subscriber = null;

    try{
      supplier = new NodeId(supplierNodeId);
      subscriber = new NodeId(subscriberNodeId);
      rmiClient.subscribeBody(dis,
                              startVV,
                              supplier,
                              subscriber,
                              Config.getDNS(subscriber),
                              Config.getPortBody(subscriber),
                              subDurationMS);
    }catch(RMINetworkException e){
      System.err.println("Could not tell " + supplierNodeId + " to " +
                         "send updates to " + subscriberNodeId + ": " +
                         e);
      e.printStackTrace();
    }catch(RMIApplicationException e){
      e.printStackTrace();
      assert false:("Could not tell " + supplierNodeId + " to " +
                    "send invals to " + subscriberNodeId + ": " + e);
    }
  }

  //-----------------------------------------------------------------------
  // Ask the receiver to subscribe to the sender to hear updates
  //-----------------------------------------------------------------------
  public void
  issueSubscribeCheckpoint(int supplierNodeId,
                           int subscriberNodeId,
                           String cpPath,
                           String[] exclChildNames,
                           VV startVV){
    NodeId supplier = null;
    NodeId subscriber = null;

    try{
      supplier = new NodeId(supplierNodeId);
      subscriber = new NodeId(subscriberNodeId);
      rmiClient.subscribeCheckpoint(cpPath,
                                    exclChildNames,
                                    startVV,
                                    supplier,
                                    subscriber,
                                    Config.getDNS(subscriber),
                                    Config.getPortCP(subscriber));
    }catch(RMINetworkException e){
      System.err.println("Could not tell " + supplierNodeId + " to " +
                         "send a checkpoint to " + subscriberNodeId + ": " +
                         e);
      e.printStackTrace();
    }catch(RMIApplicationException e){
      e.printStackTrace();
      assert false:("Could not tell " + supplierNodeId + " to " +
                    "send a checkpoint to " + subscriberNodeId + ": " + e);
    }
  }

  //-----------------------------------------------------------------------
  // Send the populate script to standard output for node nodeId
  //-----------------------------------------------------------------------
  public OutputStream getProcessOutputStream(int nodeId){
    Process process = null;

    process = (Process)this.processTable.get(new NodeId(nodeId));
    assert(process != null);
    return(process.getOutputStream());
  }

  //-----------------------------------------------------------------------
  // Send the populate script to standard output for node nodeId
  //-----------------------------------------------------------------------
  public InputStream getProcessInputStream(int nodeId){
    Process process = null;

    process = (Process)this.processTable.get(new NodeId(nodeId));
    assert(process != null);
    return(process.getInputStream());
  }

  //-----------------------------------------------------------------------
  // Perform experiment actions
  //-----------------------------------------------------------------------
  public static void runExperiment(CoordExptA expt){
    CoordCommPacket ccp = null;
    StatsRecord initStats = null;
    StatsRecord finalStats = null;
    StatsRecord diffStats = null;
    SubscriptionSet dis = null;

    //expt.killAllSync();
    expt.restartRMIRegistrySync();
    expt.startNode(9);
    expt.startNode(10);
    expt.waitStartNode(9);
    expt.waitStartNode(10);

    // Send over the script to write to every file on machine 9
    expt.sendPopulateFullScriptBound(9);
    ccp = expt.recv(9);
    assert(ccp.getSignalType() == CoordCommPacket.ACCEPT_STAMP);
    assert(ccp.getData() instanceof AcceptStamp);

    // Tell machine 10 to hear all invalidates
    dis = expt.getInterestSet();
    expt.issueSubscribe(9,
                        10,
                        AcceptVV.makeVVAllNegatives(),
  
                        dis,
                        SUBSCRIPTION_DURATION_MS,
                        MAX_DELAY_MS);
    expt.sendSync(10, (AcceptStamp)ccp.getData());
    ccp = expt.recv(10);
    assert(ccp.getSignalType() == CoordCommPacket.SYNC_DONE);
    assert(ccp.getData() == null);
    System.err.println("Received a SYNC_DONE");
    // NOTE: The stats collected on the line below include all writes
    // to all files (bound invalidates)
    initStats = expt.getAllStats();

    // Send over random writes
    expt.sendRandomWritesScript(9, true);
    ccp = expt.recv(9);
    assert(ccp.getSignalType() == CoordCommPacket.ACCEPT_STAMP);
    assert(ccp.getData() instanceof AcceptStamp);
    expt.sendSync(10, (AcceptStamp)ccp.getData());
    ccp = expt.recv(10);
	
    // TEMPORARY! Do a demand read and subscribe body
    /*
      expt.issueDemandRead(9, 10, "/1", 0, 100, 5);
      expt.issueSubscribeBody(9,
      10,
      AcceptVV.makeVVAllNegatives(),
      dis,
      SUBSCRIPTION_DURATION_MS);
    */
    // End temporary

    assert(ccp.getSignalType() == CoordCommPacket.SYNC_DONE);
    assert(ccp.getData() == null);
	
    System.err.println("done*******");
    finalStats = expt.getAllStats();
    diffStats = finalStats.getDiff(initStats);
    System.err.println("---------------------------------------");
    System.err.println("" + diffStats);
    System.err.println("Exit value for node 9 = " + expt.killNode(9));
    System.err.println("Exit value for node 10 = " + expt.killNode(10));
  }

 /** 
 *  Program starting point 
 **/ 
  public static void main(String[] argv){
    CoordExptA coord = null;

    if (argv.length < 2){
      System.err.println("Usage: <exptConfigfile> <shellType>");
      System.exit(-1);
    }
    coord = new CoordExptA(argv[0], argv[1]);
    CoordExptA.runExperiment(coord);
  }
}

 /** 
 *  Thread that prints out standard error from other processes 
 **/ 
class ErrPrintThread implements Runnable{
  private InputStream is;
  private int nodeId;
  private boolean runFlag;

 /** 
 *  Constructor 
 **/ 
  public ErrPrintThread(int nodeId_, InputStream is_){
    this.is = is_;
    this.nodeId = nodeId_;
    this.setRunFlag(true);
    (new Thread(this)).start();
  }

 /** 
 *  Set the runFlag 
 **/ 
  public synchronized void setRunFlag(boolean value){
    this.runFlag = value;
  }

 /** 
 *  Set the runFlag 
 **/ 
  private synchronized boolean getRunFlag(){
    return(this.runFlag);
  }

 /** 
 *  Main thread method 
 **/ 
  public void run(){
    int b = 0;

    try{
      while(this.getRunFlag()){
        b = this.is.read();
        System.out.write(b);
      }
    }catch(IOException e){
      System.err.println("" + e);
      e.printStackTrace();
    }
  }
}

 /** 
 *  $Log: CoordExptA.java,v $ 
 *  Revision 1.6  2005/10/13 00:24:25  zjiandan 
 *  remove Config.getMy* fixed Garbage Collection and Checkpoint exchange code 
 *  
 *  Revision 1.5  2005/03/04 22:08:26  nayate 
 *  Added a issueSubscribeCheckpoint() method 
 *  
 *  Revision 1.4  2005/03/02 21:44:07  zjiandan 
 *  Modified to work with new code 
 *  
 *  Revision 1.3  2005/01/20 21:32:03  zjiandan 
 *  fixed CLASSPATH. 
 *  
 *  Revision 1.2  2005/01/18 22:49:43  zjiandan 
 *  Rewrited Class Serialization for network deliveringto reduce the overall bandwidth. 
 *  
 *  Revision 1.1  2005/01/13 20:55:41  zjiandan 
 *  Reorganized sosp experiments files into sosp subdirectory under experiments. 
 *  
 *  Revision 1.33  2005/01/10 03:47:48  zjiandan 
 *  Fixed some bugs. Successfully run SanityCheck and Partial Replication experiments. 
 *  
 *  Revision 1.32  2004/11/02 22:24:33  zjiandan 
 *  add utility methods for core recovery self test. 
 *  
 *  Revision 1.31  2004/11/01 18:49:05  nayate 
 *  Changed the path used to access Berkeley DB. 
 *  
 *  Revision 1.30  2004/10/22 20:46:56  dahlin 
 *  Replaced TentativeState with RandomAccessState in DataStore; got rid of 'chain' in BodyMsg; all self-tests pass EXCEPT (1) get compile-time error in rmic and (2) ./runSDIMSControllerTest fails [related to (1)?] 
 *  
 *  Revision 1.29  2004/07/07 21:23:36  nayate 
 *  Minor changes 
 *  
 *  Revision 1.28  2004/05/26 16:19:35  nayate 
 *  *** empty log message *** 
 *  
 *  Revision 1.27  2004/05/26 08:00:58  lgao 
 *  *** empty log message *** 
 *  
 *  Revision 1.26  2004/05/26 02:00:26  nayate 
 *  Added a log sync method 
 *  
 *  Revision 1.25  2004/05/25 08:16:46  lgao 
 *  *** empty log message *** 
 *  
 *  Revision 1.24  2004/05/25 07:37:53  zjiandan 
 *  Minor Change. 
 *  
 *  Revision 1.23  2004/05/25 06:31:40  nayate 
 *  Minor change 
 *  
 *  Revision 1.22  2004/05/24 03:09:07  lgao 
 *  dd syncBodyCheck... 
 *  
 *  Revision 1.21  2004/05/23 23:43:30  zjiandan 
 *  Add sendSyncBody. 
 *  
 *  Revision 1.20  2004/05/22 10:55:13  nayate 
 *  Minor change 
 *  
 *  Revision 1.19  2004/05/21 21:25:27  nayate 
 *  Added code to kill a URANode that refuses to exit 
 *  
 *  Revision 1.18  2004/05/21 21:06:19  nayate 
 *  Changed the code to close a node's input stream before killing it 
 *  
 *  Revision 1.17  2004/05/21 11:04:22  nayate 
 *  Added CVS logging 
 *  
 **/ 
