 /** 
 *  Used to parse the configuration file for experiments 
 **/ 
import gnu.getopt.Getopt;
import gnu.getopt.LongOpt;
import java.util.Hashtable;
import java.util.StringTokenizer;
import java.util.Vector;
import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;

public class ExptConfig
{
    static final int MAX_CONFIG_FIELDS = 11;
    private String exptConfigFileName;
    private String configFileName;
    private NodeId nodeId;
    private int numFiles;
    private long fileSize;
    private int numWrites;
    private int numReads;
    private int exptNum;
    private int dirPct;
    private int filePct;
    private NodeId senderNodeId;
    private NodeId receiverNodeId;

 /** 
 *  Constructor 
 **/ 
    public ExptConfig(String exptConfigFileName_)
    {
	this.exptConfigFileName = exptConfigFileName_;
	this.parse();
    }

 /** 
 *  Parse the file 
 **/ 
    public void parse() 
    {
	/*
	 * CONFIGFILE <Config file name>
	 * NODEID <My nodeID>
	 * NUMFILES <Number of files>
	 * FILESIZE <Size of files>
	 * NUMWRITES <Number of random writes>
	 * NUMREADS <Number of random reads>
	 * EXPTNUM <Which specific experiment (1..8)>
	 * DIRPCT <Percent of directories we are interested in (1..100)>
	 * FILEPCT <Percent of files we are interested in (1..100)>
	 * SENDER <Node ID of the sender>
	 * RECEIVER <Node ID of the receiver>
	 * 
	 */

	int input = 0;
	LongOpt[] lo = null;
	String fileInput[] = null;
	Getopt g = null;
	int optCount = 0;

	lo = new LongOpt[MAX_CONFIG_FIELDS + 1];
	lo[0] = new LongOpt("CONFIGFILE", LongOpt.REQUIRED_ARGUMENT, null, 0);
	lo[1] = new LongOpt("NODEID", LongOpt.REQUIRED_ARGUMENT, null, 1);
	lo[2] = new LongOpt("NUMFILES", LongOpt.REQUIRED_ARGUMENT, null, 2);
	lo[3] = new LongOpt("FILESIZE", LongOpt.REQUIRED_ARGUMENT, null, 3);
	lo[4] = new LongOpt("NUMWRITES", LongOpt.REQUIRED_ARGUMENT, null, 4);
	lo[5] = new LongOpt("NUMREADS", LongOpt.REQUIRED_ARGUMENT, null, 5);
	lo[6] = new LongOpt("EXPTNUM", LongOpt.REQUIRED_ARGUMENT, null, 6);
	lo[7] = new LongOpt("DIRPCT", LongOpt.REQUIRED_ARGUMENT, null, 7);
	lo[8] = new LongOpt("FILEPCT", LongOpt.REQUIRED_ARGUMENT, null, 8);
	lo[9] = new LongOpt("SENDER", LongOpt.REQUIRED_ARGUMENT, null, 9);
	lo[10] = new LongOpt("RECEIVER", LongOpt.REQUIRED_ARGUMENT, null, 10);

	fileInput = readFile();
	g = new Getopt("CoordExptA", fileInput, "", lo, true);
	while((input = g.getopt()) != -1){
	    switch(input){
	    case 0: 
		this.configFileName = g.getOptarg();
		optCount++;
		break;
	    case 1:
		this.nodeId = new NodeId(Integer.parseInt(g.getOptarg()));
		optCount++;
		break;
	    case 2:
		this.numFiles = Integer.parseInt(g.getOptarg());
		optCount++;
		break;
	    case 3:
		this.fileSize = Long.parseLong(g.getOptarg());
		optCount++;
		break;
	    case 4: 
		this.numWrites = Integer.parseInt(g.getOptarg());
		optCount++;
		break;
	    case 5:
		this.numReads = Integer.parseInt(g.getOptarg());
		optCount++;
		break;
	    case 6:
		this.exptNum = Integer.parseInt(g.getOptarg());
		assert((this.exptNum >= 1) && (this.exptNum <= 8));
		optCount++;
		break;
	    case 7:
		this.dirPct = Integer.parseInt(g.getOptarg());
		assert((this.dirPct >= 0) && (this.dirPct <= 100));
		optCount++;
		break;
	    case 8:
		this.filePct = Integer.parseInt(g.getOptarg());
		assert((this.filePct >= 0) && (this.filePct <= 100));
		optCount++;
		break;
	    case 9:
		this.senderNodeId =
		    new NodeId(Integer.parseInt(g.getOptarg()));
		optCount++;
		break;
	    case 10:
		this.receiverNodeId =
		    new NodeId(Integer.parseInt(g.getOptarg()));
		optCount++;
		break;
	    case 11:
		if(optCount!=MAX_CONFIG_FIELDS){
		    System.err.println("Not all the required options " +
				       "are specified in the config file.");
		    System.exit(-1);
		}
		optCount = 0;
		break;
	    case '?':
		System.out.println("The option '" + (char)g.getOptopt() +
				   "' is not valid.");
		System.exit(-1);
		break;
	    default:
		System.out.println("getopt() returned " + input);
		System.exit(-1);
		break;
	    }
	}
    }
    
 /** 
 *  Read the file into an array of strings 
 **/ 
    private String[] readFile()
    {
	Vector config = null;
	String line = null;
	int lineCounter = 0;
	Object[] objA = null;
	String[] strA = null;

	config = new Vector();
	try{
	    BufferedReader in =
		new BufferedReader(new FileReader(this.exptConfigFileName));
	    assert(in != null);
	    while((line = in.readLine()) != null){
		// "-" is added infront of each line to emulate
		// the format of the input arguments.
		StringTokenizer st = new StringTokenizer("-" + line);
		while(st.hasMoreTokens()){
		    config.add(st.nextToken());
		}
	    }
	}catch(IOException e){
	    e.printStackTrace();
	    assert(false);
	}
	objA = config.toArray();
	strA = new String[objA.length];
	for(int i = 0; i < objA.length; i++){
	    strA[i] = (String)objA[i];
	}
	return strA;
    }

 /** 
 *  Return the config file name 
 **/ 
    public String getConfigFileName()
    {
	return(this.configFileName);
    }

 /** 
 *  Return the Node Id 
 **/ 
    public NodeId getNodeId()
    {
	return(this.nodeId);
    }

 /** 
 *  Return the number of files to generate 
 **/ 
    public int getNumFiles()
    {
	return(this.numFiles);
    }

 /** 
 *  Return the size of our files 
 **/ 
    public long getFileSize()
    {
	return(this.fileSize);
    }

 /** 
 *  Return the number of writes to perform 
 **/ 
    public int getNumWrites()
    {
	return(this.numWrites);
    }

 /** 
 *  Return the number of reads to perform 
 **/ 
    public int getNumReads()
    {
	return(this.numReads);
    }

 /** 
 *  Return the experiment number (1..8) 
 **/ 
    public int getExptNum()
    {
	return(this.exptNum);
    }

 /** 
 *  Return the percentage of directories in our interest set 
 **/ 
    public int getDirPct()
    {
	return(this.dirPct);
    }

 /** 
 *  Return the percentage of files in each directory in our interest set 
 **/ 
    public int getFilePct()
    {
	return(this.filePct);
    }

 /** 
 *  Return the sender node ID 
 **/ 
    public NodeId getSenderNodeId()
    {
	return(this.senderNodeId);
    }

 /** 
 *  Return the percentage of files in each directory in our interest set 
 **/ 
    public NodeId getReceiverNodeId()
    {
	return(this.receiverNodeId);
    }

 /** 
 *  Used for testing 
 **/ 
    public static void main(String[] argv){
	ExptConfig exptConfig = null;

	if(argv.length < 1){
	    System.out.println("Usage: <configuration file>");
	    System.exit(-1);
	}
	exptConfig = new ExptConfig(argv[0]);

	System.out.println("ConfigFileName: " +
			   exptConfig.getConfigFileName());
	System.out.println("Node ID: " + exptConfig.getNodeId());
	System.out.println("Num files: " + exptConfig.getNumFiles());
	System.out.println("File size: " + exptConfig.getFileSize());
	System.out.println("Num writes: " + exptConfig.getNumWrites());
	System.out.println("Num reads: " + exptConfig.getNumReads());
	System.out.println("Expt num: " + exptConfig.getExptNum());
	System.out.println("Dir pct: " + exptConfig.getDirPct());
	System.out.println("File pct: " + exptConfig.getFilePct());
	System.out.println("Sender: " + exptConfig.getSenderNodeId());
	System.out.println("Receiver: " + exptConfig.getReceiverNodeId());
    }
}

 /** 
 *  $Log: ExptConfig.java,v $ 
 *  Revision 1.1  2005/01/13 20:55:41  zjiandan 
 *  Reorganized sosp experiments files into sosp subdirectory under experiments. 
 *  
 *  Revision 1.1  2004/05/20 02:14:17  nayate 
 *  Added files to allow compilation 
 *  
 **/ 
