 /** 
 *  Generate bytes to write to a file for our experiment 
 **/ 
import java.io.PrintStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.StreamTokenizer;
import java.io.InputStreamReader;
import java.io.BufferedReader;
import java.io.IOException;
import java.util.Arrays;

public class ExptFileBytes implements Immutable, Cloneable{

 /** 
 *  Local constants 
 **/ 
  private static final int MAGIC_NUM = 0x455ABC;

 /** 
 *  Constructor 
 **/ 
  public
  ExptFileBytes(int newEventNum, int newNumBytes)
    throws InvalidExptFileBytesException{
    try{
      this.eventNum = newEventNum;
      this.numBytes = newNumBytes;
      this.dataBytes = ExptFileBytes.genDataBytes(newEventNum);
      if(this.dataBytes.length > newNumBytes){
        throw(new InvalidExptFileBytesException("File size too small"));
      }
    }catch(IOException e){
      e.printStackTrace();
      System.out.println("" + e);
      assert(false);
    }
  }
  
 /** 
 *  Constructor 
 **/ 
  public
  ExptFileBytes(ImmutableBytes ib) throws InvalidExptFileBytesException{
    byte[] fileBytes = null;
    ByteArrayInputStream bais = null;
    StreamTokenizer st = null;
    int magicNumRead = 0;
    int eventNumRead = 0;
    boolean badMagicNum = false;
    int token = 0;
    BufferedReader reader = null;

    try{
      fileBytes = ib.dangerousGetReferenceToInternalByteArray();
      bais = new ByteArrayInputStream(fileBytes);
      reader = new BufferedReader(new InputStreamReader(bais));
      st = new StreamTokenizer(reader);
      token = st.nextToken();
      if(token != StreamTokenizer.TT_NUMBER){
        badMagicNum = true;
      }else{
        magicNumRead = (int)st.nval;
        if(magicNumRead != ExptFileBytes.MAGIC_NUM){
          badMagicNum = true;
        }
      }
      if(badMagicNum){
        throw(new InvalidExptFileBytesException("Invalid magic number!"));
      }

      token = st.nextToken();
      if(token != StreamTokenizer.TT_NUMBER){
        throw(new InvalidExptFileBytesException("Invalid event number!"));
      }else{
        eventNumRead = (int)st.nval;
      }
      this.eventNum = eventNumRead;
      this.numBytes = fileBytes.length;

      // Recreate dataBytes just in case someone calls makeImmutableBytes
      this.dataBytes = ExptFileBytes.genDataBytes(eventNumRead);
      assert(this.dataBytes.length <= fileBytes.length);
    }catch(IOException e){
      e.printStackTrace();
      System.err.println("" + e);
      assert(false);
    }
  }

 /** 
 *  Return an ImmutableBytes object with some data written 
 **/ 
  public ImmutableBytes
  makeImmutableBytes(){
    // Write in the event number
    byte[] fileBytes = null;
    ImmutableBytes ib = null;

    fileBytes = new byte[this.numBytes];
    assert(this.numBytes >= this.dataBytes.length);
    System.arraycopy(this.dataBytes, 0, fileBytes, 0, this.dataBytes.length);
    ib = ImmutableBytes.dangerousCreateImmutableBytes(fileBytes);
    fileBytes = null;  // To remove any accidental reference to the array
    return(ib);
  }

 /** 
 *  Return the event number 
 **/ 
  public int
  getEventNum(){
    return(this.eventNum);
  }

 /** 
 *  Return the file size 
 **/ 
  public int
  getNumBytes(){
    return(this.numBytes);
  }

 /** 
 *  Return true if "o" equals "this" 
 **/ 
  public boolean
  equals(Object o){
    ExptFileBytes efb = null;
    boolean result = false;

    // 1 constant, 2 java-provided fields, 3 declared fields
    assert(this.getClass().getDeclaredFields().length == 6);
    if(o instanceof ExptFileBytes){
      efb = (ExptFileBytes)o;
      result = ((this.eventNum == efb.eventNum) &&
                (this.numBytes == efb.numBytes) &&
                (Arrays.equals(this.dataBytes, efb.dataBytes)));
    }else{
      result = false;
    }
    return(result);
  }

 /** 
 *  Clone this object 
 **/ 
  public Object
  clone(){
    // Because this object is immutable, we do not need to do anything
    return(this);
  }

 /** 
 *  Return a hash code (note: does nothing) 
 **/ 
  public int
  hashCode(){
    assert(false);
    return(0);
  }

 /** 
 *  Used for testing 
 **/ 
  public static void
  main(String[] argv){
    Env.verifyAssertEnabled();
    System.out.println("Testing ExptFileBytes...");
    ExptFileBytes.test1();
    ExptFileBytes.test2();
    System.out.println("...Finished");
  }

 /** 
 *  Do some simple testing 
 **/ 
  private static void
  test1(){
    ExptFileBytes efb1 = null;
    ExptFileBytes efb2 = null;
    ImmutableBytes ib1 = null;
    ImmutableBytes ib2 = null;
    byte[] b = null;

    try{
      efb1 = new ExptFileBytes(10, 100);
      ib1 = efb1.makeImmutableBytes();
      b = ib1.dangerousGetReferenceToInternalByteArray();
      ib2 = ImmutableBytes.dangerousCreateImmutableBytes(b);
      efb2 = new ExptFileBytes(ib2);
      assert(efb2.eventNum == 10);
      assert(efb2.numBytes == 100);
    }catch(InvalidExptFileBytesException e){
      System.out.println("" + e);
      assert(false);
    }
  }

 /** 
 *  Do some simple testing (part 2) 
 **/ 
  private static void
  test2(){
    ExptFileBytes efb1 = null;
    ExptFileBytes efb2 = null;
    ImmutableBytes ib1 = null;
    ImmutableBytes ib2 = null;
    byte[] b = null;

    try{
      efb1 = new ExptFileBytes(15, 1000);
      assert(efb1.getEventNum() == 15);
      assert(efb1.getNumBytes() == 1000);
      ib1 = efb1.makeImmutableBytes();
      b = ib1.dangerousGetReferenceToInternalByteArray();
      assert(efb1.equals(efb1));

      ib2 = ImmutableBytes.dangerousCreateImmutableBytes(b);
      efb2 = new ExptFileBytes(ib2);
      assert(efb1.equals(efb2));
      assert(efb2.equals(efb1));
      assert(efb2.getEventNum() == 15);
      assert(efb2.getNumBytes() == 1000);

      /*
        // Tested printing to make sure we get something readable
        try{
        System.out.write(b);
        }catch(IOException e){
        System.out.println("" + e);
        assert(false);
        }
      */
    }catch(InvalidExptFileBytesException e){
      System.out.println("" + e);
      assert(false);
    }
  }

 /** 
 *  Generate the data array 
 **/ 
  private static byte[]
  genDataBytes(int eventNum) throws IOException{
    ByteArrayOutputStream baos = null;
    PrintStream ps = null;
    byte[] dataBytes = null;

    baos = new ByteArrayOutputStream();
    ps = new PrintStream(baos);
    ps.println("" + ExptFileBytes.MAGIC_NUM);
    ps.println("" + eventNum);
    ps.flush();
    baos.flush();
    dataBytes = baos.toByteArray();
    return(dataBytes);
  }

 /** 
 *  Data members 
 **/ 
  private int eventNum;
  private int numBytes;
  private byte[] dataBytes;
}

 /** 
 *  Exception thrown if we read invalid bytes 
 **/ 
class InvalidExptFileBytesException extends Exception{

 /** 
 *  Constructor 
 **/ 
  public
  InvalidExptFileBytesException(String newMsg){
    this.msg = newMsg;
  }

 /** 
 *  Convert to a string representation 
 **/ 
  public String
  toString(){
    return(this.msg);
  }

 /** 
 *  Data members 
 **/ 
  private String msg;
}

