 /** 
 *  Code to analyze the output log generated by the Nice experiments 
 **/ 
import java.util.LinkedList;
import java.util.ListIterator;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.BufferedReader;
import java.io.Reader;
import java.io.EOFException;
import java.io.IOException;

// Used for testing
import java.io.ByteArrayOutputStream;
import java.io.ByteArrayInputStream;

public class LogAnalyzer{

 /** 
 *  Local constants 
 **/ 
  private static boolean PRINT_STALE_EVENTS = false;

 /** 
 *  Experiment starting point 
 **/ 
  public static void
  main(String[] argv){
    if(argv.length < 1){
      LogAnalyzer.analyze(Constants.NUM_FILES, System.in);
      LogAnalyzer.printStats();
    }else{
      Env.verifyAssertEnabled();
      System.out.println("Testing LogAnalyzer...");
      LogAnalyzer.test1();
      LogAnalyzer.test2();
      System.out.println("...Finished");
    }
  }

 /** 
 *  Analyze the log 
 **/ 
  private static void
  analyze(int numFiles, InputStream is){
    LinkedList[] writeTimes = null;
    boolean done = false;
    BufferedReader reader = null;
    OutputLogEntry ole = null;
    ObjId objId = null;
    int fileNum = 0;
    int dataEventNum = 0;
    long stalenessMS = 0;
    long currentTimeMS = 0;
    int numFresh = 0;

    writeTimes = new LinkedList[numFiles];
    for(int i = 0; i < numFiles; i++){
      writeTimes[i] = new LinkedList();
    }
    reader = new BufferedReader(new InputStreamReader(is));
    while(!done){
      try{
        ole = new OutputLogEntry(reader);
        fileNum = LogAnalyzer.getFileNum(ole.getObjId());
        if(ole.getOperation() == OutputLogEntry.WRITE){
          writeTimes[fileNum].addFirst(ole);
        }else{
          dataEventNum = ole.getDataEventNum();
          currentTimeMS = ole.getTimeMS() + ole.getDurationMS();
          numReads++;

          // Update response time statistics
          LogAnalyzer.totalDurationMS += ole.getDurationMS();

          // Update staleness statistics
          if(LogAnalyzer.isStale(writeTimes, fileNum, dataEventNum)){
            Env.dprintln(PRINT_STALE_EVENTS, "" + ole.getEventNum());
            stalenessMS = LogAnalyzer.getStalenessMS(writeTimes,
                                                     fileNum,
                                                     currentTimeMS,
                                                     dataEventNum);
            LogAnalyzer.totalStalenessMS += stalenessMS;
            LogAnalyzer.numStale++;
          }else{
            // For sanity checking
            numFresh++;
          }
        }
      }catch(EOFException e){
        done = true;
      }catch(InvalidOutputLogEntryException e){
        System.out.println("Got an invalid entry!");
      }catch(IOException e){
        e.printStackTrace();
        System.out.println("" + e);
        assert(false);
      }
    }
    assert(LogAnalyzer.numReads == (LogAnalyzer.numStale + numFresh));
  }

 /** 
 *  Convert the given file name into a number 
 **/ 
  private static int
  getFileNum(ObjId objId){
    String objPath = null;
    int fileNum = 0;

    objPath = objId.getPath();
    assert(objPath.startsWith("/"));
    try{
      fileNum = Integer.parseInt(objPath.substring(1));
    }catch(NumberFormatException e){
      System.out.println("" + e);
      assert(false);
    }
    return(fileNum);
  }

 /** 
 *  Return true if the read to fileNum was stale 
 **/ 
  private static boolean
  isStale(LinkedList[] writeTimes, int fileNum, int dataEventNum){
    boolean result = false;
    OutputLogEntry ole = null;

    if(!writeTimes[fileNum].isEmpty()){
      ole = (OutputLogEntry)writeTimes[fileNum].getFirst();
      assert(ole.getEventNum() >= dataEventNum);
      if(ole.getEventNum() > dataEventNum){
        result = true;
      }else{
        // Since the receiver has now seen the most recent write,
        // we can delete information every other write
        while(writeTimes[fileNum].size() > 1){
          writeTimes[fileNum].removeLast();
        }
        result = false;
      }
    }
    return(result);
  }

 /** 
 *  Return the amount of time by which this read is stale. 
 **/ 
  private static long
  getStalenessMS(LinkedList[] writeTimes,
                 int fileNum,
                 long currentTimeMS,
                 int dataEventNum){
    long stalenessMS = -1;
    ListIterator li = null;
    long oldestUnseenWriteMS = 0;
    int firstDeleteIndex = -1;
    OutputLogEntry ole = null;

    assert(!writeTimes[fileNum].isEmpty());
    li = writeTimes[fileNum].listIterator(0);
    while((firstDeleteIndex < 0) && (li.hasNext())){
      ole = (OutputLogEntry)li.next();
      assert(ole.getOperation() == OutputLogEntry.WRITE);
      if(ole.getEventNum() > dataEventNum){
        Env.dprintln(PRINT_STALE_EVENTS,
                     "Older write event: " + ole.getEventNum());
        oldestUnseenWriteMS = ole.getTimeMS();
      }else{
        assert(ole.getEventNum() == dataEventNum);
        assert(oldestUnseenWriteMS > 0);
        assert(currentTimeMS >= oldestUnseenWriteMS);
        stalenessMS = currentTimeMS - oldestUnseenWriteMS;
        firstDeleteIndex = li.nextIndex();
      }
    }
    if(!li.hasNext()){
      // There has only been one write to this object but we
      // haven't seen it.
      stalenessMS = currentTimeMS - oldestUnseenWriteMS;
    }
    assert(stalenessMS >= 0);

    // if firstDeleteIndex is not -1, it points to the first entry
    // *after* the one for the most recent write we have seen. We
    // can garbage collect this and all subsequent entries.
    if(firstDeleteIndex >= 0){
      while(writeTimes[fileNum].size() > firstDeleteIndex){
        writeTimes[fileNum].removeLast();
      }
    }
    return(stalenessMS);
  }

 /** 
 *  Print relevant statistics 
 **/ 
  private static void
  printStats(){
    System.out.println("numReads = " + LogAnalyzer.numReads);
    System.out.println("numStale = " + LogAnalyzer.numStale);
    System.out.println("totalStalenessMS = " + LogAnalyzer.totalStalenessMS);
    System.out.println("totalDurationMS = " + LogAnalyzer.totalDurationMS);
  }

 /** 
 *  Test getStalenessMS() and isStale() 
 **/ 
  private static void
  test1(){
    LinkedList[] writeTimes = null;
    OutputLogEntry ole1 = null;
    OutputLogEntry ole2 = null;
    OutputLogEntry ole3 = null;
    ObjId oid0 = null;
    long stalenessMS1 = 0;
    long stalenessMS2 = 0;

    // Test 1
    writeTimes = new LinkedList[1];
    writeTimes[0] = new LinkedList();
    assert(!LogAnalyzer.isStale(writeTimes, 0, -1));

    // Test 2
    writeTimes = new LinkedList[3];
    writeTimes[0] = new LinkedList();
    writeTimes[1] = new LinkedList();
    writeTimes[2] = new LinkedList();
    oid0 = new ObjId("/0");
    ole1 = new OutputLogEntry(1, 1, OutputLogEntry.WRITE, 2, oid0, -1);
    writeTimes[0].addFirst(ole1);
    assert(LogAnalyzer.isStale(writeTimes, 0, -1));
    assert(writeTimes[0].size() == 1); // Nothing should have gotten deleted
    assert(!LogAnalyzer.isStale(writeTimes, 0, 1));
    assert(writeTimes[0].size() == 1); // Nothing should have gotten deleted

    // Test 3
    writeTimes = new LinkedList[3];
    writeTimes[0] = new LinkedList();
    writeTimes[1] = new LinkedList();
    writeTimes[2] = new LinkedList();
    oid0 = new ObjId("/0");
    ole1 = new OutputLogEntry(1, 1, OutputLogEntry.WRITE, 2, oid0, -1);
    writeTimes[0].addFirst(ole1);
    ole2 = new OutputLogEntry(2, 55, OutputLogEntry.WRITE, 3, oid0, -1);
    writeTimes[0].addFirst(ole2);
    ole3 = new OutputLogEntry(3, 70, OutputLogEntry.WRITE, 5, oid0, -1);
    writeTimes[0].addFirst(ole3);
    assert(LogAnalyzer.isStale(writeTimes, 0, 1));
    stalenessMS1 = LogAnalyzer.getStalenessMS(writeTimes, 0, 76, 1);
    assert(stalenessMS1 == 21);
    assert(writeTimes[0].size() == 3); // No entries should be deleted
    assert(LogAnalyzer.isStale(writeTimes, 0, 2));
    stalenessMS2 = LogAnalyzer.getStalenessMS(writeTimes, 0, 88, 2);
    assert(stalenessMS2 == 18);
    assert(writeTimes[0].size() == 2); // 1 entry should get deleted
    assert(!LogAnalyzer.isStale(writeTimes, 0, 3));
  }

 /** 
 *  Do some more comprehensive testing 
 **/ 
  private static void
  test2(){
    ByteArrayOutputStream baos = null;
    ByteArrayInputStream bais = null;
    OutputLogEntry ole = null;
    ObjId oid0 = null;
    ObjId oid1 = null;
    int localNumStale = 0;
    int localNumReads = 0;
    long localStalenessMS = 0;
    long localDurationMS = 0;
    byte[] logBytes = null;

    try{
      baos = new ByteArrayOutputStream();
      oid0 = new ObjId("/0");
      oid1 = new ObjId("/1");

      // "/0" modified at time 10, "/1" modified at time 100
      ole = new OutputLogEntry(1, 10, OutputLogEntry.WRITE, -1, oid0, -1);
      ole.writeToOS(baos);
      ole = new OutputLogEntry(2, 100, OutputLogEntry.WRITE, -1, oid1, -1);
      ole.writeToOS(baos);

      // Read for "/0" got the correct value at time 110
      ole = new OutputLogEntry(3, 110, OutputLogEntry.READ, 1, oid0, 1);
      ole.writeToOS(baos);
      localNumReads++;
      localDurationMS += 1;

      // "/0" modified at time 200
      ole = new OutputLogEntry(4, 200, OutputLogEntry.WRITE, -1, oid0, -1);
      ole.writeToOS(baos);

      // Read for "/0" got the wrong value at time 210
      ole = new OutputLogEntry(5, 210, OutputLogEntry.READ, 2, oid0, 1);
      ole.writeToOS(baos);
      localNumReads++;
      localNumStale++;
      localDurationMS += 2;
      localStalenessMS += (210 - 200);

      // "/1" modified at time 300, "/0" modified at time 310
      ole = new OutputLogEntry(6, 300, OutputLogEntry.WRITE, -1, oid1, -1);
      ole.writeToOS(baos);
      ole = new OutputLogEntry(7, 310, OutputLogEntry.WRITE, -1, oid0, -1);
      ole.writeToOS(baos);

      // Read for "/1" got the wrong value at time 400
      ole = new OutputLogEntry(8, 400, OutputLogEntry.READ, 8, oid1, 2);
      ole.writeToOS(baos);
      localNumReads++;
      localNumStale++;
      localDurationMS += 8;
      localStalenessMS += (400 - 300);

      // Read for "/0" got the wrong value at time 410
      ole = new OutputLogEntry(9, 410, OutputLogEntry.READ, 3, oid0, 1);
      ole.writeToOS(baos);
      localNumReads++;
      localNumStale++;
      localDurationMS += 3;
      localStalenessMS += (410 - 200);

      // Read for "/1" got the right value at time 500
      ole = new OutputLogEntry(10, 500, OutputLogEntry.READ, 2, oid1, 6);
      ole.writeToOS(baos);
      localNumReads++;
      localDurationMS += 2;

      // Read for "/0" got the wrong value at time 510
      ole = new OutputLogEntry(11, 510, OutputLogEntry.READ, 2, oid0, 4);
      ole.writeToOS(baos);
      localNumReads++;
      localNumStale++;
      localDurationMS += 2;
      localStalenessMS += (510 - 310);

      // Read for "/1" got the right value at time 600
      ole = new OutputLogEntry(12, 600, OutputLogEntry.READ, 4, oid1, 6);
      ole.writeToOS(baos);
      localNumReads++;
      localDurationMS += 4;

      // Read for "/0" got the right value at time 610
      ole = new OutputLogEntry(13, 610, OutputLogEntry.READ, 5, oid0, 7);
      ole.writeToOS(baos);
      localNumReads++;
      localDurationMS += 5;

      logBytes = baos.toByteArray();
      // System.out.write(logBytes); // Already tested this
      bais = new ByteArrayInputStream(logBytes);
      LogAnalyzer.analyze(10, bais);

      assert(localNumReads == LogAnalyzer.numReads);
      assert(localNumStale == LogAnalyzer.numStale);
      assert(localStalenessMS == LogAnalyzer.totalStalenessMS);
      assert(localDurationMS == LogAnalyzer.totalDurationMS);
    }catch(IOException e){
      e.printStackTrace();
      System.err.println("" + e);
      assert(false);
    }
  }

 /** 
 *  Data members 
 **/ 
  private static int numStale = 0;
  private static int numReads = 0;
  private static long totalStalenessMS = 0;
  private static long totalDurationMS = 0;
}
