 /** 
 *  Emulate the receiver for the Nice experiments 
 **/ 
import java.io.IOException;
import java.io.EOFException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.InputStreamReader;
import java.io.BufferedReader;
import java.util.LinkedList;
import java.util.List;
import java.util.Collections;

public class NiceExptReceiver{

 /** 
 *  Local constants 
 **/ 
  private static final int WAIT_SIGNAL_PORT1 = 4790;
  private static final int WAIT_SIGNAL_PORT2 = 4791;
  private static final int NUM_ENTRIES_LOGGER_BUFFERS = 10000;
  private static final int NUM_READER_THREADS = 10;

 /** 
 *  Program staring point 
 **/ 
  public static void
  main(String[] argv){
    int nodeIdInt = 0;
    int algorithm = 0;
    NodeId senderNodeId = null;
    NodeId receiverNodeId = null;
    NodeId myNodeId = null;
    NiceExptReceiver niceExptReceiver = null;

    if(argv.length == 4){
      senderNodeId = new NodeId(Integer.parseInt(argv[1]));
      myNodeId = new NodeId(Integer.parseInt(argv[2]));
      algorithm = Integer.parseInt(argv[3]);
      niceExptReceiver = new NiceExptReceiver(argv[0],
                                              senderNodeId,
                                              myNodeId,
                                              algorithm);
      try{
        System.err.println("Algorithm = ");
        Constants.printAlgorithm(algorithm);

        // Initialize the sender
        niceExptReceiver.setupInitialSubscription(senderNodeId, myNodeId);
        niceExptReceiver.syncInitialWrites(senderNodeId);
        System.err.println("Waiting to be woken up on port: " +
                           NiceExptReceiver.WAIT_SIGNAL_PORT1 + 
                           " for the first time");
        Synchronizer.waitSignal(NiceExptReceiver.WAIT_SIGNAL_PORT1);
        System.err.println("Woken up, setting up body subscriptions");
        if(algorithm == Constants.PREFETCH){
          niceExptReceiver.setupBodySubscription(senderNodeId, myNodeId);
        }
        System.err.println("Waiting to be woken up on port: " +
                           NiceExptReceiver.WAIT_SIGNAL_PORT2 + 
                           " for the second time");
        Synchronizer.waitSignal(NiceExptReceiver.WAIT_SIGNAL_PORT2);
        System.err.println("Woken up the second time, starting experiment");

        niceExptReceiver.playTraceEntries(System.in, System.out);
        System.err.println("Experiment finished");
      }catch(IOException e){
        System.err.println("" + e);
      }catch(RMINetworkException e){
        e.printStackTrace();
        System.err.println("" + e);
      }catch(RMIApplicationException e){
        e.printStackTrace();
        System.err.println("" + e);
      }
    }else{
      System.err.println("Usage: java NiceExptReceiver" +
                         "<configFileName> <sender node id> <my node id> " +
                         "<algorithm>");
    }
  }

 /** 
 *  Constructor 
 **/ 
  private
  NiceExptReceiver(String configFile,
                   NodeId senderNodeId,
                   NodeId myNodeId,
                   int newAlgorithm){
    Core core = null;

    this.rmiClient = new RMIClient();
    Config.readConfig(configFile, myNodeId);
    core = new Core(rmiClient, true, true);
    this.exptController = new NiceExptController(this.rmiClient,
                                                 senderNodeId,
                                                 myNodeId,
                                                 Config.getMyDNS(),
                                                 Config.getMyPortBody());
    this.uraNode = new URANode(core, this.exptController, this.rmiClient);
    this.startTimeMS = 0;
    this.exptTimeMS = 0;
    this.algorithm = newAlgorithm;
  }

 /** 
 *  Subscribe to listen to intialization invalidates from the sender 
 **/ 
  private void
  setupInitialSubscription(NodeId senderNodeId, NodeId myNodeId)
    throws IOException, RMINetworkException, RMIApplicationException{
    SubscriptionSet ss = null;
    CounterVV startVV = null;
    CounterVV endVV = null;

    ss = SubscriptionSet.makeSubscriptionSet(":/*");
    startVV = new CounterVV(CounterVV.makeVVAllNegatives());
    endVV = new CounterVV(CounterVV.makeVVAllNegatives());
    // Note that the stream will not send the invalidate whose timestamp
    // matches endVV - it will only send invalidates with strictly lower
    // timestamps.
    endVV.setStampByServer(senderNodeId,
                           Constants.NUM_OPERATIONS + Constants.NUM_FILES);
    this.rmiClient.subscribe(ss,
                             startVV,
                             endVV,
                             senderNodeId,
                             myNodeId,
                             Config.getDNS(myNodeId),
                             Config.getPortInval(myNodeId),
                             86400000, // 1 day
                             0,
                             0);
  }

  //-------------------------------------------------------------------------
  // Subscribe to listen to updates from the sender
  //-------------------------------------------------------------------------
  private void
  setupBodySubscription(NodeId senderNodeId, NodeId myNodeId)
    throws IOException, RMINetworkException, RMIApplicationException{
    SubscriptionSet ss = null;
    CounterVV startVV = null;

    ss = SubscriptionSet.makeSubscriptionSet(":/*");
    startVV = new CounterVV(CounterVV.makeVVAllNegatives());
    this.rmiClient.subscribeBody(ss,
                                 startVV,
                                 senderNodeId,
                                 myNodeId,
                                 Config.getDNS(myNodeId),
                                 Config.getPortNice(myNodeId),
                                 86400000); // 1 day
  }

  //-------------------------------------------------------------------------
  // Wait for the initial set of writes from the sender
  //-------------------------------------------------------------------------
  private void
  syncInitialWrites(NodeId senderNodeId){
    AcceptStamp as = null;
    LocalInterface li = null;

    as = new AcceptStamp(Constants.NUM_FILES - 1, senderNodeId);
    li = this.uraNode.getLocalInterface();
    try{
      li.sync(as);
    }catch(InterruptedException e){
      e.printStackTrace();
      System.out.println("" + e);
      assert(false);
    }
  }

  //-------------------------------------------------------------------------
  // Play a trace and perform actions accordingly
  //-------------------------------------------------------------------------
  private void
  playTraceEntries(InputStream is, OutputStream os) throws IOException{
    boolean done = false;
    TraceEntry te = null;
    OutputLogEntry ole = null;
    BufferedReader br = null;
    List finishedReads = null;
    OutputLogger logger = null;
    ProdConsQueue pendingReads = null;
    Thread threads[] = null;

    logger = new OutputLogger(os, NUM_ENTRIES_LOGGER_BUFFERS);
    br = new BufferedReader(new InputStreamReader(is));
    this.startTimeMS = System.currentTimeMillis();
    System.err.println("startTimeMS = " + this.startTimeMS);
    this.exptTimeMS = 0;
    finishedReads = Collections.synchronizedList(new LinkedList());
    pendingReads = new ProdConsQueue();

    threads = new Thread[NUM_READER_THREADS];
    for(int i = 0; i < NUM_READER_THREADS; i++){
	// Note: Thread will automatically start itself
	NiceExptReaderThread nert = null;
	nert = new NiceExptReaderThread(pendingReads,
					finishedReads,
					this.startTimeMS,
					this.uraNode.getLocalInterface());
	threads[i] = new Thread(nert);
	threads[i].start();
    }

    while(!done){
      try{
        te = new TraceEntry(br);
        if(te.getOperation() == TraceEntry.WRITE){

          this.exptTimeMS = System.currentTimeMillis() - this.startTimeMS;
          this.sleepUntilExptTimeMS(te.getTimeMS());
          this.exptTimeMS = System.currentTimeMillis() - this.startTimeMS;

          this.writeFinishedReadsToOS(finishedReads, logger, te.getTimeMS());
          ole = new OutputLogEntry(te.getEventNum(),       // event num
                                   te.getTimeMS(),         // event time(ms)
                                   OutputLogEntry.WRITE,   // operation
                                   -1,                     // duration(ms)
                                   te.getObjId(),          // object id
                                   -1);                    // data event num
          logger.addEntry(ole);
        }else{
          this.waitUntilReadTime(te);
          pendingReads.enqueue(te); // The next free thread will pick this up
        }
      }catch(EOFException e){
        done = true;
        pendingReads.setDone();
        pendingReads.waitQueueEmpty();
        while(!finishedReads.isEmpty()){
          ole = (OutputLogEntry)finishedReads.remove(0);
          logger.addEntry(ole);
        }
        logger.flush();
	for(int i = 0; i < NUM_READER_THREADS; i++){
	    threads[i].interrupt();
	}
	System.err.println("lastReadStartTimeMS = " +
			   NiceExptReaderThread.getLastReadStartTimeMS());
	System.err.println("lastReadFinishTimeMS = " +
			   NiceExptReaderThread.getLastReadFinishTimeMS());
	System.err.println("numReadsStarted = " +
			   NiceExptReaderThread.getNumReadsStarted());
	System.err.println("numReadsFinished = " +
			   NiceExptReaderThread.getNumReadsFinished());
      }catch(InvalidTraceEntryException e){
        e.printStackTrace();
        System.out.println("" + e);
        assert(false);
      }
    }
  }

  //-------------------------------------------------------------------------
  // Wait until a trace entry time
  //-------------------------------------------------------------------------
  private void
  waitUntilReadTime(TraceEntry te) throws IOException{
    OutputLogEntry ole = null;

    assert(te.getOperation() == TraceEntry.READ);

    // Wait until the specified time in the log
    this.exptTimeMS = System.currentTimeMillis() - this.startTimeMS;
    this.sleepUntilExptTimeMS(te.getTimeMS());
    this.exptTimeMS = System.currentTimeMillis() - this.startTimeMS;
  }

  //-------------------------------------------------------------------------
  // Sleep until the specified time
  //-------------------------------------------------------------------------
  private void
  sleepUntilExptTimeMS(long eventTimeMS){
    while(this.exptTimeMS < eventTimeMS){
      try{
        Thread.sleep(eventTimeMS - this.exptTimeMS);
      }catch(InterruptedException e){
        // This shouldn't happen; no thread calls interrupt()
        e.printStackTrace();
        System.err.println("" + e);
        assert(false);
      }
      this.exptTimeMS = System.currentTimeMillis() - this.startTimeMS;
    }
  }

  //-------------------------------------------------------------------------
  // Write to "os" the list of those write operations that finished
  // while a given read operation was happening
  //-------------------------------------------------------------------------
  private void
  writeFinishedReadsToOS(List finishedReads,
                         OutputLogger logger,
                         long writeTimeMS)
    throws IOException{
    OutputLogEntry ole = null;

    if(!finishedReads.isEmpty()){
      ole = (OutputLogEntry)finishedReads.get(0);
      assert(ole.getOperation() == OutputLogEntry.READ);
    }
    while((!finishedReads.isEmpty()) &&
          ((ole.getTimeMS() + ole.getDurationMS()) < writeTimeMS)){
      // This read already happened
      finishedReads.remove(0);
      logger.addEntry(ole);
      if(!finishedReads.isEmpty()){
        ole = (OutputLogEntry)finishedReads.get(0);
        assert(ole.getOperation() == OutputLogEntry.READ);
      }
    }
  }

  //-------------------------------------------------------------------------
  // Data members
  //-------------------------------------------------------------------------
  private NiceExptController exptController;
  private RMIClient rmiClient;
  private URANode uraNode;
  private long startTimeMS;
  private long exptTimeMS;
  private int algorithm;
}

//---------------------------------------------------------------------------
// A producer/consumer queue
//---------------------------------------------------------------------------
class ProdConsQueue{

  //-------------------------------------------------------------------------
  // Constructor
  //-------------------------------------------------------------------------
  public
  ProdConsQueue(){
    this.queue = new LinkedList();
    this.done = false;
  }

  //-------------------------------------------------------------------------
  // Enqueue a request
  //-------------------------------------------------------------------------
  public synchronized void
  enqueue(Object obj){
    this.queue.addLast(obj);
    this.notifyAll();
  }

  //-------------------------------------------------------------------------
  // Wait for the queue to become empty (to be called by producer)
  //-------------------------------------------------------------------------
  public synchronized void
  waitQueueEmpty(){
    // Should only be called after the done flag is set
    assert(this.done);
    while(!this.queue.isEmpty()){
      try{
        this.wait();
      }catch(InterruptedException e){
        e.printStackTrace();
        System.err.println("" + e);
        assert(false);
      }
    }
  }

  //-------------------------------------------------------------------------
  // Dequeue a request (or return null if we are done)
  //-------------------------------------------------------------------------
  public synchronized Object
  dequeue(){
    Object obj = null;

    while(this.queue.isEmpty() && (!this.done)){
      try{
        this.wait();
      }catch(InterruptedException e){
        e.printStackTrace();
        System.err.println("" + e);
        assert(false);
      }
    }
    if(!this.queue.isEmpty()){
      obj = this.queue.removeFirst();
    }else{
      assert(this.done);
      obj = null;
      this.notifyAll();
    }
    return(obj);
  }

  //-------------------------------------------------------------------------
  // Indicate that we are done adding to the queue
  //-------------------------------------------------------------------------
  public synchronized void
  setDone(){
    this.done = true;
    this.notifyAll();
  }
  
  //-------------------------------------------------------------------------
  // Data members
  //-------------------------------------------------------------------------
  private LinkedList queue;
  private boolean done;
}

//---------------------------------------------------------------------------
// A single thread that reads data
//---------------------------------------------------------------------------
class NiceExptReaderThread implements Runnable{

  //-------------------------------------------------------------------------
  // Constructor
  //-------------------------------------------------------------------------
  public
  NiceExptReaderThread(ProdConsQueue newPendingReads,
                       List newFinishedReads,
                       long newStartTimeMS,
                       LocalInterface newLocalInterface){
    this.pendingReads = newPendingReads;
    this.finishedReads = newFinishedReads;
    this.startTimeMS = newStartTimeMS;
    this.localInterface = newLocalInterface;
    this.threadNum = NiceExptReaderThread.numThreads;
    NiceExptReaderThread.numThreads++;
  }

  //-------------------------------------------------------------------------
  // Main thread method
  //-------------------------------------------------------------------------
  public
  void run(){
    TraceEntry te = null;
    OutputLogEntry ole = null;

    te = (TraceEntry)this.pendingReads.dequeue();
    while(te != null){
      try{
        ole = this.performSingleRead(te.getEventNum(), te.getObjId());
        this.finishedReads.add(ole);
        te = (TraceEntry)this.pendingReads.dequeue();
      }catch(IOException e){
        e.printStackTrace();
        System.err.println("" + e);
        assert(false);
      }
    }
    System.err.println("Thread " + this.threadNum + " is done!");
  }

  //-------------------------------------------------------------------------
  // Read the specified object
  //-------------------------------------------------------------------------
  private OutputLogEntry
  performSingleRead(int eventNum, ObjId objId) throws IOException{
    long eventStartTimeMS = 0;
    long eventEndTimeMS = 0;
    long eventDurationMS = 0;
    BodyMsg bodyMsg = null;
    OutputLogEntry ole = null;
    int dataEventNum = 0;
    ImmutableBytes ib = null;
    ExptFileBytes exptFileBytes = null;
    LocalInterface li = null;

    eventStartTimeMS = System.currentTimeMillis() - this.startTimeMS;
    indicateNewReadStart(eventStartTimeMS);
    bodyMsg = this.readObjBlock(objId);
    ib = bodyMsg.getBody();
    assert(ib.getLength() == Constants.FILE_SIZE_BYTES);
    try{
      exptFileBytes = new ExptFileBytes(ib);
      dataEventNum = exptFileBytes.getEventNum();
    }catch(InvalidExptFileBytesException e){
      e.printStackTrace();
      System.err.println("" + e);
      assert(false);
    }

    eventEndTimeMS = System.currentTimeMillis() - this.startTimeMS;
    indicateNewReadEnd(eventEndTimeMS);
    eventDurationMS = eventEndTimeMS - eventStartTimeMS;
    ole = new OutputLogEntry(eventNum,
                             eventStartTimeMS,
                             OutputLogEntry.READ,
                             eventDurationMS,
                             objId,
                             dataEventNum);
    return(ole);
  }

  //-------------------------------------------------------------------------
  // Block until we can read the specified object
  //-------------------------------------------------------------------------
  private BodyMsg
  readObjBlock(ObjId objId) throws IOException{
    BodyMsg bodyMsg = null;

    try{
      // Do a non-blocking read
      
      bodyMsg = this.localInterface.read(objId,
                                         0,
                                         Constants.FILE_SIZE_BYTES,
                                         false,
                                         true);
    }catch(ObjNotFoundException e){
      // This should not happen. We should have seen invals for all of
      // the writes done by the sender's initialize() method.
      e.printStackTrace();
      System.err.println("" + e);
      assert(false);
    }catch(ReadOfInvalidRangeException e){
      // Inform the controller and try again
      this.localInterface.informDemandReadMiss(objId,
                                               0,
                                               Constants.FILE_SIZE_BYTES);
    }

    if(bodyMsg == null){
      try{
        // Already sent out out a demand request for the object
        bodyMsg =
          this.localInterface.readBlockInvalid(objId,
                                               0,
                                               Constants.FILE_SIZE_BYTES,
                                               true);
      }catch(ObjNotFoundException e){
        e.printStackTrace();
        System.out.println("" + e);
        assert(false);
      }
    }
    assert(bodyMsg != null);
    return(bodyMsg);
  }

  //-------------------------------------------------------------------------
  // Record read start time stats
  //-------------------------------------------------------------------------
  private static synchronized void indicateNewReadStart(long timeMS){
    numReadsStarted++;
    lastReadStartTimeMS = timeMS;
  }

  //-------------------------------------------------------------------------
  // Record read end time stats
  //-------------------------------------------------------------------------
  private static synchronized void indicateNewReadEnd(long timeMS){
    numReadsFinished++;
    lastReadFinishTimeMS = timeMS;
  }

  //-------------------------------------------------------------------------
  // Return the time the last read started
  //-------------------------------------------------------------------------
  public static synchronized long getLastReadStartTimeMS(){
    return(lastReadStartTimeMS);
  }
  
  //-------------------------------------------------------------------------
  // Return the time the last read finished
  //-------------------------------------------------------------------------
  public static synchronized long getLastReadFinishTimeMS(){
    return(lastReadFinishTimeMS);
  }

  //-------------------------------------------------------------------------
  // Return the number of reads started
  //-------------------------------------------------------------------------
  public static synchronized long getNumReadsStarted(){
    return(numReadsStarted);
  }

  //-------------------------------------------------------------------------
  // Return the number of reads finished
  //-------------------------------------------------------------------------
  public static synchronized long getNumReadsFinished(){
    return(numReadsFinished);
  }

  //-------------------------------------------------------------------------
  // Data members
  //-------------------------------------------------------------------------
  private ProdConsQueue pendingReads;
  private List finishedReads;
  private long startTimeMS;
  private LocalInterface localInterface;
  private int threadNum;

  private static int numThreads = 0;
  private static long lastReadFinishTimeMS = 0;
  private static long lastReadStartTimeMS = 0;
  private static int numReadsStarted = 0;
  private static int numReadsFinished = 0;
}
