 /** 
 *  Code to parse/store an entry in log files printed by the Nice experiments. 
 **/ 
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.BufferedReader;
import java.io.Reader;
import java.io.EOFException;
import java.io.PrintStream;
import java.io.OutputStream;
import java.util.StringTokenizer;

// Used for testing
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;

public class OutputLogEntry implements Immutable, Cloneable{

 /** 
 *  Constants 
 **/ 
  public static final int READ = 1;
  public static final int WRITE = 2;

 /** 
 *  Constructor 
 **/ 
  public
  OutputLogEntry(int newEventNum,
                 long newTimeMS,
                 int newOperation,
                 long newDurationMS,
                 ObjId newObjId,
                 int newDataEventNum){
    assert((newOperation == READ) || (newOperation == WRITE));
    this.eventNum = newEventNum;
    this.timeMS = newTimeMS;
    this.operation = newOperation;
    this.durationMS = newDurationMS;
    this.objId = (ObjId)newObjId.clone();
    this.dataEventNum = newDataEventNum;
  }

 /** 
 *  Constructor 
 **/ 
  public
  OutputLogEntry(BufferedReader br)
    throws IOException, InvalidOutputLogEntryException{
    String line = null;
    StringTokenizer st = null;

    line = br.readLine();
    if(line == null){
      throw(new EOFException());
    }
    st = new StringTokenizer(line);
    this.eventNum = OutputLogEntry.parseEventNum(st);
    this.timeMS = OutputLogEntry.parseEventTimeMS(st);
    this.operation = OutputLogEntry.parseOperationType(st);
    this.durationMS = OutputLogEntry.parseEventDurationMS(st);
    this.objId = OutputLogEntry.parseObjId(st);
    this.dataEventNum = OutputLogEntry.parseDataEventNum(st);
  }

 /** 
 *  Write this object to an output stream 
 **/ 
  public void
  writeToOS(OutputStream os) throws IOException{
    PrintStream ps = null;

    assert((this.operation == OutputLogEntry.READ) ||
           (this.operation == OutputLogEntry.WRITE));
    ps = new PrintStream(os);
    ps.print("" + this.eventNum);
    ps.print("\t" + this.timeMS);
    if(this.operation == OutputLogEntry.READ){
      ps.print("\tREAD");
    }else{
      ps.print("\tWRITE");
    }
    ps.print("\t" + this.durationMS);
    ps.print("\t\"" + this.objId.getPath() + "\"");
    ps.print("\t" + this.dataEventNum);
    ps.println();
    ps.flush();
  }

 /** 
 *  Return the event number 
 **/ 
  public int
  getEventNum(){
    return(this.eventNum);
  }

 /** 
 *  Return the time of the event 
 **/ 
  public long
  getTimeMS(){
    return(this.timeMS);
  }

 /** 
 *  Return the operation type 
 **/ 
  public int
  getOperation(){
    return(this.operation);
  }

 /** 
 *  Return the duration of the event 
 **/ 
  public long
  getDurationMS(){
    return(this.durationMS);
  }

 /** 
 *  Return the time of the event 
 **/ 
  public ObjId
  getObjId(){
    return(this.objId);
  }

 /** 
 *  Return the data event number 
 **/ 
  public int
  getDataEventNum(){
    return(this.dataEventNum);
  }

 /** 
 *  Return true if "o" equals this 
 **/ 
  public boolean
  equals(Object o){
    OutputLogEntry ole = null;
    boolean result = false;

    // 2 constants, 2 java-provided fields, 6 declared fields
    assert(this.getClass().getDeclaredFields().length == 10);

    if(o instanceof OutputLogEntry){
      ole = (OutputLogEntry)o;
      result = ((this.eventNum == ole.eventNum) &&
                (this.timeMS == ole.timeMS) &&
                (this.operation == ole.operation) &&
                (this.durationMS == ole.durationMS) &&
                (this.objId.equals(ole.objId)) &&
                (this.dataEventNum == ole.dataEventNum));
    }else{
      result = false;
    }
    return(result);
  }

 /** 
 *  Clone this object 
 **/ 
  public Object
  clone(){
    // Because this object is immutable, we do not need to do anything
    return(this);
  }

 /** 
 *  Return a hashCode (note: doesn't currently do anything) 
 **/ 
  public int
  hashCode(){
    assert(false);
    return(0);
  }

 /** 
 *  Used for testing 
 **/ 
  public static void
  main(String[] argv){
    if(argv.length < 1){
      Env.verifyAssertEnabled();
      System.out.println("Testing OutputLogEntry.java...");
      OutputLogEntry.test1();
      OutputLogEntry.test2();
      System.out.println("...Finished");
    }else{
      OutputLogEntry.readLoopTest();
    }
  }

 /** 
 *  Test OutputLogEntry 
 **/ 
  private static void
  test1(){
    OutputLogEntry ole1 = null;
    OutputLogEntry ole2 = null;
    ByteArrayOutputStream baos = null;
    ByteArrayInputStream bais = null;
    BufferedReader br = null;
    long timeMS = 0;

    try{
      baos = new ByteArrayOutputStream();
      timeMS = System.currentTimeMillis();
      ole1 = new OutputLogEntry(10,               // event num
                                timeMS,           // event time (ms)
                                READ,             // event operation
                                2,                // event duration (ms)
                                new ObjId("/0"),  // object id
                                -1);              // data event num
      ole1.writeToOS(baos);
      assert(ole1.equals(ole1));
      //ole1.writeToOS(System.out); // Already tested

      bais = new ByteArrayInputStream(baos.toByteArray());
      br = new BufferedReader(new InputStreamReader(bais));
      ole2 = new OutputLogEntry(br);
      //ole2.writeToOS(System.out); // Already tested
      try{
          ole2 = new OutputLogEntry(br);
          assert(false);
      }catch(EOFException e){
        // Do nothing; this was expected.
      }
      assert(ole1 != ole2);
      assert(ole1.equals(ole2));
    }catch(IOException e){
      assert(false);
    }catch(InvalidOutputLogEntryException e){
      System.err.println("" + e);
      assert(false);
    }
  }

 /** 
 *  Test OutputLogEntry 2 
 **/ 
  private static void
  test2(){
    OutputLogEntry ole = null;
    long timeMS = 0;

    timeMS = System.currentTimeMillis();
    ole = new OutputLogEntry(3,                 // event num
                             timeMS,            // event time (ms)
                             WRITE,             // operation type
                             2,                 // event duration (ms)
                             new ObjId("/0"),   // obj id
                             1);                // data event num
    assert(ole.getEventNum() == ole.eventNum);
    assert(ole.getTimeMS() == ole.timeMS);
    assert(ole.getOperation() == ole.operation);
    assert(ole.getDurationMS() == ole.durationMS);
    assert(ole.getObjId().equals(ole.objId));
    assert(ole.getDataEventNum() == ole.dataEventNum);
  }

 /** 
 *  Try reading a file from stdin 
 **/ 
  private static void
  readLoopTest(){
    OutputLogEntry ole = null;
    BufferedReader reader = null;

    try{
      reader = new BufferedReader(new InputStreamReader(System.in));
      while(true){
        ole = new OutputLogEntry(reader);
        ole.writeToOS(System.out);
      }
    }catch(EOFException e){
      System.out.println("Done!");
    }catch(InvalidOutputLogEntryException e){
      System.out.println("" + e);
      assert(false);
    }catch(IOException e){
      System.out.println("" + e);
      assert(false);
    }
  }

 /** 
 *  Parse the event number (mostly copied from TraceEntry.java) 
 **/ 
  private static int
  parseEventNum(StringTokenizer st)
    throws IOException, InvalidOutputLogEntryException{
    String token = null;
    int readEventNum = 0;
    boolean badEventNum = false;

    // Read the event number
    if(st.hasMoreTokens()){
      token = st.nextToken();
      try{
        readEventNum = Integer.parseInt(token);
      }catch(NumberFormatException nfe){
        badEventNum = true;
      }
    }else{
      badEventNum = true;
    }
    if(badEventNum){
      throw(new InvalidOutputLogEntryException("First field should be " +
                                               "the event number!"));
    }
    return(readEventNum);
  }

 /** 
 *  Parse the event time in milliseconds (mostly copied from TraceEntry.java) 
 **/ 
  private static long
  parseEventTimeMS(StringTokenizer st)
    throws IOException, InvalidOutputLogEntryException{
    String token = null;
    long readEventTimeMS = 0;
    boolean badTimeMS = false;

    // Read the event time in ms
    if(st.hasMoreTokens()){
      token = st.nextToken();
      try{
        readEventTimeMS = Long.parseLong(token);
      }catch(NumberFormatException nfe){
        badTimeMS = true;
      }
    }else{
      badTimeMS = true;
    }
    if(badTimeMS){
      throw(new InvalidOutputLogEntryException("The second field should be " +
                                               "the event time in ms!"));
    }
    return(readEventTimeMS);
  }

 /** 
 *  Parse the operation type (mostly copied from TraceEntry.java) 
 **/ 
  private static int
  parseOperationType(StringTokenizer st)
    throws IOException, InvalidOutputLogEntryException{
    String token = null;
    int operationType = READ;
    boolean badOperation = false;

    // Read the operation type
    if(st.hasMoreTokens()){
      token = st.nextToken();
      if(token.equalsIgnoreCase("read")){
        operationType = OutputLogEntry.READ;
      }else if(token.equalsIgnoreCase("write")){
        operationType = OutputLogEntry.WRITE;
      }else{
        badOperation = true;
      }
    }else{
      badOperation = true;
    }
    if(badOperation){
      throw(new InvalidOutputLogEntryException("The third column should " +
                                               "either be READ or WRITE!"));
    }
    return(operationType);
  }

 /** 
 *  Parse the event duration in milliseconds (mostly copied from 
 *  TraceEntry.java) 
 **/ 
  private static long
  parseEventDurationMS(StringTokenizer st)
    throws IOException, InvalidOutputLogEntryException{
    String token = null;
    long readEventDurationMS = 0;
    boolean badDurationMS = false;

    // Read the event time in ms
    if(st.hasMoreTokens()){
      token = st.nextToken();
      try{
        readEventDurationMS = Long.parseLong(token);
      }catch(NumberFormatException nfe){
        badDurationMS = true;
      }
    }else{
      badDurationMS = true;
    }
    if(badDurationMS){
      throw(new InvalidOutputLogEntryException("The fourth field should be " +
                                               "the event duration in ms!"));
    }
    return(readEventDurationMS);
  }

 /** 
 *  Parse the object ID 
 **/ 
  private static ObjId
  parseObjId(StringTokenizer st)
    throws IOException, InvalidOutputLogEntryException{
    String token = null;
    String objIdPath = null;
    ObjId readObjId = null;
    boolean badObjId = false;

    // Read the event time in ms
    if(st.hasMoreTokens()){
      token = st.nextToken();
      if((token.startsWith("\"")) && (token.endsWith("\""))){
        objIdPath = token.substring(1, token.length() - 1);
        readObjId = new ObjId(objIdPath);
      }else{
        badObjId = true;
      }
    }else{
      badObjId = true;
    }
    if(badObjId){
      throw(new InvalidOutputLogEntryException("The fifth field should be " +
                                               "the object ID"));
    }
    return(readObjId);
  }

 /** 
 *  Parse the data event number (mostly copied from TraceEntry.java) 
 **/ 
  private static int
  parseDataEventNum(StringTokenizer st)
    throws IOException, InvalidOutputLogEntryException{
    String token = null;
    int readDataEventNum = 0;
    boolean badDataEventNum = false;

    // Read the event number
    if(st.hasMoreTokens()){
      token = st.nextToken();
      try{
        readDataEventNum = Integer.parseInt(token);
      }catch(NumberFormatException nfe){
        badDataEventNum = true;
      }
    }else{
      badDataEventNum = true;
    }
    if(badDataEventNum){
      throw(new InvalidOutputLogEntryException("The sixth field should be " +
                                               "the data event number!"));
    }
    return(readDataEventNum);
  }

 /** 
 *  Data members 
 **/ 
  private int eventNum;
  private long timeMS;
  private int operation;
  private long durationMS;
  private ObjId objId;
  private int dataEventNum;
}

 /** 
 *  Exception that is thrown when we read an invalid entry in our trace file 
 **/ 
class InvalidOutputLogEntryException extends Exception{

 /** 
 *  Constructor 
 **/ 
  public
  InvalidOutputLogEntryException(String newMsg){
    this.msg = newMsg;
  }

 /** 
 *  Convert this exception to a string 
 **/ 
  public String
  toString(){
    return(this.msg);
  }

 /** 
 *  Data members 
 **/ 
  private String msg;
}
