 /** 
 *  Code to parse/store an entry in log files printed by the Nice experiments. 
 **/ 
import java.io.IOException;
import java.io.OutputStream;
import java.io.ByteArrayOutputStream;

// Used for testing
import java.io.ByteArrayInputStream;
import java.io.BufferedReader;
import java.io.InputStreamReader;

public class OutputLogger{

 /** 
 *  Local constants 
 **/ 
  private static int INIT_BUFFER_SIZE = 100000;

 /** 
 *  Constructor 
 **/ 
  public
  OutputLogger(OutputStream newOS, int newMaxNumEntries){
    this.os = newOS;
    this.numEntriesSinceFlush = 0;
    this.maxNumEntries = newMaxNumEntries;

    // Initialize capacity to 100K
    this.baos = new ByteArrayOutputStream(INIT_BUFFER_SIZE);
  }

 /** 
 *  Add a new OutputLogEntry to the log 
 **/ 
  public synchronized void
  addEntry(OutputLogEntry ole) throws IOException{
    ole.writeToOS(this.baos);
    this.numEntriesSinceFlush++;
    if(this.numEntriesSinceFlush >= this.maxNumEntries){
      // Write our stored bytes to the underlying output stream
      this.baos.writeTo(this.os);
      this.baos.flush();
      this.baos.reset();
      this.numEntriesSinceFlush = 0;
    }
  }

 /** 
 *  Flush this stream 
 **/ 
  public synchronized void
  flush() throws IOException{
    // Really flush this stream (don't just copy "baos" to "os")
    this.baos.flush();
    this.baos.writeTo(this.os);
    this.os.flush();
    this.baos.reset();
    this.numEntriesSinceFlush = 0;
  }

 /** 
 *  Used for testing 
 **/ 
  public static void
  main(String[] argv){
    OutputLogger ol = null;

    Env.verifyAssertEnabled();
    System.out.println("Testing OutputLogger...");
    OutputLogger.test1();
    // OutputLogger.screenTest(); // Tried this already
    System.out.println("...Finished");
  }

 /** 
 *  Test the class 
 **/ 
  private static void
  test1(){
    ByteArrayOutputStream baos = null;
    OutputLogger ol = null;
    OutputLogEntry ole1 = null;
    OutputLogEntry ole2 = null;
    OutputLogEntry ole3 = null;
    OutputLogEntry ole1copy = null;
    OutputLogEntry ole2copy = null;
    OutputLogEntry ole3copy = null;
    ByteArrayInputStream bais = null;
    BufferedReader br = null;

    try{
      baos = new ByteArrayOutputStream();
      ol = new OutputLogger(baos, 2);
      ole1 = new OutputLogEntry(1,
                                100,
                                OutputLogEntry.WRITE,
                                2,
                                new ObjId("/1"),
                                -1);
      ole2 = new OutputLogEntry(2,
                                200,
                                OutputLogEntry.READ,
                                2,
                                new ObjId("/2"),
                                -1);
      ole3 = new OutputLogEntry(3,
                                300,
                                OutputLogEntry.READ,
                                3,
                                new ObjId("/3"),
                                -1);
      ol.addEntry(ole1);
      ol.addEntry(ole2);
      ol.addEntry(ole3);

      // Make sure the first two entries are written
      bais = new ByteArrayInputStream(baos.toByteArray());
      baos.reset();
      br = new BufferedReader(new InputStreamReader(bais));
      ole1copy = new OutputLogEntry(br);
      assert(ole1.equals(ole1copy) && ole1copy.equals(ole1));
      ole2copy = new OutputLogEntry(br);
      assert(ole2.equals(ole2copy) && ole2copy.equals(ole2));

      // Make sure the third entry isn't available
      assert(!br.ready());

      // Make the third entry visible
      ol.flush();
      bais = new ByteArrayInputStream(baos.toByteArray());
      br = new BufferedReader(new InputStreamReader(bais));
      assert(br.ready());
      ole3copy = new OutputLogEntry(br);
      assert(ole3.equals(ole3copy) && ole3copy.equals(ole3));
      assert(!br.ready());
    }catch(IOException e){
      e.printStackTrace();
      System.out.println("" + e);
      assert(false);
    }catch(InvalidOutputLogEntryException e){
      e.printStackTrace();
      System.out.println("" + e);
      assert(false);
    }
  }


 /** 
 *  Test the class visually 
 **/ 
  private static void
  screenTest(){
    OutputLogger ol = null;
    OutputLogEntry ole1 = null;
    OutputLogEntry ole2 = null;
    OutputLogEntry ole3 = null;
    OutputLogEntry ole1copy = null;
    OutputLogEntry ole2copy = null;
    OutputLogEntry ole3copy = null;
    ByteArrayInputStream bais = null;
    BufferedReader br = null;

    try{
      ol = new OutputLogger(System.out, 2);
      ole1 = new OutputLogEntry(1,
                                100,
                                OutputLogEntry.WRITE,
                                1,
                                new ObjId("/1"),
                                -1);
      ole2 = new OutputLogEntry(2,
                                200,
                                OutputLogEntry.READ,
                                2,
                                new ObjId("/2"),
                                -1);
      ole3 = new OutputLogEntry(3,
                                300,
                                OutputLogEntry.READ,
                                3,
                                new ObjId("/3"),
                                -1);
      ol.addEntry(ole1);
      ol.addEntry(ole2);
      ol.addEntry(ole3);

      // Make sure the first two entries are written
      System.out.println("Should have written two entries!");

      // Make the third entry visible
      ol.flush();
      System.out.println("Should have written the third entry!");
    }catch(IOException e){
      e.printStackTrace();
      System.out.println("" + e);
      assert(false);
    }
  }

 /** 
 *  Data members 
 **/ 
  private OutputStream os;
  private int maxNumEntries;
  private int numEntriesSinceFlush;
  private ByteArrayOutputStream baos;
}
