 /** 
 *  Code to generate a read/write log the sender and receiver 
 **/ 
import java.io.IOException;
import java.io.OutputStream;
import java.util.Random;

public class TraceGenerator{

 /** 
 *  Locally used constants 
 **/ 
  private static long RAND_SEED = 2789;

 /** 
 *  Program starting point 
 **/ 
  public static void
  main(String[] argv){
    double[] normalizedZipf = null;

    if(argv.length <= 0){
      try{
        normalizedZipf = TraceGenerator.makeNormalizedZipf(Constants.NUM_FILES,
                                                           Constants.ALPHA);
        TraceGenerator.makeTrace(System.out, normalizedZipf);
      }catch(IOException e){
        System.out.println("" + e);
        assert(false);
      }
    }else{
      // Test the distribution of random file numbers that we generate
      Random rand = null;
      int fileNum = 0;
      int valCount[] = null;

      valCount = new int[10];  // Note: Automatically initialized to 0's
      rand = new Random(TraceGenerator.RAND_SEED);
      normalizedZipf = TraceGenerator.makeNormalizedZipf(valCount.length,
                                                         Constants.ALPHA);
      for(int i = 0; i < 10000; i++){
        fileNum = TraceGenerator.getReadFileNum(normalizedZipf, rand);
        valCount[fileNum]++;
      }

      for(int i = 0; i < valCount.length; i++){
        System.out.println("fileNum " + i + " occurred " + valCount[i] +
                           " times");
      }
    }
  }

 /** 
 *  Generate the trace 
 **/ 
  private static void
  makeTrace(OutputStream os, double[] normalizedZipf) throws IOException{
    Random randGen = null;
    long exptTimeMS = 0;
    double readProb = 0.0;
    int fileNum = 0;
    ObjId objId = null;
    TraceEntry te = null;
    long delayTimeMS = 0;

    randGen = new Random(TraceGenerator.RAND_SEED);

    // Note: Event numbers start from 1, not 0
    for(int i = 1; i <= Constants.NUM_OPERATIONS; i++){
      delayTimeMS = (long)(randGen.nextDouble() *
                           Constants.MAX_INTER_OPERATION_TIME_MS);
      exptTimeMS += delayTimeMS;
      readProb = randGen.nextDouble();
      if(readProb >= Constants.READ_FRAC){
        fileNum = (int)(randGen.nextDouble() * Constants.NUM_FILES);
        objId = new ObjId("/" + fileNum);
        te = new TraceEntry(i,
                            exptTimeMS,
                            TraceEntry.WRITE,
                            normalizedZipf[fileNum],
                            objId);
        te.writeToOS(os);
      }else{
        fileNum = TraceGenerator.getReadFileNum(normalizedZipf, randGen);
        objId = new ObjId("/" + fileNum);
        te = new TraceEntry(i,
                            exptTimeMS,
                            TraceEntry.READ,
                            Core.DEFAULT_PRIORITY,
                            objId);
        te.writeToOS(os);
      }
    }
  }

 /** 
 *  Make a normalized, Zipf-distributed histogram 
 **/ 
  private static double[]
  makeNormalizedZipf(int numFiles, double alpha){
    int ii = 0;
    double[] normalizedZipf = null;
    double sum = 0;
    double check = 0;

    normalizedZipf = new double[Constants.NUM_FILES];
    for(ii = 0; ii < numFiles; ii++){
      normalizedZipf[ii] = 1.0/Math.pow(ii + 1, alpha);
      sum += normalizedZipf[ii];
    }

    for(ii = 0; ii < numFiles; ii++){
      normalizedZipf[ii] = normalizedZipf[ii] / sum;
      check += normalizedZipf[ii];
    }

    assert(check > .99999 && check < 1.00001);
    return(normalizedZipf);
  }

 /** 
 *  Generate a file number to read (chosen from our Zipf distribution) 
 **/ 
  private static int
  getReadFileNum(double[] normalizedZipf, Random randGen){
    int fileNum = 0;
    double randNum = 0.0;
    boolean fileNumChosen = false;

    assert((normalizedZipf != null) &&
           (normalizedZipf.length == Constants.NUM_FILES));
    randNum = randGen.nextDouble();
    while((!fileNumChosen) && (fileNum < Constants.NUM_FILES)){
      if(randNum > normalizedZipf[fileNum]){
        randNum -= normalizedZipf[fileNum];
        fileNum++;
      }else{
        fileNumChosen = true;
      }
    }
    if(fileNum >= Constants.NUM_FILES){
      // Floating-point precision issue
      fileNum = Constants.NUM_FILES - 1; // File numbers start from 0
    }
    return(fileNum);
  }
}
