 /** 
 *  SenderScriptGen: Generate files used by our experiments 
 **/ 
import java.util.Vector;
import java.util.LinkedList;
import java.util.ListIterator;
import java.util.Random;
import java.io.PrintStream;
import java.io.OutputStream;

public class SenderScriptGen
{
    public static final int BRANCHING_FACTOR = 10;
    public static final int RANDOM_SEED = 4356;

 /** 
 *  Constructor 
 **/ 
    public SenderScriptGen()
    {
    }

 /** 
 *  Print script starting messages 
 **/ 
    public static void printScriptPreamble()
    {
	System.out.println("c Start of script");
    }

 /** 
 *  Print script intermediate messages 
 **/ 
    public static void printScriptIntermediate(int numWrites)
    {
	System.out.println("c Wait for B to subscribe + receive all invals");
	System.out.println("l 18000");
	System.out.println("c Do " + numWrites + " random writes");
    }

 /** 
 *  Print script ending messages 
 **/ 
    public static void printScriptPostamble()
    {
	System.out.println("c End of script");
	System.out.println("c Coordinator will now send demand RMIs");
    }


    /* Memoryless writes generation functions:
             Note: only generates a full balanced tree if the numFiles is in the form of 10^(x).
       */
 
    public void createSequentialWrites(long numFiles, long fileSize,  boolean bound, OutputStream os){
	PrintStream ps = new PrintStream(os);
	for(long i=0; i<numFiles; i++){
	    String s =  createFileName(i, numFiles);
	    ps.println("w " + s + " " + 0 + "  " + fileSize + " " + (bound ? 1 : 0)
                       + " " + 0);
	}
    }

public void createSequentialReads(long numFiles, long fileSize,  OutputStream os){
	PrintStream ps = new PrintStream(os);
	for(long i=0; i<numFiles; i++){
	    String s =  createFileName(i, numFiles);
            ps.println("r " + s + " " + 0 + "  " + fileSize );
	}
    }

public void createSequentialNonBlockReads(long numFiles, long fileSize,  OutputStream os){
	PrintStream ps = new PrintStream(os);
	for(long i=0; i<numFiles; i++){
	    String s =  createFileName(i, numFiles);
            ps.println("n " + s + " " + 0 + "  " + fileSize );
	}
    }

    public void createPercentageWrites(long fileSize, boolean bound, double dRatio, double fRatio, long numFiles, OutputStream os){
	PrintStream ps = new PrintStream(os);
	Random r = new Random(RANDOM_SEED);
	long numWrites = Math.round(numFiles * dRatio);

	for(long i=0; i<numWrites; i=i+(long)(1/fRatio)){
	    String s = createFileName(i, numFiles);
	    long off = 0;//Math.abs(r.nextLong())%fileSize;
	    long len = fileSize;//Math.abs(r.nextLong())%(fileSize-off+1);
	    assert(off+len<=fileSize);
	    ps.println("w " + s + " " + off + "  " + len + " " + (bound ? 1 : 0) 
                       + " " + 0);
	}
	
    }

    public void createRandomWrites(long fileSize, boolean bound, long numWrites, long numFiles, double ratio, OutputStream os){
	assert(ratio>0 && ratio<=1);
	PrintStream ps = new PrintStream(os);
	Random r = new Random(RANDOM_SEED);
	for(long i=0; i<numWrites; i++){
	    long baseName = Math.abs(r.nextLong())%Math.round(numFiles*ratio);
	    String s = createFileName(baseName, numFiles);
	    long off = 0; /*Math.abs(r.nextLong())%fileSize; */
	    long len = fileSize; /*Math.abs(r.nextLong())%(fileSize-off+1);*/
	    assert(off+len<=fileSize);
	    ps.println("w " + s + " " + off + " " + len + " " + (bound ? 1 : 0)
                       + " " + 0);
	}
	// always print a write what each device will seek for at the end
	// easy to do sync. body if each device only sync. on this file.
	long seed = Math.round(numFiles*ratio);
	while(seed>1){
	    String s = createFileName(seed-1, numFiles);
	    ps.println("w " + s + " " + 0 + " " + fileSize + " " + (bound ? 1 : 0)
                       + " " + 0);
	    seed = seed/10;
	}
    }

    public String createFileName(long n, long l){
      assert n <= l;
      byte[] bName = null;
      if (l<BRANCHING_FACTOR){
        bName = new byte[2];
        bName[0] = '/';
        bName[1] = (byte) ('0' + n);
      } else {
	long numLen = Math.round(Math.log(l)/Math.log(BRANCHING_FACTOR));
	int fileLen = 2 * (int)numLen; // including /'s as seperator
        bName = new byte[fileLen];
	for(int i=0; i<fileLen; i++){
	    if(i%2 == 0){// even position
		bName[i] = '/';
	    } else {// odd position
		long base = Math.round(Math.pow(BRANCHING_FACTOR, numLen-(i+1)/2));
		bName[i] = (byte) ('0' + n/base);
		n = n%base;
		assert(bName[i]>='0' && bName[i]<='9');
	    }
	}
      }
      return new String(bName);
    }


 /** 
 *  Program staring point 
 **/ 
    public static void main(String[] argv)
    {
	int numFiles = 0;
	int fileSize = 0;
	int numWrites = 0;
	boolean bound = false;
	String startDirName = null;
	SenderScriptGen ssg = null;

	if(argv.length == 5){
	    ssg = new SenderScriptGen();
	    numFiles = Integer.parseInt(argv[0]);
	    startDirName = argv[1];
	    fileSize = Integer.parseInt(argv[2]);
	    assert((argv[3].equals("1")) || (argv[3].equals("0")));
	    bound = (argv[3].equals("1"));
	    numWrites = Integer.parseInt(argv[4]);

	    ssg.createPercentageWrites(fileSize, bound, 0.1, 0.1, numFiles, System.out);
/*
	    ssg.makeTree(numFiles, startDirName);
	    ssg.printScriptPreamble();
	    ssg.printAllObjectMods(fileSize, bound, System.out);
	    ssg.printScriptIntermediate(numWrites);
	    ssg.printRandomWrites(fileSize, bound, numWrites, System.out);
	    ssg.printScriptPostamble();*/
	}else{
	    System.err.println("Usage: java SenderScriptGen <numfiles> " +
			       "<startDirName> <fileSize> <bound> " +
			       "<numWrites>");
	}
    }
}

 /** 
 *  $Log: SenderScriptGen.java,v $ 
 *  Revision 1.2  2005/01/18 22:49:43  zjiandan 
 *  Rewrited Class Serialization for network deliveringto reduce the overall bandwidth. 
 *  
 *  Revision 1.1  2005/01/13 20:55:41  zjiandan 
 *  Reorganized sosp experiments files into sosp subdirectory under experiments. 
 *  
 *  Revision 1.9  2005/01/10 03:47:48  zjiandan 
 *  Fixed some bugs. Successfully run SanityCheck and Partial Replication experiments. 
 *  
 *  Revision 1.8  2004/05/25 08:16:46  lgao 
 *  *** empty log message *** 
 *  
 *  Revision 1.7  2004/05/24 03:09:07  lgao 
 *  dd syncBodyCheck... 
 *  
 *  Revision 1.6  2004/05/21 10:23:15  nayate 
 *  Minor changes 
 *  
 *  Revision 1.5  2004/05/21 08:08:12  lgao 
 *  *** empty log message *** 
 *  
 *  Revision 1.4  2004/05/21 00:04:28  lgao 
 *  *** empty log message *** 
 *  
 *  Revision 1.3  2004/05/20 19:28:37  nayate 
 *  Minor change + 1 off-by-one bug fix 
 *  
 *  Revision 1.2  2004/05/20 10:04:40  lgao 
 *  Add memoryless functions to generate writes, including percentage generation. 
 *  
 *  Revision 1.1  2004/05/20 02:14:17  nayate 
 *  Added files to allow compilation 
 *  
 *  Revision 1.5  2004/05/19 01:46:37  arun 
 *  *** empty log message *** 
 *  
 *  Revision 1.4  2004/05/18 09:23:37  nayate 
 *  Changed SenderScriptGen to use a chosen random seed 
 *  
 *  Revision 1.3  2004/05/18 08:56:10  nayate 
 *  Changed the sender script generator to first generate entries for all 
 *  files 
 *  
 *  Revision 1.2  2004/05/17 22:06:34  nayate 
 *  Added CVS logging 
 *  
 **/ 
