 /** 
 *  Code to parse/save an entry from the Harvard trace files 
 **/ 
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.BufferedReader;
import java.io.Reader;
import java.io.EOFException;
import java.io.PrintStream;
import java.io.OutputStream;
import java.util.StringTokenizer;

// Used for testing
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;

public class HarvardTraceEntry{

 /** 
 *  Constants 
 **/ 
  public static final int TCP_PROTOCOL = 101;
  public static final int UDP_PROTOCOL = 102;

  public static final int NFS_V3_CALL = 201;
  public static final int NFS_V3_RESPONSE = 202;
  public static final int NFS_V2_CALL = 203;
  public static final int NFS_V2_RESPONSE = 204;

  private static final int NUM_SKIP_LAST_TOKENS_CALL = 6;
  private static final int NUM_SKIP_LAST_TOKENS_RESPONSE = 10;

  private static final String[] REMOVE_TOKENS = {"JUMBO", "LONGPACKET"};
  private static final String EXTRANEOUS_RESPONSE =
  "status=XXX pl = XXX con = XXX len = XXX";
  private static final String EXTRANEOUS_CALL = "con = XXX len = XXX";

 /** 
 *  Constructor 
 **/ 
  public
  HarvardTraceEntry(long newTimeSec,
                    long newTimeUSec,
                    int newSourceAddr,
                    int newSourcePort,
                    int newDestinationAddr,
                    int newDestinationPort,
                    short newInetProtocol,
                    short newNFSProtocol,
                    long newXID,
                    short newMethodNum,
                    String newMethodName,
                    int newResponseCode,
                    FieldValuePair[] newFieldValues){
    this.timeSec = newTimeSec;
    this.timeUSec = newTimeUSec;
    this.sourceAddr = newSourceAddr;
    this.sourcePort = newSourcePort;
    this.destinationAddr = newDestinationAddr;
    this.destinationPort = newDestinationPort;
    this.inetProtocol = newInetProtocol;
    this.nfsProtocol = newNFSProtocol;
    this.xid = newXID;
    this.methodNum = newMethodNum;
    this.methodName = newMethodName;
    this.responseCode = newResponseCode;
    this.fieldValues = newFieldValues;
  }

 /** 
 *  Constructor 
 **/ 
  public
  HarvardTraceEntry(BufferedReader br)
    throws IOException, InvalidHarvardTraceEntryException{
    int numSkipLastTokens = 0;
    String line = null;
    StringTokenizer st = null;
    String fieldValuePairStr = null;

    line = br.readLine();
    if(line == null){
      throw(new EOFException());
    }
    st = new StringTokenizer(line);
    line = HarvardTraceEntry.cleanupLine(line);
    this.parseTime(st);
    this.parseSourceAddress(st);
    this.parseDestinationAddress(st);
    this.parseInetProtocol(st);
    this.parseNFSProtocol(st);
    this.parseXID(st);
    this.parseMethodNum(st);
    this.parseMethodName(st);
    if(this.isResponse()){
      this.parseResponseCode(st);
      numSkipLastTokens = NUM_SKIP_LAST_TOKENS_RESPONSE;
    }else{
      numSkipLastTokens = NUM_SKIP_LAST_TOKENS_CALL;
    }
    this.parseFieldValuePairs(st, numSkipLastTokens);
  }

 /** 
 *  Write this object to an output stream 
 **/ 
  public void
  writeToOS(OutputStream os) throws IOException{
    PrintStream ps = null;

    ps = new PrintStream(os);
    ps.print("" + this.timeSec +
             "." + HarvardTraceEntry.getDecimalString(this.timeUSec, 6));
    ps.print(" " + Integer.toHexString(this.sourceAddr) +
             "." + HarvardTraceEntry.getHexString(this.sourcePort, 4));
    ps.print(" " + Integer.toHexString(this.destinationAddr) +
             "." + HarvardTraceEntry.getHexString(this.destinationPort, 4));
    ps.print(" " + this.getInetProtocolString());
    ps.print(" " + this.getNFSProtocolString());
    ps.print(" " + HarvardTraceEntry.getHexString(this.xid, 8));
    ps.print(" " + this.methodNum);
    ps.print(" " + this.methodName);
    if(this.isResponse()){
      ps.print((this.responseCode == 0) ? " OK" : " " + this.responseCode);
    }
    for(int i = 0; i < this.fieldValues.length; i++){
      ps.print(" " + this.fieldValues[i].getFieldName() +
               " " + this.fieldValues[i].getValue());
    }
    if(this.isResponse()){
      ps.print(" " + EXTRANEOUS_RESPONSE);
    }else{
      ps.print(" " + EXTRANEOUS_CALL);
    }
    ps.println();
    ps.flush();
  }

 /** 
 *  Return true if "o" equals this 
 **/ 
  public boolean
  equals(Object o){
    HarvardTraceEntry hte = null;
    boolean result = false;

    // 11 constants, 2 java-provided fields, 13 declared fields
    assert(this.getClass().getDeclaredFields().length == 26);

    if(o instanceof HarvardTraceEntry){
      hte = (HarvardTraceEntry)o;
      result = ((this.timeSec == hte.timeSec) &&
                (this.timeUSec == hte.timeUSec) &&
                (this.sourceAddr == hte.sourceAddr) &&
                (this.sourcePort == hte.sourcePort) &&
                (this.destinationAddr == hte.destinationAddr) &&
                (this.destinationPort == hte.destinationPort) &&
                (this.inetProtocol == hte.inetProtocol) &&
                (this.nfsProtocol == hte.nfsProtocol) &&
                (this.xid == hte.xid) &&
                (this.methodNum == hte.methodNum) &&
                (this.methodName.equals(hte.methodName)) &&
                (this.responseCode == hte.responseCode) &&
                (this.compareFieldValues(hte.fieldValues)));
    }else{
      result = false;
    }
    return(result);
  }

 /** 
 *  Return the seconds component of the time 
 **/ 
  public long
  getTimeSec(){
    return(this.timeSec);
  }

 /** 
 *  Return the micro-seconds component of the time 
 **/ 
  public long
  getTimeUSec(){
    return(this.timeUSec);
  }

 /** 
 *  Return the source address 
 **/ 
  public int
  getSourceAddr(){
    return(this.sourceAddr);
  }

 /** 
 *  Return the source port 
 **/ 
  public int
  getSourcePort(){
    return(this.sourcePort);
  }

 /** 
 *  Return the destination address 
 **/ 
  public int
  getDestinationAddr(){
    return(this.destinationAddr);
  }

 /** 
 *  Return the destination port 
 **/ 
  public int
  getDestinationPort(){
    return(this.destinationPort);
  }

 /** 
 *  Return the Internet packet protocol used 
 **/ 
  public short
  getInetProtocol(){
    return(this.inetProtocol);
  }

 /** 
 *  Return the NFS RPC version and data direction 
 **/ 
  public short
  getNFSProtocol(){
    return(this.nfsProtocol);
  }

 /** 
 *  Return the transaction Id of this RPC call 
 **/ 
  public long
  getXID(){
    return(this.xid);
  }

 /** 
 *  Return the method number 
 **/ 
  public short
  getMethodNum(){
    return(this.methodNum);
  }

 /** 
 *  Return the method name 
 **/ 
  public String
  getMethodName(){
    return(this.methodName);
  }

 /** 
 *  Return the response code 
 **/ 
  public int
  getResponseCode(){
    return(this.responseCode);
  }

 /** 
 *  Return the field/value pairs (note: we don't make a copy!) 
 **/ 
  public FieldValuePair[]
  getFieldValues(){
    return(this.fieldValues);
  }
  
 /** 
 *  Return the filehandle if any, else return null; 
 **/ 
  public String
  getFileHandle(){
    String fh = null;

    for(int i = 0; i < this.fieldValues.length; i++){
      if((this.fieldValues[i].getFieldName().equals("fh"))){
        fh = this.fieldValues[i].getValue();
        break;
      }
    }
    return fh;
  }

 /** 
 *  Return the fileId if any, else return null; 
 **/ 
  public String
  getFileId(){
    String fh = null;

    for(int i = 0; i < this.fieldValues.length; i++){
      if((this.fieldValues[i].getFieldName().equals("fileid"))){
        fh = this.fieldValues[i].getValue();
        break;
      }
    }
    return fh;
  }

 /** 
 *  Return the offset if any, else return -1; 
 **/ 
  public long
  getOffset(){
    long fh = -1;

    for(int i = 0; i < this.fieldValues.length; i++){
      if((this.fieldValues[i].getFieldName()).equals("off")){
        fh = (Long.decode("0x"+this.fieldValues[i].getValue())).longValue();
        break;
      }
    }
    return fh;
  }

 /** 
 *  Return the length if any, else return -1; 
 **/ 
  public long
  getLen(){
    long fh = -1;

    for(int i = 0; i < this.fieldValues.length; i++){
      if(this.fieldValues[i].getFieldName().equals("count")){
        fh = (Long.decode("0x"+(this.fieldValues[i].getValue()))).longValue();
        break;
      }
    }
    return fh;
  }

 /** 
 *  Return the UserID if any, else return null; 
 **/ 
  public String
  getUID(){
    String fh = null;

    for(int i = 0; i < this.fieldValues.length; i++){
      if(this.fieldValues[i].getFieldName().equals("uid")){
        fh = this.fieldValues[i].getValue();
        break;
      }
    }
    return fh;
  }

 /** 
 *  Return the GroupID if any, else return null; 
 **/ 
  public String
  getGID(){
    String fh = null;

    for(int i = 0; i < this.fieldValues.length; i++){
      if(this.fieldValues[i].getFieldName().equals("gid")){
        fh = this.fieldValues[i].getValue();
        break;
      }
    }
    return fh;
  }
 /** 
 *  Clone this object 
 **/ 
  public Object
  clone(){
    // Don't currently support cloning...
    assert(false);
    return(null);
  }

 /** 
 *  Return a hashCode (note: doesn't currently do anything) 
 **/ 
  public int
  hashCode(){
    assert(false);
    return(0);
  }

 /** 
 *  Used for testing 
 **/ 
  public static void
  main(String[] argv){
    if(argv.length < 1){
      Env.verifyAssertEnabled();
      System.out.println("Testing TraceEntry.java...");
      HarvardTraceEntry.test1();
      HarvardTraceEntry.test2();
      System.out.println("...Finished");
    }else{
      HarvardTraceEntry.readLoopTest();
    }
  }

 /** 
 *  Test HarvardTraceEntry 
 **/ 
  private static void
  test1(){
    HarvardTraceEntry hte1 = null;
    HarvardTraceEntry hte2 = null;
    ByteArrayOutputStream baos = null;
    ByteArrayInputStream bais = null;
    BufferedReader br = null;
    long timeMS = 0;
    FieldValuePair[] fieldValues = null;

    try{
      baos = new ByteArrayOutputStream();
      fieldValues = new FieldValuePair[1];
      fieldValues[0] = new FieldValuePair("fh", "somefilehandle448");
      hte1 = new HarvardTraceEntry(268L,
                                   305L,
                                   0x408,
                                   0x409,
                                   0x5ac,
                                   0x5ad,
                                   (short)TCP_PROTOCOL,
                                   (short)NFS_V3_RESPONSE,
                                   (long)0x4566abcd,
                                   (short)8,
                                   "somemethod",
                                   2,
                                   fieldValues);
      hte1.writeToOS(baos);
      assert(hte1.equals(hte1));
      //hte1.writeToOS(System.out); // Already tested
      
      assert hte1.getFileHandle().equals("somefilehandle448");

      bais = new ByteArrayInputStream(baos.toByteArray());
      br = new BufferedReader(new InputStreamReader(bais));
      hte2 = new HarvardTraceEntry(br);
      //hte2.writeToOS(System.out); // Already tested
      try{
        hte2 = new HarvardTraceEntry(br);
        assert(false);
      }catch(EOFException e){
        // Do nothing; this was expected.
      }
      assert(hte1 != hte2);
      assert(hte1.equals(hte2));
      
    }catch(IOException e){
      assert(false);
    }catch(InvalidHarvardTraceEntryException e){
      System.err.println("" + e);
      assert(false);
    }
  }

 /** 
 *  Test TraceEntry 2 
 **/ 
  private static void
  test2(){
    HarvardTraceEntry hte = null;
    FieldValuePair[] fieldValues = null;

    fieldValues = new FieldValuePair[2];
    fieldValues[0] = new FieldValuePair("fh", "somefilehandle448");
    fieldValues[1] = new FieldValuePair("fh2", "somenewfilehandle884");
    hte = new HarvardTraceEntry(4765L,
                                68829L,
                                0x804,
                                0x904,
                                0xca5,
                                0xda5,
                                (short)UDP_PROTOCOL,
                                (short)NFS_V2_RESPONSE,
                                (long)0xabcd4566,
                                (short)15,
                                "someothermethod",
                                80,
                                fieldValues);

    assert(hte.getTimeSec() == hte.timeSec);
    assert(hte.getTimeUSec() == hte.timeUSec);
    assert(hte.getSourceAddr() == hte.sourceAddr);
    assert(hte.getSourcePort() == hte.sourcePort);
    assert(hte.getDestinationAddr() == hte.destinationAddr);
    assert(hte.getDestinationPort() == hte.destinationPort);
    assert(hte.getInetProtocol() == hte.inetProtocol);
    assert(hte.getNFSProtocol() == hte.nfsProtocol);
    assert(hte.getXID() == hte.xid);
    assert(hte.getMethodNum() == hte.methodNum);
    assert(hte.getMethodName().equals(hte.methodName));
    assert(hte.getResponseCode() == hte.responseCode);
    assert(hte.getFieldValues() == hte.fieldValues);
  }

 /** 
 *  Try reading a file from stdin 
 **/ 
  private static void
  readLoopTest(){
    HarvardTraceEntry hte = null;
    BufferedReader reader = null;

    try{
      reader = new BufferedReader(new InputStreamReader(System.in));
      while(true){
        hte = new HarvardTraceEntry(reader);
        hte.writeToOS(System.out);
      }
    }catch(EOFException e){
      System.out.println("Done!");
    }catch(InvalidHarvardTraceEntryException e){
      e.printStackTrace();
      System.out.println("" + e);
      assert(false);
    }catch(IOException e){
      System.out.println("" + e);
      assert(false);
    }
  }

 /** 
 *  Convert to a String with a decimal number padded with 0's 
 **/ 
  private static String
  getDecimalString(long value, int numDigits){
    String valString = null;
    String prefixString = "";

    valString = "" + value;
    assert(valString.length() <= numDigits);
    for(int i = 0; i < (numDigits - valString.length()); i++){
      prefixString = prefixString + '0';
    }
    return(prefixString + valString);
  }

 /** 
 *  Convert to a String with a hexadecimal number padded with 0's 
 **/ 
  private static String
  getHexString(long value, int numDigits){
    String valString = null;
    String prefixString = "";

    valString = "" + Long.toHexString(value);
    assert(valString.length() <= numDigits);
    for(int i = 0; i < (numDigits - valString.length()); i++){
      prefixString = prefixString + '0';
    }
    return(prefixString + valString);
  }

 /** 
 *  Return a String representation of the Internet protocol used 
 **/ 
  private String
  getInetProtocolString(){
    String inetProtocolString = null;

    if(this.inetProtocol == TCP_PROTOCOL){
      inetProtocolString = "T";
    }else if(this.inetProtocol == UDP_PROTOCOL){
      inetProtocolString = "U";
    }
    return(inetProtocolString);
  }

 /** 
 *  Return a String representation of the NFS RPC protocol used 
 **/ 
  private String
  getNFSProtocolString(){
    String nfsProtocolString = null;

    if(this.nfsProtocol == NFS_V3_CALL){
      nfsProtocolString = "C3";
    }else if(this.nfsProtocol == NFS_V3_RESPONSE){
      nfsProtocolString = "R3";
    }else if(this.nfsProtocol == NFS_V2_CALL){
      nfsProtocolString = "C2";
    }else if(this.nfsProtocol == NFS_V2_RESPONSE){
      nfsProtocolString = "R2";
    }
    return(nfsProtocolString);
  }

 /** 
 *  Return a "clean" line, which means that we remove extraneous tokens. 
 **/ 
  private static String
  cleanupLine(String line){
    String newLine = null;

    for(int i = 0; i < REMOVE_TOKENS.length; i++){
      if(line.endsWith(REMOVE_TOKENS[i])){
        newLine = line.substring(0, line.length() - REMOVE_TOKENS[i].length());
      }
    }
    return(newLine);
  }

 /** 
 *  Parse the event time 
 **/ 
  private void
  parseTime(StringTokenizer st)
    throws IOException, InvalidHarvardTraceEntryException{
    String token = null;
    int periodIndex = 0;
    String timeSecStr = null;
    String timeUSecStr = null;

    if(!st.hasMoreTokens()){
      throw(new InvalidHarvardTraceEntryException());
    }
    token = st.nextToken();
    periodIndex = token.indexOf('.');
    if(periodIndex < 0){
      throw(new InvalidHarvardTraceEntryException());
    }
    timeSecStr = token.substring(0, periodIndex);
    timeUSecStr = token.substring(periodIndex + 1, token.length());
    try{
      this.timeSec = Long.parseLong(timeSecStr);
      this.timeUSec = Long.parseLong(timeUSecStr);
    }catch(NumberFormatException nfe){
      throw(new InvalidHarvardTraceEntryException());
    }
  }

 /** 
 *  Parse the source address and port numbers 
 **/ 
  private void
  parseSourceAddress(StringTokenizer st)
    throws IOException, InvalidHarvardTraceEntryException{
    String token = null;
    int periodIndex = 0;
    String dnsHexStr = null;
    String portHexStr = null;

    if(!st.hasMoreTokens()){
      throw(new InvalidHarvardTraceEntryException());
    }
    token = st.nextToken();
    periodIndex = token.indexOf('.');
    if(periodIndex < 0){
      throw(new InvalidHarvardTraceEntryException());
    }
    dnsHexStr = token.substring(0, periodIndex);
    portHexStr = token.substring(periodIndex + 1, token.length());
    try{
      this.sourceAddr = (Integer.decode("0x" + dnsHexStr)).intValue();
      this.sourcePort = (Integer.decode("0x" + portHexStr)).intValue();
    }catch(NumberFormatException nfe){
      throw(new InvalidHarvardTraceEntryException());
    }
  }


 /** 
 *  Parse the destination address and port numbers 
 **/ 
  private void
  parseDestinationAddress(StringTokenizer st)
    throws IOException, InvalidHarvardTraceEntryException{
    String token = null;
    int periodIndex = 0;
    String dnsHexStr = null;
    String portHexStr = null;

    if(!st.hasMoreTokens()){
      throw(new InvalidHarvardTraceEntryException());
    }
    token = st.nextToken();
    periodIndex = token.indexOf('.');
    if(periodIndex < 0){
      throw(new InvalidHarvardTraceEntryException());
    }
    dnsHexStr = token.substring(0, periodIndex);
    portHexStr = token.substring(periodIndex + 1, token.length());
    try{
      this.destinationAddr = (Integer.decode("0x" + dnsHexStr)).intValue();
      this.destinationPort = (Integer.decode("0x" + portHexStr)).intValue();
    }catch(NumberFormatException nfe){
      throw(new InvalidHarvardTraceEntryException());
    }
  }

 /** 
 *  Parse the Internet protocol used (TCP or UDP only) 
 **/ 
  private void
  parseInetProtocol(StringTokenizer st)
    throws IOException, InvalidHarvardTraceEntryException{
    String token = null;

    if(!st.hasMoreTokens()){
      throw(new InvalidHarvardTraceEntryException());
    }
    token = st.nextToken();
    if(token.equalsIgnoreCase("U")){
      this.inetProtocol = UDP_PROTOCOL;
    }else if(token.equalsIgnoreCase("T")){
      this.inetProtocol = TCP_PROTOCOL;
    }else{
      throw(new InvalidHarvardTraceEntryException());
    }
  }

 /** 
 *  Parse the NFS RPC protocol and direction (C3 = nfs v3 call, R3 = 
 *       nfs v3 response, C2 = nfs v2 call, R2 = nfs v2 response) 
 **/ 
  private void
  parseNFSProtocol(StringTokenizer st)
    throws IOException, InvalidHarvardTraceEntryException{
    String token = null;

    if(!st.hasMoreTokens()){
      throw(new InvalidHarvardTraceEntryException());
    }
    token = st.nextToken();
    if(token.equalsIgnoreCase("C3")){
      this.nfsProtocol = NFS_V3_CALL;
    }else if(token.equalsIgnoreCase("R3")){
      this.nfsProtocol = NFS_V3_RESPONSE;
    }else if(token.equalsIgnoreCase("C2")){
      this.nfsProtocol = NFS_V2_CALL;
    }else if(token.equalsIgnoreCase("R2")){
      this.nfsProtocol = NFS_V2_RESPONSE;
    }else{
      throw(new InvalidHarvardTraceEntryException());
    }
  }

 /** 
 *  Parse the RPC transaction ID used by clients to match calls to responses 
 **/ 
  private void
  parseXID(StringTokenizer st)
    throws IOException, InvalidHarvardTraceEntryException{
    String token = null;

    if(!st.hasMoreTokens()){
      throw(new InvalidHarvardTraceEntryException());
    }
    token = st.nextToken();
    try{
      this.xid = (Long.decode("0x" + token)).longValue();
    }catch(NumberFormatException e){
      System.err.println("" + e);
      throw(new InvalidHarvardTraceEntryException());
    }
  }

 /** 
 *  Parse the RPC method number 
 **/ 
  private void
  parseMethodNum(StringTokenizer st)
    throws IOException, InvalidHarvardTraceEntryException{
    String token = null;

    if(!st.hasMoreTokens()){
      throw(new InvalidHarvardTraceEntryException());
    }
    token = st.nextToken();
    try{
      this.methodNum = (Short.decode("0x" + token)).shortValue();
    }catch(NumberFormatException e){
      throw(new InvalidHarvardTraceEntryException());
    }
  }

 /** 
 *  Parse the RPC method name 
 **/ 
  private void
  parseMethodName(StringTokenizer st)
    throws IOException, InvalidHarvardTraceEntryException{

    if(!st.hasMoreTokens()){
      throw(new InvalidHarvardTraceEntryException());
    }
    this.methodName = st.nextToken();
  }

 /** 
 *  Parse the RPC response (if this entry is for a response) 
 **/ 
  private void
  parseResponseCode(StringTokenizer st)
    throws IOException, InvalidHarvardTraceEntryException{
    String token = null;

    if(!st.hasMoreTokens()){
      throw(new InvalidHarvardTraceEntryException());
    }
    token = st.nextToken();
    if(token.equalsIgnoreCase("OK")){
      this.responseCode = 0;
    }else{
      try{
        this.responseCode = (Integer.decode("0x" + token)).intValue();
      }catch(NumberFormatException e){
        throw(new InvalidHarvardTraceEntryException());
      }
    }
  }

 /** 
 *  Parse the field-name/value pairs with the last 
 *  "numSkipLastTokens" tokens removed 
 **/ 
  private void
  parseFieldValuePairs(StringTokenizer st, int numSkipLastTokens)
    throws IOException, InvalidHarvardTraceEntryException{
    String token = null;
    int numTokens = 0;
    int numPairs = 0;
    String fieldName = null;
    String value = null;
    FieldValuePair fvp = null;

    numTokens = st.countTokens();
    if((numTokens < numSkipLastTokens) ||
       (((numTokens - numSkipLastTokens) % 2) != 0)){
      throw(new InvalidHarvardTraceEntryException());
    }

    numPairs = (numTokens - numSkipLastTokens) / 2;
    this.fieldValues = new FieldValuePair[numPairs];
    for(int i = 0; i < numPairs; i++){
      fieldName = st.nextToken();
      value = st.nextToken();
      this.fieldValues[i] = new FieldValuePair(fieldName, value);
    }
  }

 /** 
 *  Return true if this entry refers to an RPC response 
 **/ 
  public boolean
  isResponse(){
    return((this.nfsProtocol == NFS_V2_RESPONSE) ||
           (this.nfsProtocol == NFS_V3_RESPONSE));
  }

 /** 
 *  Return true if the field values of "this" equals the passed-in ones 
 **/ 
  private boolean
  compareFieldValues(FieldValuePair[] hteFieldValues){
    boolean result = true;
    boolean found = true;

    if(this.fieldValues.length == hteFieldValues.length){
      result = true;
      for(int i = 0; (i < this.fieldValues.length) && result; i++){
        // Find the i'th entry of this.fieldValues in hteFieldValues
        found = false;
        for(int j = 0; (j < this.fieldValues.length) && (!found); j++){
          found = this.fieldValues[i].equals(hteFieldValues[j]);
        }
        if(found){
          // Find the i'th entry of hteFieldValues in this.fieldValues
          found = false;
          for(int j = 0; (j < this.fieldValues.length) && (!found); j++){
            found = hteFieldValues[i].equals(this.fieldValues[j]);
          }
        }
        result = found;
      }
    }else{
      result = false;
    }
    return(result);
  }

 /** 
 *  Internal class used to store field names and values 
 **/ 
  public static class FieldValuePair{

 /** 
 *  Constructor 
 **/ 
    public
    FieldValuePair(String newFieldName, String newValue){
      this.fieldName = newFieldName;
      this.value = newValue;
    }

 /** 
 *  Return true if "o" equals "this" 
 **/ 
    public boolean
    equals(Object o){
      FieldValuePair fvp = null;
      boolean result = false;

      // 1 Java-provided variable + 2 fields
      assert(this.getClass().getDeclaredFields().length == 3);
      if(o instanceof FieldValuePair){
        fvp = (FieldValuePair)o;
        result = (this.fieldName.equals(fvp.fieldName) &&
                  this.value.equals(fvp.value));
      }else{
        result = false;
      }
      return(result);
    }

 /** 
 *  Return a hashCode for this object 
 **/ 
    public int
    hashCode(){
      assert(false);
      return(0);
    }

 /** 
 *  Return the field name 
 **/ 
    public String
    getFieldName(){
      return(this.fieldName);
    }

 /** 
 *  Return the value 
 **/ 
    public String
    getValue(){
      return(this.value);
    }

 /** 
 *  Data members 
 **/ 
    private String fieldName;
    private String value;
  }

 /** 
 *  Data members 
 **/ 
  private long timeSec;
  private long timeUSec;
  private int sourceAddr;
  private int sourcePort;
  private int destinationAddr;
  private int destinationPort;
  private short inetProtocol;
  private short nfsProtocol;
  private long xid;
  private short methodNum;
  private String methodName;
  private int responseCode;
  private FieldValuePair[] fieldValues;
}

 /** 
 *  Exception that is thrown when we read an inval entry in our trace file 
 **/ 
class InvalidHarvardTraceEntryException extends Exception{

 /** 
 *  Constructor 
 **/ 
  public
  InvalidHarvardTraceEntryException(String newMsg){
    this.msg = newMsg;
  }

 /** 
 *  Constructor 
 **/ 
  public
  InvalidHarvardTraceEntryException(){
    this.msg = "Invalid entry";
  }

 /** 
 *  Convert this exception to a string 
 **/ 
  public String
  toString(){
    return(this.msg);
  }

 /** 
 *  Data members 
 **/ 
  private String msg;
}
