 /** 
 *  Code to preprocess an entry from the Harvard trace files into 
 *   <<destNodeId ObjectId OperationType(W|R) Time Offset Length>> 
 *  It includes two phases:  
 *    - assign integer id to file handles 
 *         store id touched by uid in one table1 
 *         store id not touched by uid in another table2) 
 *    - assign prefix "/u/" for obj in table1 or "/o/" to obj in talbe2 
 *  
 *  Note: This class takes the first writer in the trace as the interested user 
 *        As each "call" record does not have a userId in it, 
 *        in the first phase, we just track the file handles in the  
 *        "Response" Records. 
 **/ 
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.Reader;
import java.io.EOFException;
import java.io.PrintStream;
import java.io.OutputStream;
import java.util.StringTokenizer;
import java.util.Hashtable;
import java.util.Enumeration;
import java.util.HashMap;

public class HarvardTracePreprocesser{
  int fileIdCounter;
  int nodeIdCounter;
  Hashtable t1;//raw filehandle -> integer id
  Hashtable t2;
  Hashtable nodes;
  Hashtable users;
  String rawFileName;
  String interestedUID;
  int nodeId = -1;
  HashMap readCountTable;
  HashMap writeCountTable;
  
  private static final String PRE1 = "/u/";
  private static final String PRE2 = "/o/";
  private static final int READ = TraceEntry.READ;
  private static final int WRITE = TraceEntry.WRITE;
  private static final String RESULTFILE_SUFFEX = ".practi.trc";
  
 /** 
 *  Constructor 
 **/ 
  public HarvardTracePreprocesser(String rawFileName, String iuid, 
                                  String nodeIdHex){
    fileIdCounter = 0;
    nodeIdCounter = 0;
    t1 = new Hashtable();
    t2 = new Hashtable();
    nodes = new Hashtable();
    users = new Hashtable();
    this.rawFileName = rawFileName;
    this.interestedUID=iuid;
    //System.out.println("input:" + nodeIdHex);
    this.nodeId = (Integer.decode("0x" + nodeIdHex)).intValue();
    //System.out.println("output:"+ this.nodeId);
    this.readCountTable= new HashMap();
    this.writeCountTable= new HashMap();
  }

 /** 
 *  Constructor 
 **/ 
  public HarvardTracePreprocesser(String rawFileName, String iuid){
    fileIdCounter = 0;
    nodeIdCounter = 0;
    t1 = new Hashtable();
    t2 = new Hashtable();
    nodes = new Hashtable();
    users = new Hashtable();
    this.rawFileName = rawFileName;
    this.interestedUID=iuid;
    nodeId = -1;
    this.readCountTable= new HashMap();
    this.writeCountTable= new HashMap();
  }

 /** 
 *  Constructor 
 **/ 
  public HarvardTracePreprocesser(String rawFileName){
    fileIdCounter = 0;
    nodeIdCounter = 0;
    t1 = new Hashtable();
    t2 = new Hashtable();
    nodes = new Hashtable();
    users = new Hashtable();
    this.rawFileName = rawFileName;
    this.interestedUID=null;
    nodeId = -1;
  }
  
 /** 
 *  Phase 1 -- create table1 and table2 for objIds and nodes table  
 **/ 
  private void parse1(){
    boolean dbg = true;
    boolean dbgVerbose = false;
    HarvardTraceEntry hte = null;
    BufferedReader br = null;
    
    String firstUID = interestedUID;
    String cuid = null;
    String cgid = null;
    String cfh = null;
    String cnode = null;
    if(dbg){
      System.err.println("parse1 start...");
    }
    try{
      br = new BufferedReader(new FileReader(rawFileName));
    }catch (Exception e){ 
      e.printStackTrace();
      assert(false);
    }
    assert br != null;

    long linecount = 0;

    try{
      while(true){
        hte = new HarvardTraceEntry(br);
        if(dbgVerbose){
          System.err.println("fid:" + hte.getFileId());
          System.err.println("off:" + hte.getOffset());
          System.err.println("len:" + hte.getLen());
          System.err.println("UID:" + hte.getUID());
          System.err.println("GID:" + hte.getGID());
        }
        assert hte != null;
        cnode = ""+hte.getSourceAddr();
        if((cnode!=null)&&(!nodes.containsKey(cnode))){
          nodes.put(cnode, new Integer(nodeIdCounter));
          nodeIdCounter++;
        }

        cuid = hte.getUID();
        if(cuid != null){
          cgid = hte.getGID();
          assert cgid != null;
          //cuid += cgid; //cat uid with gid to get unique user id
          if(firstUID==null){
            firstUID = cuid;
          }
        }
        
        cfh = hte.getFileId();
        if(dbgVerbose){
          hte.writeToOS(System.out);
          System.out.println("fileId: " +cfh);
        }
        Integer thisFileId = null;
        linecount ++;
        
        if(cfh != null){
          /*
          System.out.println("fid:" + hte.getFileId());
          System.out.println("filename = " + cfh);
          System.out.println("cuid = " + cuid + "|" + cuid.length());
          System.out.println("firstuid = " + firstUID + "|" + firstUID.length());
          */
          if(!t1.containsKey(cfh)){//not in table1
            //System.out.println("not in table 1: " + cfh);
            if((firstUID!=null)&&
               (cuid!=null)&&
               (cuid.equals(firstUID))){//interested file
              //System.out.println("Care about this file: " + cfh);
              if(t2.containsKey(cfh)){
                //System.out.println("Removed from t2: " + cfh);
                Integer value = (Integer)(t2.remove(cfh));
                assert value!= null;
                thisFileId = value;
                t1.put(cfh, value);
              }else{
                //System.out.println("Adding to t1: " + cfh);
                t1.put(cfh, new Integer(fileIdCounter));
                thisFileId = new Integer(fileIdCounter);
                fileIdCounter ++;
              }
            }else if(!t2.containsKey(cfh)){
              //System.out.println("Adding to t2: " + cfh);
              t2.put(cfh, new Integer(fileIdCounter));
              thisFileId = new Integer(fileIdCounter);
              fileIdCounter ++;
            }//do nothing if it's not interested and already in t2
              
          }//do nothing if already in t1 
        }
        Integer count = (Integer)this.readCountTable.get(thisFileId);
        if(count == null){
          if(hte.getMethodName().equalsIgnoreCase("read")){
            this.readCountTable.put(thisFileId, new Integer(1));
            this.writeCountTable.put(thisFileId, new Integer(0));
          }else if(hte.getMethodName().equalsIgnoreCase("write")){
            this.readCountTable.put(thisFileId, new Integer(0));
            this.writeCountTable.put(thisFileId, new Integer(1));
          }
        }else{
          if(hte.getMethodName().equalsIgnoreCase("read")){          
            count = new Integer(count.intValue() + 1);
            this.readCountTable.put(thisFileId, count);
          }else{
            count = (Integer)this.writeCountTable.get(thisFileId);
            assert(count != null);
            count = new Integer(count.intValue() + 1);
            this.writeCountTable.put(thisFileId, count);
          }
        }
      }
     
    }catch(EOFException eof){
      //end of file
      //ok
      try{
        if(br!= null){
          br.close();
        }
      }catch(Exception e){
        assert false;
      }
    }catch(IOException ioe){
      assert false;
    }catch(InvalidHarvardTraceEntryException i){
      System.err.println(i);
      assert false;
    }
    if(dbg){
      System.err.println("parse1 finish...");
      System.err.println("total filenum: " + fileIdCounter);
      System.err.println("total nodeNum: " + nodeIdCounter);
      System.err.println("total interested files: " 
                         + t1.size());
      System.err.println("total other files: " + t2.size());
      System.err.println("fileIdCounter:"+fileIdCounter);
    }
    assert t1.size()+t2.size()==fileIdCounter;
  }

 /** 
 *  phase 0 -- find median-workload-userid to track of  
 **/ 
  public void parse0(){
    boolean dbg = true;
    boolean dbgVerbose = false;
    HarvardTraceEntry hte = null;
    BufferedReader br = null;
    
    String firstUID = null;
    String cuid = null;
    String cgid = null;
    String cfh = null;
    String cnode = null;
    if(dbg){
      System.err.println("parse0 start...");
    }
    try{
      br = new BufferedReader(new FileReader(rawFileName));
    }catch (Exception e){ 
      e.printStackTrace();
      assert(false);
    }
    assert br != null;
    
    try{
      while(true){
        hte = new HarvardTraceEntry(br);
       
        assert hte != null;
        
        if((nodeId != -1) && (hte.getSourceAddr() != nodeId)){
          continue;//specified only get interested node requests but it isn't
          //ignore;
        }

        cuid = hte.getUID();
        if(cuid != null){
          cgid = hte.getGID();
          assert cgid != null;
          //cuid += cgid; //cat uid with gid to get unique user id
        }
        
        cfh = hte.getFileId();
        if(dbgVerbose){
          hte.writeToOS(System.out);
          System.out.println("fileHandle: " +cfh);
        }
        if(cfh != null){
          if(!users.containsKey(cuid)){
            users.put(cuid, new Integer(1));
          }else{
            int oldValue = ((Integer)(users.get(cuid))).intValue();
            users.remove(cuid);
            
            users.put(cuid, new Integer(oldValue+1));
          }
        }
        
      }
     
    }catch(EOFException eof){
      //end of file
      //ok
      try{
        if(br!= null){
          br.close();
        }
      }catch(Exception e){
        assert false;
      }
    }catch(IOException ioe){
      assert false;
    }catch(InvalidHarvardTraceEntryException i){
      System.err.println(i);
      assert false;
    }

    for(Enumeration e = users.keys(); e.hasMoreElements();){
      String uid = (String)e.nextElement();
      int count = ((Integer)users.get(uid)).intValue();
      System.out.println("uid: " +uid + " count: " + count);
      
    }
    
    if(dbg){
      System.err.println("total gids"+ users.size());
      System.err.println("parse0 finish...");
    }

    
  }

 /** 
 *  Phase 2 -- create table1 and table2 for objIds 
 **/ 
  public BufferedReader parse2(){
    boolean dbg = true;
    boolean dbgVerbose = false;
    HarvardTraceEntry hte = null;
    BufferedReader br = null;
    BufferedWriter bw = null;
    String resultFileName = rawFileName+RESULTFILE_SUFFEX;
    String cfh = null;

    int newNodeId = -1;
    String newObjId = null;
    int newOperation = -1;
    long newTimeSec = 0;
    long newTimeUSec = 0;
    long newOffset = 0;
    long newLen = 0; 
    long durationSec = 0;
    long durationUSec = 0;
    long startSec = 0;
    long startUSec = 0;
    long endSec = 0;
    long endUSec = 0;

    long nWrites = 0;
    long nReads = 0;
     if(dbg){
      System.err.println("parse2 start...");
    }
    this.parse1();//generate the file tables
    try{
      File f = new File(resultFileName);
      if(f.exists()){
        boolean del = f.delete();
        assert del == true;
      }
    }catch (Exception e){ 
      e.printStackTrace();
      assert(false);
    }
    try{
      br = new BufferedReader(new FileReader(rawFileName));
      bw = new BufferedWriter(new FileWriter(resultFileName, 
                                             true));
    }catch (Exception e){ 
      e.printStackTrace();
      assert(false);
    }
    assert br != null;
    
    try{
      while(true){
        hte = new HarvardTraceEntry(br);
        
	if(startSec == 0){
          startSec = hte.getTimeSec();
	  startUSec = hte.getTimeUSec();
	}else{
          endSec = hte.getTimeSec();
	  endUSec = hte.getTimeUSec();
	} 
        
        //System.err.println("dest:" + hte.getSourceAddr());
        if((nodeId != -1) && (hte.getSourceAddr() != nodeId)){
          continue;//specified only get interested node requests but it isn't
          //ignore;
        }

        if(!hte.isResponse()){
          continue;
        }

        assert t1.size()+t2.size()==fileIdCounter;
        String op = hte.getMethodName();
        
        if((op.equals("write"))||(op.equals("read"))){
          double priority = 0.0;
          double numReads = 0;
          double numWrites = 0;

          //get nodeId
          String nodeId = ""+hte.getSourceAddr();
          newNodeId = ((Integer)(nodes.get(nodeId))).intValue();
          
          //get fileId
          cfh = hte.getFileId();
          assert cfh != null;
          if(t2.containsKey(cfh)){
            
            newObjId = PRE2 + t2.get(cfh); 
            assert !t1.containsKey(cfh);
          }else{
            assert t1.containsKey(cfh);
            newObjId = PRE1 + t1.get(cfh);
            
            numReads =
              ((Integer)this.readCountTable.get(t1.get(cfh))).intValue();
            numWrites =
              ((Integer)this.writeCountTable.get(t1.get(cfh))).intValue();
            if(numWrites > 0){
              priority = numReads / numWrites;
            }else{
              assert(numReads > 0);
              priority = 1E20;
            }
          }
          
          //set operation
          if(op.equals("write")){
            newOperation = WRITE;
            nWrites++;
          }else{
            newOperation = READ;
            nReads++;
          }

          //set time
          newTimeSec = hte.getTimeSec();
          newTimeUSec = hte.getTimeUSec();
          
          //set offset -- for simplification: just set it to 0
          //newOffset = hte.getOffset();
          newOffset = 0;
          assert newOffset >-1;
          
          //set len
          newLen = hte.getLen();
          assert newLen > -1;
          String s = newNodeId 
            + " " + newObjId
            + " " + ((newOperation==WRITE)?"WRITE":"READ")
            + " " + newTimeSec
            + " " + newTimeUSec
            + " " + newOffset
            + " " + newLen
            + " " + priority
            +"\n";
          if(dbgVerbose){
            System.err.println(s);
          }
          bw.write(s, 0, s.length());
          bw.flush();
        }
      }
     
    }catch(EOFException eof){
      try{
        //end of file
        if(br != null){
          br.close();
          br = new BufferedReader(new FileReader(resultFileName));
        }
        if(bw != null){
          bw.close();
        }
      }catch(Exception e){
        assert false;
      }
    }catch(IOException ioe){
      assert false;
    }catch(InvalidHarvardTraceEntryException i){
      System.err.println(i);
      assert false;
    }
    
    assert br != null;
    durationSec = endSec - startSec;
    //durationUSec = endUSec - startUSec;
    //System.err.println("startSec:"+startSec);
    //System.err.println("endSec:"+endSec);
    assert durationSec >0;
    if(dbg){
      System.err.println("parse2 finish...");
      System.err.println("total writenum: " + nWrites);
      System.err.println("total readNum: " + nReads);
      System.err.println("totalDuration: " + durationSec + " seconds");
    }
    return br;
  }

  public static TraceEntry getEntry(BufferedReader br)
    throws IOException, InvalidTraceEntryException{
    TraceEntry ret = null;
    try{
      ret = new TraceEntry(br);
      assert ret != null;
    }catch(EOFException eof){
      ret = null;
    }
    return ret;
  }
  
  public static void main(String[] argv)
    throws Exception{
    if(argv.length <1){
      System.err.println("usage: [rawFileName]| [rawFileName] [userid] [nodeId]");
      System.exit(-1);
    }
    HarvardTracePreprocesser htp = null;
    if(argv.length ==1){
      htp = new HarvardTracePreprocesser(argv[0]);
    }else if(argv.length == 2){
      htp = new HarvardTracePreprocesser(argv[0], 
                                         argv[1]);
    }else if(argv.length == 3){
      htp = new HarvardTracePreprocesser(argv[0], 
                                         argv[1],
                                         argv[2]);
    }
    //htp.parse0();
    BufferedReader br = htp.parse2();
    
    System.out.println(htp.getEntry(br));
    System.out.println(htp.getEntry(br));
    br.close();
  }
}
