 /** 
 *  Code to parse/store an entry in trace files used for the Nice experiments. 
 **/ 
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.BufferedReader;
import java.io.Reader;
import java.io.EOFException;
import java.io.PrintStream;
import java.io.OutputStream;
import java.util.StringTokenizer;

// Used for testing
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;

public class TraceEntry implements Immutable, Cloneable{

 /** 
 *  Constants 
 **/ 
  public static final int READ = 1;
  public static final int WRITE = 0;

 /** 
 *  Constructor 
 **/ 
  public
  TraceEntry(NodeId newNodeId,
             ObjId newObjId,
             int newOperation,
             long newTimeSec,
             long newTimeUSec,
             long newOffset,
             long newLength,
             double newPriority){
    assert((newOperation == READ) || (newOperation == WRITE));
    this.nodeId = (NodeId)newNodeId.clone();
    this.objId = (ObjId)newObjId.clone();
    this.operation = newOperation;
    this.timeSec = newTimeSec;
    this.timeUSec = newTimeUSec;
    this.offset = newOffset;
    this.length = newLength;
    this.priority = newPriority;
  }

 /** 
 *  Constructor 
 **/ 
  public
  TraceEntry(BufferedReader br) throws IOException, InvalidTraceEntryException{
    String line = null;
    StringTokenizer st = null;

    line = br.readLine();
    if(line == null){
      throw(new EOFException());
    }
    st = new StringTokenizer(line);
    /*
    this.nodeId = (NodeId)newNodeId.clone();
    this.objId = (ObjId)newObjId.clone();
    this.operation = newOperation;
    this.timeSec = newTimeSec;
    this.timeUSec = newTimeUSec;
    this.offset = newOffset;
    this.length = newLength;
     */
    this.nodeId = TraceEntry.parseNodeId(st);
    this.objId = TraceEntry.parseObjId(st);
    this.operation = TraceEntry.parseOperationType(st);
    this.timeSec = TraceEntry.parseEventTimeSec(st);
    this.timeUSec = TraceEntry.parseEventTimeUSec(st);
    this.offset = TraceEntry.parseEventOffset(st);
    this.length = TraceEntry.parseEventLength(st);
    this.priority = TraceEntry.parseEventPriority(st);
  }

 /** 
 *  Write this object to an output stream 
 **/ 
  public void
  writeToOS(OutputStream os) throws IOException{
    PrintStream ps = null;

    assert((this.operation == TraceEntry.READ) ||
           (this.operation == TraceEntry.WRITE));
    ps = new PrintStream(os);
    ps.print("" + this.nodeId);
    ps.print("\t" + this.objId.getPath());
    if(this.operation == TraceEntry.READ){
      ps.print("\tREAD");
    }else{
      ps.print("\tWRITE");
    }
    ps.print("\t" + this.timeSec);
    ps.print("\t" + this.timeUSec);
    ps.print("\t" + this.offset);
    ps.print("\t" + this.length);
    ps.print("\t" + this.priority);
    ps.println();
    ps.flush();
  }

 /** 
 *  Return the node Id from the event 
 **/ 
  public NodeId
  getNodeId(){
    return(this.nodeId);
  }

 /** 
 *  Return the object Id from the event 
 **/ 
  public ObjId
  getObjId(){
    return(this.objId);
  }

 /** 
 *  Return the operation type 
 **/ 
  public int
  getOperation(){
    return(this.operation);
  }

 /** 
 *  Return the time of the event 
 **/ 
  public long
  getTimeSec(){
    return(this.timeSec);
  }

 /** 
 *  Return the time of the event 
 **/ 
  public long
  getTimeUSec(){
    return(this.timeUSec);
  }

 /** 
 *  Return the offset 
 **/ 
  public long
  getOffset(){
    return(this.offset);
  }

 /** 
 *  Return the length 
 **/ 
  public long
  getLength(){
    return(this.length);
  }

 /** 
 *  Return the priority 
 **/ 
  public double
  getPriority(){
    return(this.priority);
  }

 /** 
 *  Return true if "o" equals this 
 **/ 
  public boolean
  equals(Object o){
    TraceEntry te = null;
    boolean result = false;

    // 2 constants, 2 java-provided fields, 7 declared fields
    assert(this.getClass().getDeclaredFields().length == 12);

    if(o instanceof TraceEntry){
      te = (TraceEntry)o;
      result = ((this.nodeId.equals(te.nodeId)) &&
                (this.objId.equals(te.objId)) &&
                (this.operation == te.operation) &&
                (this.timeSec == te.timeSec) &&
                (this.timeUSec == te.timeUSec) &&
                (this.offset == te.offset) &&
                (this.length == te.length) &&
                (this.priority == te.priority));
    }else{
      result = false;
    }
    return(result);
  }

 /** 
 *  Clone this object 
 **/ 
  public Object
  clone(){
    // Because this object is immutable, we do not need to do anything
    return(this);
  }

 /** 
 *  Return a hashCode (note: doesn't currently do anything) 
 **/ 
  public int
  hashCode(){
    assert(false);
    return(0);
  }

  public String toString(){
    String s = getNodeId() 
      + " " + getObjId()
      + " " + getOperation()
      + " " + getTimeSec()
      + " " + getTimeUSec()
      + " " + getOffset()
      + " " + getLength()
      + " " + getPriority();
    return s;
  }

 /** 
 *  Used for testing 
 **/ 
  public static void
  main(String[] argv){
    if(argv.length < 1){
      Env.verifyAssertEnabled();
      System.out.println("Testing TraceEntry.java...");
      TraceEntry.test1();
      TraceEntry.test2();
      System.out.println("...Finished");
    }else{
      TraceEntry.readLoopTest();
    }
  }

 /** 
 *  Test TraceEntry (NOTE: Currently never called) 
 **/ 
  private static void
  test0(){
    String line = null;
    StringTokenizer tokenizer = null;
    TraceEntry te = null;
    BufferedReader reader = null;

    try{
      reader = new BufferedReader(new InputStreamReader(System.in));
      while(true){
        line = reader.readLine();
        if(line == null){
          break;
        }
        tokenizer = new StringTokenizer(line);
        while(tokenizer.hasMoreTokens()){
          System.out.println("" + tokenizer.nextToken());
        }
      }
    }catch(IOException e){
      System.out.println("" + e);
      assert(false);
    }
  }

 /** 
 *  Test TraceEntry 
 **/ 
  private static void
  test1(){
    TraceEntry te1 = null;
    TraceEntry te2 = null;
    ByteArrayOutputStream baos = null;
    ByteArrayInputStream bais = null;
    BufferedReader br = null;

    try{
      baos = new ByteArrayOutputStream();
      te1 = new TraceEntry(new NodeId(1),
                           new ObjId("/0"),
                           READ,
                           128,
                           353,
                           1024,
                           2024,
                           0.3);
      te1.writeToOS(baos);
      assert(te1.equals(te1));
      //te1.writeToOS(System.out); // Already tested

      bais = new ByteArrayInputStream(baos.toByteArray());
      br = new BufferedReader(new InputStreamReader(bais));
      te2 = new TraceEntry(br);
      //te2.writeToOS(System.out); // Already tested
      try{
          te2 = new TraceEntry(br);
          assert(false);
      }catch(EOFException e){
        // Do nothing; this was expected.
      }
      assert(te1 != te2);
      assert(te1.equals(te2));
    }catch(IOException e){
      assert(false);
    }catch(InvalidTraceEntryException e){
      System.err.println("" + e);
      assert(false);
    }
  }

 /** 
 *  Test TraceEntry 2 
 **/ 
  private static void
  test2(){
    TraceEntry te = null;
    long timeMS = 0;

    te = new TraceEntry(new NodeId(3),
                        new ObjId("/2"),
                        READ,
                        1024,
                        353,
                        128,
                        2024,
                        0.1);
    assert(te.getNodeId().equals(te.nodeId));
    assert(te.getObjId().equals(te.objId));
    assert(te.getOperation() == te.operation);
    assert(te.getTimeSec() == te.timeSec);
    assert(te.getTimeUSec() == te.timeUSec);
    assert(te.getOffset() == te.offset);
    assert(te.getLength() == te.length);
    assert(te.getPriority() == te.priority);
  }

 /** 
 *  Try reading a file from stdin 
 **/ 
  private static void
  readLoopTest(){
    TraceEntry te = null;
    BufferedReader reader = null;

    try{
      reader = new BufferedReader(new InputStreamReader(System.in));
      while(true){
        te = new TraceEntry(reader);
        te.writeToOS(System.out);
      }
    }catch(EOFException e){
      System.out.println("Done!");
    }catch(InvalidTraceEntryException e){
      System.out.println("" + e);
      assert(false);
    }catch(IOException e){
      System.out.println("" + e);
      assert(false);
    }
  }

 /** 
 *  Parse the node ID 
 **/ 
  private static NodeId
  parseNodeId(StringTokenizer st)
    throws IOException, InvalidTraceEntryException{
    String token = null;
    long readNodeId = 0;
    boolean badNodeId = false;

    // Read the node id
    if(st.hasMoreTokens()){
      token = st.nextToken();
      try{
        readNodeId = Long.parseLong(token);
      }catch(NumberFormatException nfe){
        badNodeId = true;
      }
    }else{
      badNodeId = true;
    }
    if(badNodeId){
      throw(new InvalidTraceEntryException("First column should be " +
                                           "the node Id!"));
    }
    return(new NodeId(readNodeId));
  }

 /** 
 *  Parse the object ID 
 **/ 
  private static ObjId
  parseObjId(StringTokenizer st)
    throws IOException, InvalidTraceEntryException{
    String token = null;
    String objIdPath = null;
    ObjId readObjId = null;
    boolean badObjId = false;


    if(st.hasMoreTokens()){
      token = st.nextToken();
      /*
      if((token.startsWith("\"")) && (token.endsWith("\""))){
        objIdPath = token.substring(1, token.length() - 1);
        readObjId = new ObjId(objIdPath);
      }else{
        badObjId = true;
      }
      */
      objIdPath = token;
      readObjId = new ObjId(objIdPath);
      badObjId = false;
    }else{
      badObjId = true;
    }
    
    if(badObjId){
      throw(new InvalidTraceEntryException("Second column should be " +
                                           "the object ID"));
    }
    
    return(readObjId);
  }

 /** 
 *  Parse the operation type 
 **/ 
  private static int
  parseOperationType(StringTokenizer st)
    throws IOException, InvalidTraceEntryException{
    String token = null;
    int operationType = READ;
    boolean badOperation = false;

    if(st.hasMoreTokens()){
      token = st.nextToken();
      if(token.equalsIgnoreCase("read")){
        operationType = TraceEntry.READ;
      }else if(token.equalsIgnoreCase("write")){
        operationType = TraceEntry.WRITE;
      }else{
        badOperation = true;
      }
    }else{
      badOperation = true;
    }
    if(badOperation){
      throw(new InvalidTraceEntryException("Third column should either be " +
                                           "READ or WRITE!"));
    }
    return(operationType);
  }

 /** 
 *  Parse the event time (in seconds) 
 **/ 
  private static long
  parseEventTimeSec(StringTokenizer st)
    throws IOException, InvalidTraceEntryException{
    String token = null;
    long readEventTimeSec = 0;
    boolean badTimeSec = false;

    if(st.hasMoreTokens()){
      token = st.nextToken();
      try{
        readEventTimeSec = Long.parseLong(token);
      }catch(NumberFormatException nfe){
        badTimeSec = true;
      }
    }else{
      badTimeSec = true;
    }
    if(badTimeSec){
      throw(new InvalidTraceEntryException("Fourth column should be " +
                                           "the event time in sec!"));
    }
    return(readEventTimeSec);
  }

 /** 
 *  Parse the event time (micro-seconds component) 
 **/ 
  private static long
  parseEventTimeUSec(StringTokenizer st)
    throws IOException, InvalidTraceEntryException{
    String token = null;
    long readEventTimeUSec = 0;
    boolean badTimeUSec = false;

    if(st.hasMoreTokens()){
      token = st.nextToken();
      try{
        readEventTimeUSec = Long.parseLong(token);
      }catch(NumberFormatException nfe){
        badTimeUSec = true;
      }
    }else{
      badTimeUSec = true;
    }
    if(badTimeUSec){
      throw(new InvalidTraceEntryException("Fifth column should be " +
                                           "the event time u-sec component!"));
    }
    return(readEventTimeUSec);
  }

 /** 
 *  Parse the offset 
 **/ 
  private static long
  parseEventOffset(StringTokenizer st)
    throws IOException, InvalidTraceEntryException{
    String token = null;
    long readOffset = 0;
    boolean badOffset = false;

    if(st.hasMoreTokens()){
      token = st.nextToken();
      try{
        readOffset = Long.parseLong(token);
      }catch(NumberFormatException nfe){
        badOffset = true;
      }
    }else{
      badOffset = true;
    }
    if(badOffset){
      throw(new InvalidTraceEntryException("Sixth column should be " +
                                           "the offset!"));
    }
    return(readOffset);
  }

 /** 
 *  Parse the offset 
 **/ 
  private static long
  parseEventLength(StringTokenizer st)
    throws IOException, InvalidTraceEntryException{
    String token = null;
    long readLength = 0;
    boolean badLength = false;

    if(st.hasMoreTokens()){
      token = st.nextToken();
      try{
        readLength = Long.parseLong(token);
      }catch(NumberFormatException nfe){
        badLength = true;
      }
    }else{
      badLength = true;
    }
    if(badLength){
      throw(new InvalidTraceEntryException("Seventh column should be " +
                                           "the offset!"));
    }
    return(readLength);
  }

 /** 
 *  Parse the priority 
 **/ 
  private static double
  parseEventPriority(StringTokenizer st)
    throws IOException, InvalidTraceEntryException{
    String token = null;
    double readPriority = 0.0;
    boolean badPriority = false;

    if(st.hasMoreTokens()){
      token = st.nextToken();
      try{
        readPriority = Double.parseDouble(token);
      }catch(NumberFormatException nfe){
        badPriority = true;
      }
    }else{
      badPriority = true;
    }
    if(badPriority){
      throw(new InvalidTraceEntryException("Eighth column should be " +
                                           "the priority!"));
    }
    return(readPriority);
  }

 /** 
 *  Data members 
 **/ 
  private NodeId nodeId;
  private ObjId objId;
  private int operation;
  private long timeSec;
  private long timeUSec;
  private long offset;
  private long length;
  private double priority;
}

 /** 
 *  Exception that is thrown when we read an inval entry in our trace file 
 **/ 
class InvalidTraceEntryException extends Exception{

 /** 
 *  Constructor 
 **/ 
  public
  InvalidTraceEntryException(String newMsg){
    this.msg = newMsg;
  }

 /** 
 *  Convert this exception to a string 
 **/ 
  public String
  toString(){
    return(this.msg);
  }

 /** 
 *  Data members 
 **/ 
  private String msg;
}
