 /** 
 *  Receiver: 
 *  Subscribe to listen to invalidates from the sender and wait. 
 *  The controller starts two timers when the inval stream 
 *  is initiated, stops one when it has seen NUM_FILES invalidates, 
 *  and stops the other when it has seen NUM_FILES object bodies. 
 *  
 *  Sender: 
 *  Generate NUM_FILES writes of FILE_SIZE_BYTES byte files 
 **/ 
import java.io.IOException;

public class WriteRemoteExpt{

 /** 
 *  Private data 
 **/ 
  private static final int NUM_FILES = 1000;
  private static final int FILE_SIZE_BYTES = 1000;
  private static final int WAIT_SIGNAL_PORT = 4789;

 /** 
 *  Make a URANode object 
 **/ 
  private static URANode
  makeReceiverURANode(String configFile,
                      RMIClient rmiClient,
                      Controller controller,
                      NodeId receiverNodeId){
    Core core = null;
    URANode uraNode = null;

    Config.readConfig(configFile, receiverNodeId);
    core = new Core(rmiClient, true, true);
    uraNode = new URANode(core, controller, rmiClient);
    return(uraNode);
  }

 /** 
 *  Subscribe to listen to invalidates/data from the sender and wait 
 **/ 
  private static void
  receiveData(NodeId senderNodeId, NodeId myNodeId, RMIClient rmiClient)
    throws IOException, RMINetworkException, RMIApplicationException{
    SubscriptionSet ss = null;
    CounterVV startVV = null;
    CounterVV endVV = null;

    ss = SubscriptionSet.makeSubscriptionSet(":/*");
    startVV = new CounterVV(CounterVV.makeVVAllNegatives());
    endVV = new CounterVV(CounterVV.makeVVAllNegatives());
    // Note that the stream will not send the invalidate whose timestamp
    // matches endVV - it will only send invalidates with strictly lower
    // timestamps.
    endVV.setStampByServer(senderNodeId, (long)NUM_FILES);
    rmiClient.subscribe(ss,
                        startVV,
                        endVV,
                        senderNodeId,
                        myNodeId,
                        Config.getDNS(myNodeId),
                        Config.getPortInval(myNodeId),
                        86400000, // 1 day
                        0,
                        0);
    rmiClient.subscribeBody(ss,
                            startVV,
                            senderNodeId,
                            myNodeId,
                            Config.getDNS(myNodeId),
                            Config.getPortNice(myNodeId),
                            86400000); // 1 day
  }

  //-------------------------------------------------------------------------
  // Wait to receive everything we expect
  //-------------------------------------------------------------------------
  private static void
  getTimes(RemoteExptController exptController){
    System.out.println("Start time = " +
                       exptController.getStartTimeMS() +
                       "ms");
    System.out.println("Inval end time = " +
                       exptController.getInvEndTimeMS() +
                       "ms");
    System.out.println("Body end time = " +
                       exptController.getObjEndTimeMS() +
                       "ms");
  }

  //-------------------------------------------------------------------------
  // Generate NUM_FILES writes of FILE_SIZE_BYTES byte files.
  // Another machine will remotely read these files
  //-------------------------------------------------------------------------
  private static void
  performWrites(URANode uraNode) throws IOException{
    LocalInterface li = null;
    ImmutableBytes ib = null;
    byte[] b = null;
    double printPct = 0.0;

    li = uraNode.getLocalInterface();
    b = new byte[FILE_SIZE_BYTES];
    ib = ImmutableBytes.dangerousCreateImmutableBytes(b);
    b = null; // Delete byte array reference

    for(int i = 0; i < NUM_FILES; i++){
      li.write(new ObjId("/" + i),
               0,
               FILE_SIZE_BYTES,
               Core.DEFAULT_PRIORITY,
               ib,
               false); // unbound invals
      if(((double)i / NUM_FILES) >= printPct){
        System.out.println("Finished: " + ((i * 100) / NUM_FILES) + "%");
        printPct += 0.1;
      }
    }
  }

  //-------------------------------------------------------------------------
  // Program staring point
  //-------------------------------------------------------------------------
  public static void
  main(String[] argv){
    int nodeIdInt = 0;
    NodeId senderNodeId = null;
    NodeId receiverNodeId = null;
    NodeId myNodeId = null;
    URANode uraNode = null;
    RMIClient rmiClient = null;
    LocalInterface localInterface = null;
    RemoteExptController exptController = null;

    if((argv.length == 4) && argv[1].equals("receiver")){
      senderNodeId = new NodeId(Integer.parseInt(argv[2]));
      myNodeId = new NodeId(Integer.parseInt(argv[3]));
      rmiClient = new RMIClient();
      exptController = new RemoteExptController(NUM_FILES);
      uraNode = WriteRemoteExpt.makeReceiverURANode(argv[0],
                                                    rmiClient,
                                                    exptController,
                                                    myNodeId);
      try{
        WriteRemoteExpt.receiveData(senderNodeId, myNodeId, rmiClient);
        WriteRemoteExpt.getTimes(exptController);
      }catch(IOException e){
        System.err.println("" + e);
      }catch(RMINetworkException e){
        e.printStackTrace();
        System.err.println("" + e);
      }catch(RMIApplicationException e){
        e.printStackTrace();
        System.err.println("" + e);
      }
    }else if((argv.length == 3) && argv[1].equals("sender")){
      // Start URA Node
      nodeIdInt = Integer.parseInt(argv[2]);
      myNodeId = new NodeId(nodeIdInt);
      uraNode = new URANode(argv[0],
                            myNodeId,
                            Controller.NULL_CONTROLLER,
                            true);
      // Wait to be woken up; the RMIServer has been started, so the
      // receiver can subscribe to listen to invalidates
      System.out.println("Waiting to be woken up on port: " +
                         WAIT_SIGNAL_PORT);
      Synchronizer.waitSignal(WAIT_SIGNAL_PORT);

      // Issue NUM_FILES writes of FILE_SIZE_BYTES bytes each
      try{
        WriteRemoteExpt.performWrites(uraNode);
      }catch(IOException e){
        System.err.println("" + e);
      }
    }else{
      System.err.println("Usage: java WriteRemoteExpt " +
                         "<configFileName> receiver <sender node id> " +
                         "<my node id> OR");
      System.err.println("java WriteRemoteExpt <configFileName> sender " +
                         "<my node id>");
    }
  }
}
