/*-
* See the file LICENSE for redistribution information.
*
* Copyright (c) 2002-2005
*      Sleepycat Software.  All rights reserved.
*
* $Id: SyncedLogManager.java,v 1.1 2005/03/16 21:35:45 dahlin Exp $
*/

package com.sleepycat.je.log;

import java.io.IOException;
import java.nio.ByteBuffer;

import com.sleepycat.je.DatabaseException;
import com.sleepycat.je.cleaner.TrackedFileSummary;
import com.sleepycat.je.cleaner.UtilizationTracker;
import com.sleepycat.je.dbi.EnvironmentImpl;

/**
 * The SyncedLogManager uses the synchronized keyword to implement protected
 * regions.
 */
public class SyncedLogManager extends LogManager {

    /**
     * There is a single log manager per database environment.
     */
    public SyncedLogManager(EnvironmentImpl envImpl,
                            boolean readOnly)
        throws DatabaseException {

        super(envImpl, readOnly);
    }

    protected LogResult logItem(LoggableObject item,
                                boolean isProvisional,
                                boolean flushRequired,
                                long oldNodeLsn,
                                boolean isDeletedNode,
                                int entrySize,
                                int itemSize,
                                boolean marshallOutsideLatch,
                                ByteBuffer marshalledBuffer,
                                UtilizationTracker tracker)
        throws IOException, DatabaseException {

        synchronized (logWriteLatch) {
            return logInternal(item, isProvisional, flushRequired,
                               oldNodeLsn, isDeletedNode, entrySize,
                               itemSize, marshallOutsideLatch,
                               marshalledBuffer, tracker);
        }
    }

    protected void flushInternal() 
        throws LogException, DatabaseException {

        try {
            synchronized (logWriteLatch) {
                logBufferPool.writeBufferToFile(0);
            }
        } catch (IOException e) {
            throw new LogException(e.getMessage());
        } 
    }

    /**
     * @see LogManager#countObsoleteLNs
     */
    public void countObsoleteNodes(long lsn1,
                                   LogEntryType type1,
                                   boolean obsolete1,
                                   long lsn2,
                                   LogEntryType type2,
                                   boolean obsolete2)
        throws DatabaseException {

        UtilizationTracker tracker = envImpl.getUtilizationTracker();
        synchronized (logWriteLatch) {
            countObsoleteNodesInternal(tracker,
                                       lsn1, type1, obsolete1,
                                       lsn2, type2, obsolete2);
        }
    }

    /**
     * @see LogManager#countObsoleteNodes
     */
    public void countObsoleteNodes(TrackedFileSummary[] summaries)
        throws DatabaseException {

        UtilizationTracker tracker = envImpl.getUtilizationTracker();
        synchronized (logWriteLatch) {
            countObsoleteNodesInternal(tracker, summaries);
        }
    }
}
