/*-
* See the file LICENSE for redistribution information.
*
* Copyright (c) 2002-2005
*      Sleepycat Software.  All rights reserved.
*
* $Id: DatabaseConfigTest.java,v 1.1 2005/03/16 21:35:49 dahlin Exp $
*/

package com.sleepycat.je;

import java.io.File;
import java.io.IOException;
import java.util.Comparator;

import junit.framework.TestCase;

import com.sleepycat.je.util.TestUtils;

/**
 * Basic database configuration testing.
 */
public class DatabaseConfigTest extends TestCase {
    private static final boolean DEBUG = false;
  
    private File envHome;
    private Environment env;

    public DatabaseConfigTest() {
        envHome = new File(System.getProperty(TestUtils.DEST_DIR));
    }

    public void setUp()
        throws IOException {

        TestUtils.removeLogFiles("Setup", envHome, false);
    }
    
    public void tearDown()
        throws Exception {

        try {
            /* Close in case we hit an exception and didn't close */
            if (env != null) {
            	env.close();
            } 
        } catch (DatabaseException e) {
            /* Ok if already closed */
        }
        env = null; // for JUNIT, to reduce memory usage when run in a suite.
        TestUtils.removeLogFiles("TearDown", envHome, false);
    }

    /**
     * Test that we can retrieve a database configuration and that it
     * clones its configuration appropriately.
     */
    public void testConfig()
        throws Throwable {

        try {
            EnvironmentConfig envConfig = new EnvironmentConfig();
            envConfig.setAllowCreate(true);
            env = new Environment(envHome, envConfig);

            /* 
             * Make sure that the database keeps its own copy of the
             * configuration object.
             */
            DatabaseConfig dbConfigA = new DatabaseConfig();
            dbConfigA.setAllowCreate(true);
            Database dbA = env.openDatabase(null, "foo", dbConfigA);

            /* Change the original dbConfig */
            dbConfigA.setAllowCreate(false);
            DatabaseConfig getConfig1 = dbA.getConfig();
            assertEquals(true, getConfig1.getAllowCreate());
            assertEquals(false,getConfig1.getSortedDuplicates());

            /* 
             * Change the retrieved config, ought to have no effect on
             * what the Database is storing.
             */
            getConfig1.setSortedDuplicates(true);
            DatabaseConfig getConfig2 = dbA.getConfig();
            assertEquals(false,getConfig2.getSortedDuplicates());
            
            dbA.close();
            env.close();
        } catch (Throwable t) {
            t.printStackTrace();
            throw t;
        }
    }

    /**
     * Make sure we can instantiate a comparator at the time it's set.
     */
    public void testComparator()
        throws Throwable {

        try {
            /* Can't be instantiated, a nested class */
            try {
                DatabaseConfig config = new DatabaseConfig();
                config.setBtreeComparator(BadComparator1.class);
                fail("Comparator shouldn't be instantiated");
            } catch (IllegalArgumentException e) {
                /* Expected. */
                if (DEBUG) {
                    System.out.println(e);
                }
            }

            /* No zero-parameter constructor */
            try {
                DatabaseConfig config = new DatabaseConfig();
                config.setBtreeComparator(BadComparator2.class);
                fail("Comparator shouldn't be instantiated");
            } catch (IllegalArgumentException e) {
                /* Expected. */
                if (DEBUG) {
                    System.out.println(e);
                }
            }

            /* Valid comparator */
            DatabaseConfig config = new DatabaseConfig();
            config.setBtreeComparator(TestComparator.class);

        } catch (Throwable t) {
            t.printStackTrace();
            throw t;
        }
    }
    

    /**
     * Test that any conflicts between configuration object settings
     * and the underlying impl object are detected.
     */
    public void testConfigConfict() 
        throws Throwable {
        
        try {
            EnvironmentConfig envConfig = new EnvironmentConfig();
            envConfig.setAllowCreate(true);
            env = new Environment(envHome, envConfig);

            /*
             *  Test conflicts of duplicate allowed configuration
             */

            /* 1a. Create allowing duplicates. */
            DatabaseConfig firstConfig = new DatabaseConfig();
            firstConfig.setAllowCreate(true);
            firstConfig.setSortedDuplicates(true);
            Database firstHandle = env.openDatabase(null, "fooDups",
                                                    firstConfig);
            /* 1b. Try to open w/no duplicates. */
            DatabaseConfig secondConfig = new DatabaseConfig();
            secondConfig.setSortedDuplicates(false);
            try {
                Database secondhandle =
                    env.openDatabase(null, "fooDups", secondConfig);
                fail("Conflict in duplicates allowed should be detected.");
            } catch (DatabaseException e) {
            }
            
            firstHandle.close();
            env.removeDatabase(null, "fooDups");

            /* 2a. Create dis-allowing duplicates. */
            firstConfig.setSortedDuplicates(false);
            firstHandle = env.openDatabase(null, "fooDups", firstConfig);
            /* 2b. Try to open w/duplicates. */
            secondConfig.setSortedDuplicates(true);
            try {
                Database secondhandle =
                    env.openDatabase(null, "fooDups", secondConfig);
                fail("Conflict in duplicates allowed should be detected.");
            } catch (DatabaseException e) {
            }
            firstHandle.close();

            /*
             * Test conflicts of read only. If the environment is read/write
             * we should be able to open handles in read only or read/write
             * mode. If the environment is readonly, the database handles 
             * must also be read only.
             */
            DatabaseConfig readOnlyConfig = new DatabaseConfig();
            readOnlyConfig.setReadOnly(true);
            Database roHandle = env.openDatabase(null, "fooDups",
                                                 readOnlyConfig);
            roHandle.close();

            /* Open the environment in read only mode. */
            env.close();
            envConfig = new EnvironmentConfig();
            envConfig.setReadOnly(true);
            env = new Environment(envHome, envConfig);
            
            /* Open a readOnly database handle, should succeed */
            roHandle = env.openDatabase(null, "fooDups",
                                        readOnlyConfig);
            roHandle.close();

            /* Open a read/write database handle, should succeed */
            try {
                Database rwHandle = env.openDatabase(null, "fooDups", null);
                fail("Should not be able to open read/write");
            } catch (DatabaseException e) {
                if (DEBUG) {
                    System.out.println(e);
                }
            }
            env.close();

            /*
             * Check comparator changes.
             */
            /* 1a. Open w/a null comparator */
            env = new Environment(envHome, null);
            firstConfig = new DatabaseConfig();
            firstConfig.setAllowCreate(true);
            firstHandle = env.openDatabase(null,
                                           "fooComparator",
                                           firstConfig);
            DatabaseConfig firstRetrievedConfig = firstHandle.getConfig();
            assertEquals(null, firstRetrievedConfig.getBtreeComparator());
            assertEquals(null, firstRetrievedConfig.getDuplicateComparator());

            /* 
             * 1b. Open a db w/a different comparator, shouldn't take effect
             * because override is not set.
             */
            secondConfig = new DatabaseConfig();
            Comparator btreeComparator = new TestComparator();
            Comparator dupComparator = new TestComparator();
            secondConfig.setBtreeComparator(btreeComparator.getClass());
            secondConfig.setDuplicateComparator(dupComparator.getClass());
            Database secondHandle =
		env.openDatabase(null, "fooComparator", secondConfig);
            DatabaseConfig retrievedConfig = secondHandle.getConfig();
            assertEquals(null, retrievedConfig.getBtreeComparator());
            assertEquals(null, retrievedConfig.getDuplicateComparator());
            secondHandle.close();

            /* 1c. Allow override */
            secondConfig.setOverrideBtreeComparator(true);
            secondConfig.setOverrideDuplicateComparator(true);
            secondHandle = env.openDatabase(null,
                                            "fooComparator",
                                            secondConfig);

            retrievedConfig = secondHandle.getConfig();
            assertEquals(btreeComparator.getClass(),
                         retrievedConfig.getBtreeComparator().getClass());
            assertEquals(dupComparator.getClass(),
                         retrievedConfig.getDuplicateComparator().getClass());
            firstHandle.close();
            secondHandle.close();
            env.close();
        } catch (Throwable t) {
            t.printStackTrace();
            env.close();
            throw t;
        }
    }


    public void testIsTransactional()
        throws Throwable {

        try {
            /* Open environment in transactional mode.*/
            EnvironmentConfig envConfig = new EnvironmentConfig();
            envConfig.setTransactional(true);
            envConfig.setAllowCreate(true);
            env = new Environment(envHome, envConfig);

            /* Create a db, open transactionally with implied auto-commit. */
            DatabaseConfig dbConfig = new DatabaseConfig();
            dbConfig.setAllowCreate(true);
            dbConfig.setTransactional(true);
            Database myDb = env.openDatabase(null, "testDB", dbConfig);
            assertTrue(myDb.isTransactional());
            assertTrue(myDb.getConfig().getTransactional());
            myDb.close();

            /* Open an existing db, can open it non-transactionally. */
            dbConfig.setTransactional(false);
            myDb = env.openDatabase(null, "testDB", null);
            assertFalse(myDb.isTransactional());
            assertFalse(myDb.getConfig().getTransactional());
            myDb.close();

            /* Open another db, pass an explicit transaction. */
            dbConfig.setTransactional(true);
            Transaction txn = env.beginTransaction(null, null);
            myDb = env.openDatabase(txn, "testDB2", dbConfig);
            assertTrue(myDb.isTransactional());
            assertTrue(myDb.getConfig().getTransactional());

            DatabaseEntry key = new DatabaseEntry();
            DatabaseEntry data = new DatabaseEntry();
            key.setData(TestUtils.getTestArray(0));
            data.setData(TestUtils.getTestArray(0));
            try {
                myDb.put(null, key, data);
            } catch (DatabaseException DBE) {
                fail("didn't expect DatabaseException, implied autocommit");
            }

            key.setData(TestUtils.getTestArray(1));
            data.setData(TestUtils.getTestArray(1));
            try {
                myDb.put(txn, key, data);
            } catch (DatabaseException DBE) {
                fail("didn't expect DatabaseException with txn passed");
            }

            try {
                myDb.get(txn, key, data, LockMode.DEFAULT);
            } catch (DatabaseException DBE) {
                fail("didn't expect DatabaseException with txn passed");
            }

            txn.commit();

            try {
                myDb.get(null, key, data, LockMode.DEFAULT);
            } catch (DatabaseException DBE) {
                fail("didn't expect DatabaseException because no txn passed");
            }

            myDb.close();

            env.close();
        } catch (Throwable t) {
            t.printStackTrace();
            throw t;
        }
    }

    public void testOpenReadOnly()
        throws Throwable {

        try {
            EnvironmentConfig envConfig = new EnvironmentConfig();
            envConfig.setTransactional(true);
            envConfig.setAllowCreate(true);
            env = new Environment(envHome, envConfig);

            DatabaseEntry key = new DatabaseEntry();
            DatabaseEntry data = new DatabaseEntry();

            Transaction txn = env.beginTransaction(null, null);
            DatabaseConfig dbConfig = new DatabaseConfig();
            dbConfig.setTransactional(true);
            dbConfig.setAllowCreate(true);
            Database myDb = env.openDatabase(txn, "testDB2", dbConfig);

            key.setData(TestUtils.getTestArray(0));
            data.setData(TestUtils.getTestArray(0));
            try {
                myDb.put(txn, key, data);
            } catch (DatabaseException DBE) {
                fail("unexpected DatabaseException during put");
            }

            txn.commit();
            myDb.close();

            dbConfig = new DatabaseConfig();
            dbConfig.setTransactional(true);
            dbConfig.setReadOnly(true);
            txn = env.beginTransaction(null, null);
            myDb = env.openDatabase(txn, "testDB2", dbConfig);
            assertTrue(myDb.isTransactional());
            assertTrue(myDb.getConfig().getTransactional());

            key.setData(TestUtils.getTestArray(0));
            data.setData(TestUtils.getTestArray(0));
            try {
                myDb.put(txn, key, data);
                fail("expected DatabaseException because open RDONLY");
            } catch (DatabaseException DBE) {
            }

            key.setData(TestUtils.getTestArray(0));
            data.setData(TestUtils.getTestArray(0));
            assertEquals(OperationStatus.SUCCESS,
                         myDb.get(txn, key, data, LockMode.DEFAULT));

            Cursor cursor = myDb.openCursor(txn, null);

            assertEquals(OperationStatus.SUCCESS,
                         cursor.getFirst(key, data, LockMode.DEFAULT));

            try {
                cursor.delete();
                fail("expected Exception from delete on RD_ONLY db");
            } catch (DatabaseException DBE) {
            }

            key.setData(TestUtils.getTestArray(1));
            data.setData(TestUtils.getTestArray(1));
            try {
                myDb.put(txn, key, data);
                fail("expected DatabaseException because open RDONLY");
            } catch (DatabaseException DBE) {
            }

	    cursor.close();
            txn.commit();
            myDb.close();

            env.close();
        } catch (Throwable t) {
            t.printStackTrace();
            throw t;
        }
    }

    /**
     * Test exclusive creation.
     */
    public void testExclusive()
        throws Throwable {

        try {
            EnvironmentConfig envConfig = new EnvironmentConfig();

            /* 
             * Make sure that the database keeps its own copy of the
             * configuration object.
             */
            envConfig.setAllowCreate(true);
            env = new Environment(envHome, envConfig);
            DatabaseConfig dbConfig = new DatabaseConfig();
            dbConfig.setAllowCreate(true);
            dbConfig.setExclusiveCreate(true);

            /* Should succeed and create the database. */
            Database dbA = env.openDatabase(null, "foo", dbConfig);
            dbA.close();
                                            
            /* Should not succeed, because the database exists. */
            try {
                Database dbB = env.openDatabase(null, "foo", dbConfig);
                fail("Database already exists");
            } catch (DatabaseException e) {
            }
            env.close();
        } catch (Throwable t) {
            t.printStackTrace();
            throw t;
        }
    }


    /*
     * This Comparator can't be instantiated because it's private and not
     * static.
     */
    private class BadComparator1 implements Comparator {
        public BadComparator1(int foo) {
        }
        
        public int compare(Object o1, Object o2) {
            return 0;
        }
    }

    /*
     * This Comparator can't be instantiated because it doesn't have 
     * zero parameter constructor.
     */
    public static class BadComparator2 implements Comparator {
        public BadComparator2(int i) {
        }

        public int compare(Object o1, Object o2) {
            return 0;
        }
    }    

    /*
     * OK comparator for setting comparators.
     */
    public static class TestComparator implements Comparator {
        public TestComparator() {
        }

        public int compare(Object o1, Object o2) {
            return 0;
        }
    }
}
