package code.lasr.db.berkdb;

import java.io.*;
import java.util.*;
import java.util.concurrent.TimeUnit;

//These are needed for the publicly accessible functions.
//Otherwise, use Berkeley DB internal classes (e.g. DatabaseException)
//Also, DO NOT import lasr.db.*; there are a lot of name conflicts.


import code.lasr.db.BufferBinding;
import code.lasr.db.DbException;
import code.lasr.db.DbHandle;
import code.lasr.db.DbTransaction;
import code.lasr.util.Debug;

import com.sleepycat.bind.EntryBinding;
import com.sleepycat.bind.serial.SerialBinding;
import com.sleepycat.bind.serial.StoredClassCatalog;
import com.sleepycat.bind.tuple.TupleBinding;
import com.sleepycat.je.*;

/**
 * A Berkeley DB environment containing configuration, serialization, etc.
 *
 * Should NOT be used by any object except BerkDbTransaction.
 **/
public class BerkDatabase
{
  // constants
  static final LockMode LOCK_MODE = LockMode.READ_UNCOMMITTED; // LockMode.READ_COMMITTED;

  // path used by the Berkeley DB
  private final String dbPath;

  // the database environment
  private final Environment theEnv;

  // the class catalog and its associated DB support the serialization 
  // of objects into and out of the other database
  private final Database classCatalogDb;
  private final StoredClassCatalog classCatalog;

  // database configuration options
  // for now, since we are only supporting one option, I create
  // two different configuration objects with the option set to true/false.
  // we might want to change this in the future if we support more
  private final DatabaseConfig dbConfigDup, dbConfigNoDup;

  // transaction configuration
  private final TransactionConfig txnConfig;

  // caches DB handles
  private final Map dbCache;

  private static BufferBinding bufferBinding;
  
  private static EntryBinding serialBind;


  // used to remove Berkeley DB related files
  static class BerkDbFileFilter implements java.io.FilenameFilter
  {
    public boolean accept(File dir, String name)
    {
      return name.endsWith("jdb") || name.endsWith("lck");
    }
  }

  public BerkDatabase(String path, long cache, boolean sync) throws code.lasr.db.DbException{
    try {
      dbPath = path;

      // open up the path to where the database will be stored
      File myDbEnvPath = new File(path);
      if (!myDbEnvPath.exists()){
        myDbEnvPath.mkdir();
        System.out.println("made dir berkeley db " + path);
      }else{

        System.out.println("initialized berkeley db " + path);
      }

      // database configuration
      dbConfigNoDup = new DatabaseConfig();
      dbConfigNoDup.setReadOnly(false);
      dbConfigNoDup.setAllowCreate(true);
      dbConfigNoDup.setTransactional(true);

      dbConfigDup = new DatabaseConfig();
      dbConfigDup.setReadOnly(false);
      dbConfigDup.setAllowCreate(true);
      dbConfigDup.setTransactional(true);
      dbConfigDup.setSortedDuplicates(true);    
      
      // transaction configuration
      // turn off blocking calls; transactions fail if no lock
      txnConfig = new TransactionConfig();
      // blocking or not?
      txnConfig.setNoWait(false);
      txnConfig.setReadUncommitted(true);
      txnConfig.setNoSync(!sync);

      // create cache, make it synchronized
      dbCache = Collections.synchronizedMap(new TreeMap());

      // create the environment

      // initialize the database environment    
      EnvironmentConfig theEnvConfig = new EnvironmentConfig();
      theEnvConfig.setLocking(true);
      theEnvConfig.setLockTimeout(20000L, TimeUnit.MILLISECONDS);
      theEnvConfig.setReadOnly(false);
      theEnvConfig.setAllowCreate(true);
      theEnvConfig.setTransactional(true);
      if(cache != -1){
        theEnvConfig.setCacheSize(cache);
      }
      
      EnvironmentMutableConfig mutableConfig = new EnvironmentMutableConfig();

      //
      // For a long time we said here
      //      mutableConfig.setTxnNoSync(true);
      // Unfortunately, this doesn't work -- we rely on PersistentLog as
      // a redo log and we assume that once something is in our local
      // UpdateLog that it is persistent. Unfortunately, commenting 
      // this out slows us down a lot (see mike's notes/2006/8/8b.txt).
      //
      mutableConfig.setTxnNoSync(!sync);

      // We print out a little message here until we figure out how to
      // efficiently sync the log
      System.out.print("DBG: MDD -- PersistentLog -- sync= " + sync);
      if(!sync){
        System.out.println("turned on; recovery won't work!");
      }else{
        System.out.println("turned off; recovery should work...");
      }

      //theEnvConfig.setConfigParam("je.env.runCleaner", "false");
      theEnv = new Environment(myDbEnvPath, theEnvConfig);
      theEnv.setMutableConfig(mutableConfig);

      // instantiate the class catalog
      DbHandle dbh = code.lasr.db.Database.getHandle("ClassCatalogDB", 
          false);
      classCatalogDb = openDatabase(dbh);
      classCatalog = new StoredClassCatalog(classCatalogDb);
      serialBind = new SerialBinding(classCatalog, Object.class);
      bufferBinding = new BufferBinding();

    } 
    catch (DatabaseException e) { 
      e.printStackTrace();
      throw new code.lasr.db.DbException(e); }

  }

  public BerkDatabase(String path) throws code.lasr.db.DbException{
    this(path, -1, true);
  }

  /**
   * Closes the environment.
   **/
  public void close() 
  {
    // clean out accessory variables
    dbCache.clear();

    // set configs to null

    // close the class catalog objects
    try { classCatalog.close(); }
    catch (DatabaseException e) { /* oh well, closing anyway */ }
    try { classCatalogDb.close(); }
    catch (DatabaseException e) { /* oh well, closing anyway */ }

    // close the environment
    try { theEnv.cleanLog(); }
    catch (DatabaseException e) { /* oh well, closing anyway */ }
    try { theEnv.close(); }
    catch (DatabaseException e) { /* oh well, closing anyway */ }

  }


  /**
   * Starts a new transaction.  This one is used by Database
   * to generate a new transaction for external use (vs.
   * createTransaction() which creates transactions for 
   * use within BerkDbTransaction).
   **/
  public DbTransaction newTransaction()
  {
    try { return new BerkDbTransaction(this); }
    catch (DbException e) {
      // TODO elwong This should never happen
      // To test this theory, do an exit here
      System.err.println("Unexpected DbException!");
      Debug.exit(e);
      return null;
    }
  }


  /**
   * Clears out the database.
   **/
  public void clearAll()
  {
    // remove all associated files
    if (dbPath != null && !dbPath.equals("null")) {
      File[] filelist = 
        new File(dbPath).listFiles(new BerkDbFileFilter());
      if (filelist != null) {
        for (int i=0; i < filelist.length; i++) {
          filelist[i].delete();
          code.lasr.util.Debug.println("debug", "Deleting "  + 
              filelist[i]);
        }
      }
    }
  }


  /**
   * Returns a reference to the database with the specified name.
   **/
  synchronized Database openDatabase(Transaction txn, DbHandle dbHandle)
  throws DatabaseException
  {
    Database db = (Database) dbCache.get(dbHandle);
    if (db == null) {
      db = theEnv.openDatabase(txn, dbHandle.getDbName(), 
          dbHandle.allowDuplicates() ? 
              dbConfigDup : dbConfigNoDup);

      // Note that this is NOT safe.  Berkeley DB is not happy when
      // handles are opened by transactions that abort; it leaves
      // the handle in an INVALID state.  Instead, we must take care 
      // of this upon commit using addHandleToCache().           
      // dbCache.put(dbHandle, db);
    }
    return db;
  }

  synchronized void addToCache(Map<DbHandle, Database> map)
  {
    dbCache.putAll(map);
  }

  public Database openDatabase(DbHandle dbHandle)
  throws DatabaseException
  {
    Database db = (Database) dbCache.get(dbHandle);
    while (db == null) {
      try { 
        Transaction txn = createTransaction();
        db = openDatabase(txn, dbHandle);
        txn.commit();	
        dbCache.put(dbHandle, db);
      }
      catch (DatabaseException e) {
        // Perhaps commit failed because someone else wrote it,
        // so try reading it.  Note that a DatabaseException
        // here is ok, but it isn't outside of the commit().
        db = (Database) dbCache.get(dbHandle);
      }
    }
    return db;
  }

  /**
   * Serializes the object.  Also adds it to the class catalog db.
   **/
  static DatabaseEntry serialize(Object obj, Class<?> type)
  {
      return serialize(obj, type, new DatabaseEntry());
  }
  static DatabaseEntry serialize(Object obj, Class<?> type, DatabaseEntry rv)
  {
    
      EntryBinding bind; 
      if(type == byte[].class){
        bind = bufferBinding;
      }else{
        bind = TupleBinding.getPrimitiveBinding(type);
      }
      if (bind == null) bind = serialBind;
      bind.objectToEntry(obj, rv);
      return rv;        
  }
  
  static Object unserialize(DatabaseEntry d, Class<?> type)
  {
      EntryBinding bind;
      if(type == byte[].class){
        bind = bufferBinding;
      }else{
        bind = TupleBinding.getPrimitiveBinding(type);
      }
      if (bind == null) bind = serialBind;
      return bind.entryToObject(d);
  }

  /**
   * Creates a new Berkeley DB transaction which is internally
   * used by BerkDbTransaction.
   **/
  Transaction createTransaction() throws DatabaseException
  {
    return theEnv.beginTransaction(null, txnConfig);
  }


  public void closeDatabase(DbHandle dbHandle) throws DatabaseException
  {    
    Database db = (Database) dbCache.get(dbHandle);
    if(db != null){
      closeDatabase(db);
    }else{
      System.err.println(dbHandle + " not found");
    }
  }

  /**
   * Closes the specified database.
   **/
  void closeDatabase(Database db) throws DatabaseException
  {
    db.close();
  }	


  /**
   * Get names of all the databases.
   **/
  java.util.List getDatabaseNames() throws DatabaseException
  {
    return theEnv.getDatabaseNames();
  }


  /**
   * Remove a database.
   **/
  public void removeDatabase(Transaction txn, DbHandle dbh)  
  throws DatabaseException
  {
    /* close the handle in the cache */
    Database db = (Database) dbCache.get(dbh);
    if (db != null) {
      dbCache.remove(dbh);
      closeDatabase(db);
    }
    theEnv.removeDatabase(txn, dbh.getDbName());
  }


  /**
   * Truncate a database.
   **/
  public void truncateDatabase(Transaction txn, DbHandle dbh)  
  throws DatabaseException
  {
    /* close the handle in the cache */
    Database db = (Database) dbCache.get(dbh);
    if (db != null) {
      dbCache.remove(dbh);
      closeDatabase(db);
    }
    theEnv.truncateDatabase(txn, dbh.getDbName(), false);
  }

}

