package code.lasr.db.berkdb;


// Berkeley DB stuff
import code.lasr.db.*;
import code.lasr.util.*;

import com.sleepycat.je.*;

public class BerkDbIterator implements DbIterator
{
    private Cursor cursor;
    private DatabaseEntry currKey = new DatabaseEntry();
    private DatabaseEntry currValue = new DatabaseEntry();
    private boolean closedFlag;
    private int eqKeyCount;
    private BerkDatabase intDb;
    private DbHandle dbHandle;
    
    BerkDbIterator(BerkDatabase intDb, BerkDbTransaction txn, DbHandle dbHandle) 
	throws DbException
    { 
    	// open the database we need
    	com.sleepycat.je.Database db = null;
    	this.intDb = intDb;
    	try { db = intDb.openDatabase(txn.txn, dbHandle); }
    	catch (DatabaseException e) 
    	{ 
    		throw new OperationFailedException("Could not open database " + 
    				dbHandle, e); 
    	}
    	cursor = txn.openCursor(dbHandle);
        closedFlag = false;
        this.dbHandle = dbHandle;
    }
    
    // hasNextFlag is true if we *know* there is another value.
    // Otherwise, there either isn't another value or we aren't sure.
    private boolean hasNextFlag = false;
    
    /**
     * Checks whether there are any values left in the database.
     * Note that this function advances the iterator.  We *rely* on this
     * fact in next(), which calls hasNext() to advance the pointer.
     */
    public boolean hasNext() throws DbException  
    {
        if (closedFlag) throw new DbException("Cursor is closed");

    	// If hasNextFlag is true, then we know for sure there is more data.
    	// This also prevents us from calling hasNext() multiple times successively
    	// if there is data left.
        
    	// Otherwise, attempt to advance.  Note that although hasNextFlag is false and
    	// thus we do *not* know whether there is more data, attempting to advance the 
        // cursor when it is at the end of the database does not cause any problems.

    	if (!hasNextFlag) {
            try {
                OperationStatus rv;
                rv = cursor.getNext(currKey, currValue, 
                                    BerkDatabase.LOCK_MODE);
                hasNextFlag = (rv == OperationStatus.SUCCESS);
            }
            catch (DatabaseException ex) { throw new DbException(ex); }
    	}
    	return hasNextFlag;
    }

    // advanceFlag is true if we need to advance to get the next value.
    // Otherwise, it is false.
    public Object next() throws DbException
    {
        if (closedFlag) throw new DbException("Cursor is closed");

    	// If hasNextFlag is true, we know for sure there is another element that
    	// is waiting in currValue.
    	if (hasNextFlag) {
    		// Reset hasNextFlag to false since we are using the value.
    		hasNextFlag = false;
    		return currValue;
    	}
    	
    	// If hasNextFlag is false, we are not sure, so we have to advance
    	// the iterator and find out.
    	else {
    		// This uses the assumption that hasNext() both advances the pointer and
    		// updates hasNextFlag.  So call hasNext() and see what happens.  If
    		// it returns true, we know again that there is another element.
    		if (hasNext()) {
    			hasNextFlag = false;
    			return currValue;    			
    		}
    		
    		// Otherwise, we know for sure now that there is no more data.
    		else return null;
    	}
    }
    
    public void reset() throws DbException
    {
        if (closedFlag) throw new DbException("Cursor is closed");
        
        try {
        	// Reset the iterator by setting the cursor back to the first value.
        	OperationStatus rv;
            rv = cursor.getFirst(currKey, currValue, 
                               	BerkDatabase.LOCK_MODE);
            
            // If it was successful, we *know* once again that there is data
            // available and thus hasNextFlag will be true.  If not, we know there 
            // is no data in this database and hasNextFlag will be false.
            hasNextFlag = (rv == OperationStatus.SUCCESS);
        }
        catch (DatabaseException ex) { throw new DbException(ex); }
    }

    public void delete() throws DbException 
    { 
        if (closedFlag) throw new DbException("Cursor is closed");
	try { cursor.delete(); }
	catch (DatabaseException ex) { throw new DbException(ex); }
    }

    public Object getKey() throws DbException
    {
	return intDb.unserialize(currKey, dbHandle.getKeyType());
    }

    public Object getValue() throws DbException
    {
        if (closedFlag) throw new DbException("Cursor is closed");
	return intDb.unserialize(currValue, dbHandle.getValType());
    }

    public boolean seek(Object k) throws DbException
    {
        try {
            DatabaseEntry keyEntry = new DatabaseEntry(Util.getBytes(k));
            DatabaseEntry valEntry = new DatabaseEntry();

            if (cursor.getSearchKeyRange(keyEntry, valEntry, 
                                         BerkDatabase.LOCK_MODE) ==
                OperationStatus.NOTFOUND) return false;
            else return true;
        }
        catch (DatabaseException e) { throw new DbException(e); }
    }
    

    public boolean seek(Object k, Object v) throws DbException
    {
        try {
            DatabaseEntry keyEntry = new DatabaseEntry(Util.getBytes(k));
            DatabaseEntry valEntry = new DatabaseEntry(Util.getBytes(v));
            
            if (cursor.getSearchBothRange(keyEntry, valEntry, 
                                          BerkDatabase.LOCK_MODE) ==
                OperationStatus.NOTFOUND) return false;
            else return true;
        }
        catch (DatabaseException e) { throw new DbException(e); }
    }

    public void close() throws DbException
    {
        if (!closedFlag) {
            closedFlag = true;
            try { cursor.close(); }
            catch (DatabaseException e) { throw new DbException(e); }
        }
    }
}
