package code.lasr.db.berkdb;

import java.util.*;

//we need all the DB exceptions and interfaces

//TODO don't know if we really need this

//Berkeley DB stuff
import code.lasr.db.*;
import code.lasr.util.Debug;
import code.lasr.util.Util;

import com.sleepycat.je.*;

/** 
 * This is considered a Berkeley DB transaction.
 **/
public class BerkDbTransaction implements DbTransaction
{
	// the transaction
	Transaction txn;
	BerkDatabase intDb;
	
	// transaction finished
	protected boolean finishedFlag = false;
	private Map<DbHandle, com.sleepycat.je.Database> 	
	handleMap = new TreeMap<DbHandle, com.sleepycat.je.Database>();

	// list of iterators opened
	private LinkedList<DbIterator> iteratorList = new LinkedList();

	// list of handles opened

	/** 
	 * Initializes a transaction with the specified database.
	 **/
	public BerkDbTransaction(BerkDatabase intDb) throws DbException
	{
	  this.intDb = intDb;
		try {
			// TODO elwong It is unclear why these statements would ever fail
			// but they sometimes do
			txn = intDb.createTransaction();
		}
		catch (DatabaseException e) {
			throw new UnexpectedException(e); 
		}
	}


	/**
	 * Removes a database.
	 **/
	public final void removeDatabase(DbHandle dbh) throws DbException
	{
		try { intDb.removeDatabase(txn, dbh); }
		catch (DatabaseException e) { throw new DbException(e); }
	}

	public void removeAll(DbHandle dbh) throws DbException
	{
		try { intDb.truncateDatabase(txn, dbh); }
		catch (DatabaseException e) { throw new DbException(e); }
	}

	/** 
	 * Write an object to the database.
	 **/
	public final void write(DbHandle dbHandle, Object k, Object v) 
	throws OperationFailedException, UnexpectedException
	{
		// set up infrastructure
		DatabaseEntry theKey = intDb.serialize(k, dbHandle.getKeyType());

		// TODO don't know if we should need Util.getBytes()
		DatabaseEntry theData = intDb.serialize(v, dbHandle.getValType());

		// open the database we need
		com.sleepycat.je.Database db = null;
		try { 
			db = intDb.openDatabase(txn, dbHandle); 
			handleMap.put(dbHandle, db);
		}
		catch (DatabaseException e) 
		{ 
			throw new OperationFailedException("Could not open database " + 
					dbHandle, e); 
		}

		// now try writing
		OperationStatus rv;
		try { rv = db.put(txn, theKey, theData); }
		catch (LockNotGrantedException e) 
		{ 
			throw new OperationFailedException("Failed writing to " + 
					dbHandle + " on key " + k, e); 
		}
		catch (DatabaseException e)
		{ throw new UnexpectedException(e); }

		// check the return value
		if (rv == OperationStatus.KEYEXIST) 
			throw new OperationFailedException("db.put() returned " + rv);
		else if (rv != OperationStatus.SUCCESS) 
			throw new UnexpectedException("db.put() returned " + rv);
	}


	/** 
	 * Read an object from the database.
	 **/
	public final Object read(DbHandle dbHandle, Object k) 
	throws OperationFailedException, UnexpectedException
	{
		// set up infrastructure
		DatabaseEntry searchKey = intDb.serialize(k, dbHandle.getKeyType());
		DatabaseEntry foundData = new DatabaseEntry();

		// open the needed database
		Object theObj = null;
		com.sleepycat.je.Database db = null;
		try { 
			db = intDb.openDatabase(txn, dbHandle); 
			handleMap.put(dbHandle, db);
		}
		catch (DatabaseNotFoundException e) { return null; }
		catch (DatabaseException e) 
		{ 
			throw new OperationFailedException("Could not open database " + 
					dbHandle, e); 
		}

		// read from the database
		OperationStatus rv;
		try { rv = db.get(txn, searchKey, foundData, BerkDatabase.LOCK_MODE); }
		catch (LockNotGrantedException e) 
		{ 
			throw new OperationFailedException("Failed reading from " + 
					dbHandle + " on key " + k, e); 
		}
		catch (DatabaseException e) { throw new UnexpectedException(e); }

		if (rv == OperationStatus.NOTFOUND)
			return null;
		else if (rv != OperationStatus.SUCCESS)
			throw new UnexpectedException("db.get() returned " + rv);
		return intDb.unserialize(foundData, dbHandle.getValType());
	}



	/**
	 * Removes an object from the database.
	 **/
	public final void removeAll(DbHandle dbHandle, Object k) 
	throws UnexpectedException, OperationFailedException
	{
		DatabaseEntry theKey = intDb.serialize(k, dbHandle.getKeyType());
		com.sleepycat.je.Database db = null;

		// open the database we need
		try { 
			db = intDb.openDatabase(txn, dbHandle); 
			handleMap.put(dbHandle, db);
		}
		catch (DatabaseException e) 
		{ 
			throw new OperationFailedException("Could not open database " + 
					dbHandle, e); 
		}

		// delete
		try { 
			// we don't really care about the return value, just remove all
			db.delete(txn, theKey);
		}
		catch (LockNotGrantedException ex) { 
			throw new OperationFailedException(ex);
		}
		catch (DatabaseException ex) { 
			throw new UnexpectedException(ex); 
		}
	}


	public final void remove(DbHandle dbHandle, Object k)
	throws DbException
	{ 
		remove(dbHandle, k, null);
	}

	public final void remove(DbHandle dbHandle, Object key, Object value) 
	throws OperationFailedException, NotFoundException, UnexpectedException
	{
		DatabaseEntry theKey = intDb.serialize(key, dbHandle.getKeyType());
		DatabaseEntry theValue;
		if (value != null) 
			theValue = intDb.serialize(value, dbHandle.getValType());
		else
			theValue = new DatabaseEntry();
		com.sleepycat.je.Database db = null;

		// open the database we need
		try { 
			db = intDb.openDatabase(txn, dbHandle); 
			handleMap.put(dbHandle, db);
		}
		catch (DatabaseException e) 
		{ 
			throw new OperationFailedException("Could not open database " + 
					dbHandle, e); 
		}

		// open a cursor
		OperationStatus rv;
		Cursor cursor = openCursor(dbHandle);

		// find the data element of interest and delete it
		try {
			if (value == null) 
				rv = cursor.getSearchKey(theKey, theValue, 
						BerkDatabase.LOCK_MODE);
			else 
				rv = cursor.getSearchBoth(theKey, theValue, 
						BerkDatabase.LOCK_MODE);
			if (rv == OperationStatus.SUCCESS) rv = cursor.delete();
		}
		catch (LockNotGrantedException e) { throw new OperationFailedException(e); }
		catch (DatabaseException e) { throw new UnexpectedException(e); }
		finally { 
			try { cursor.close(); }
			catch (DatabaseException e) { throw new UnexpectedException(e); }
		}

		// check return value
		if (rv == OperationStatus.NOTFOUND) {
			if (value == null) throw new NotFoundException(key, dbHandle);
			else throw new NotFoundException(key, value, dbHandle);
		}

		else if (rv != OperationStatus.SUCCESS) 
			throw new UnexpectedException("db.delete() returned " + rv);

		// else if (value != null) return value;
		// else return BerkDatabase.unserialize(theValue);
	}


	/**
	 * Returns all the keys in the database.
	 **/
	public final Object[] listAllKeys(DbHandle dbHandle) throws DbException
	{
		Cursor cursor = null;

		// open the database we need
		com.sleepycat.je.Database db = null;
		try { 
			db = intDb.openDatabase(txn, dbHandle); 
			handleMap.put(dbHandle, db);
		}
		catch (DatabaseNotFoundException e) { return null; }
		catch (DatabaseException e) 
		{ 
			throw new OperationFailedException("Could not open database " + 
					dbHandle, e); 
		}

		// open a cursor on the databse
		// TODO elwong Is this an Unexpected or OperationFailedException?
		cursor = openCursor(dbHandle);

		// the prefix we are searching for
		DatabaseEntry theKey = new DatabaseEntry();
		DatabaseEntry theData = new DatabaseEntry();
		try { 
			OperationStatus rv = cursor.getFirst(theKey, theData, 
					BerkDatabase.LOCK_MODE);
			if (rv == OperationStatus.NOTFOUND) {
				// no data at all
				return null;
			}
			else if (rv == OperationStatus.SUCCESS) {
				// grab all the data that matches
				List l = new LinkedList();
				Object key = intDb.unserialize(theKey, dbHandle.getKeyType());
				l.add(key);
				while (cursor.getNext(theKey, theData, BerkDatabase.LOCK_MODE) == OperationStatus.SUCCESS) {
					l.add(intDb.unserialize(theKey, dbHandle.getKeyType()));
				}
				return l.toArray();
			}
			else {
				throw new OperationFailedException("cursor.getFirst() " +
						"returned " + rv);
			}
		}
		catch (LockNotGrantedException e) 
		{
			throw new OperationFailedException(e);
		}
		catch (DatabaseException e) { throw new UnexpectedException(e); }
		finally {
			try { cursor.close(); }
			catch (DatabaseException e) { throw new UnexpectedException(e); }
		}
	}


	/**
	 * Returns all the objects in the database that match a given prefix.
	 **/
	public final Object[] retrieveAllValues(DbHandle dbHandle) throws DbException
	{
		// open a cursor on the databse
		// TODO elwong Is this an Unexpected or OperationFailedException?
		Cursor cursor = openCursor(dbHandle);
		if (cursor == null) return null;

		// the prefix we are searching for
		DatabaseEntry theKey = new DatabaseEntry();
		DatabaseEntry theData = new DatabaseEntry();

		try { 
			// no data at all
			OperationStatus rv = cursor.getFirst(theKey, theData, 
					BerkDatabase.LOCK_MODE);
			if (rv == OperationStatus.NOTFOUND) return null;
			else if (rv == OperationStatus.SUCCESS) {
				List l = new LinkedList();
				l.add(intDb.unserialize(theData, dbHandle.getValType()));
				while (cursor.getNext(theKey, theData, BerkDatabase.LOCK_MODE) == OperationStatus.SUCCESS)
					l.add(intDb.unserialize(theData, dbHandle.getValType()));
				return l.toArray();
			}
			else 
				throw new OperationFailedException("cursor.getFirst() " +
						"returned " + rv);
		}
		catch (LockNotGrantedException e) {
			throw new OperationFailedException(e);
		}
		catch (DatabaseException e) { throw new UnexpectedException(e); }
		finally {
			try { cursor.close(); }
			catch (DatabaseException e) { throw new UnexpectedException(e); }
		}
	}


	/**
	 * Pop the top element off the database.
	 **/
	public final Object pop(DbHandle dbHandle) 
	throws OperationFailedException, UnexpectedException
	{
		Cursor cursor = openCursor(dbHandle);

		DatabaseEntry theKey = new DatabaseEntry();
		DatabaseEntry theData = new DatabaseEntry();
		OperationStatus rv;
		try { 
			rv = cursor.getFirst(theKey, theData, BerkDatabase.LOCK_MODE);
			if (rv == OperationStatus.SUCCESS) {
				// read top successfully, now remove it
				rv = cursor.delete();
				if (rv == OperationStatus.SUCCESS)
					return intDb.unserialize(theData, dbHandle.getValType());
				else
					throw new UnexpectedException("cursor.delete() " +
							"returned " + rv);
			}
			else if (rv == OperationStatus.NOTFOUND) 
				// queue is empty, return null
				return null;
			else
				// whaaaa?
				throw new UnexpectedException("cursor.getFirst() " +
						"returned " + rv);
		}
		catch (LockNotGrantedException e) 
		{
			throw new OperationFailedException("Failed popping from " + 
					dbHandle, e);
		}
		catch (DatabaseException e) { throw new UnexpectedException(e); }
		finally { 
			try { cursor.close(); }
			catch(DatabaseException e) { throw new UnexpectedException(e); }
		}
	}


	/** 
	 * Get an iterator to the first (key, value) in the database.
	 **/
	public final DbIterator getIterator(DbHandle dbHandle) throws DbException
	{
		DbIterator it = new BerkDbIterator(intDb, this, dbHandle); 
		iteratorList.add(it);
		return it;
	}


	/**
	 * Close all iterators.  Used by commit(), sync() and abort().
	 **/
	public final void closeAllIterators() 
	{
		Iterator it = iteratorList.iterator();
		while (it.hasNext()) {
			DbIterator dbIt = (DbIterator) it.next();
			try { dbIt.close(); }
			catch (DbException e) { /* oh well */ }
		}
	}


	/**
	 * Commit a transaction, not necessarily to persistent storage.
	 **/
	public final void commit() throws TxnFailedException
	{
		if (finishedFlag) return;
		finishedFlag = true; 
		closeAllIterators();
		try { 
		    //			txn.commitWriteNoSync(); 
		    txn.commitSync();
		    intDb.addToCache(handleMap);
		}
		catch (DatabaseException e) { throw new TxnFailedException(e); }
	}

	/**
	 * Commit a transaction, not necessarily to persistent storage.
	 **/
	public final void sync() throws TxnFailedException
	{
		if (finishedFlag) return;
		finishedFlag = true; 
		closeAllIterators();
		try { 
			txn.commitSync(); 
			intDb.addToCache(handleMap);
		}
		catch (DatabaseException e) { throw new TxnFailedException(e); }
	}

	/**
	 * Abort a transaction.
	 **/
	public final void abort()
	{
		// if the transaction has already been successfully aborted,
		// then abort succeeds.
		if (finishedFlag) return;
		finishedFlag = true;
		closeAllIterators(); 
		try { txn.abort(); }
		catch (DatabaseException e) { }
	}


	/**
	 * Open a cursor.  For internal use only.
	 **/
	Cursor openCursor(DbHandle dbHandle) 
	throws UnexpectedException, OperationFailedException
	{ 
		// open the database we need
		com.sleepycat.je.Database db;
		try { 
			db = intDb.openDatabase(txn, dbHandle); 
			handleMap.put(dbHandle, db); 
		}
		catch (DatabaseNotFoundException e) { return null; }
		catch (DatabaseException e) 
		{ 
			throw new OperationFailedException("Could not open database " + 
					dbHandle, e); 
		}

		// open a cursor
		try { return db.openCursor(txn, null); }
		catch (DatabaseException e) { throw new UnexpectedException(e); }
	}


	/**
	 * Finalize by aborting just in case.
	 **/
	protected void finalize() 
	{ 
		if (finishedFlag) return;
		closeAllIterators();
		abort();
	}
}

