package code.lasr.util;

import java.io.*;
import java.net.*;
import java.util.Date;
import java.security.MessageDigest;


/**
 *  Contains utility functions which are useful across multiple locations,
 *  but do NOT depend on the rest of the PIB world.
**/
public class Util
{
    protected static int uid_count = 0;
    protected static long lastTimestamp = 0;

    public static synchronized void initialize(long time)
    {
        lastTimestamp = time;
    }

    /**
     * Generates a unique identifier.
     **/
    public static synchronized String getUID() 
    {
	long timestamp = getTime();
	if (lastTimestamp != timestamp) 
	    uid_count = 0;
	lastTimestamp = timestamp;

	// assumes timestamps have 14 digits and UIDs have up to 4
	String timestampString = "00000000000000" + 
	    new Long(timestamp).toString();
	String uidString = "0000" + new Long(uid_count++).toString();
	return timestampString.substring(timestampString.length()-14) +
	    "-" + uidString.substring(uidString.length()-4);
    }

    /**
     *  gets the byte representation of an object
     *  Does not include transient fields in the byte representation
     **/
    public static byte[] getBytes(Object o){
	try{
	    ByteArrayOutputStream os = new ByteArrayOutputStream();
	    ObjectOutputStream oos = new ObjectOutputStream(os);
	    oos.writeObject(o);
	    oos.flush();
	    os.flush();
	    byte[] res = os.toByteArray();
	    oos.close();
	    return res;
	}catch(Exception e){
	    throw new RuntimeException(e);
	}
    } 

    /** gets an object from its byte representation **/
    public static Object fromBytes(byte[] serialized) throws java.io.IOException,java.lang.ClassNotFoundException {
        ObjectInputStream oi = new ObjectInputStream(new ByteArrayInputStream(serialized));
        return oi.readObject();
    }


    protected static MessageDigest md;
    /** returns the hash name of a byte array **/
    public synchronized static Hash getHash(byte bytes[]){
        Hash ret=null;
        try{
            if (md == null) md = MessageDigest.getInstance("MD5");
            md.reset();
            md.update(bytes);
	    byte[] tmp = md.digest();
            ret = new Hash(tmp);
        }catch(Exception e){
            throw new RuntimeException("hashing is broken!");
        }
        return ret;
        //        String byteString = new String(bytes);
        //        return Integer.toString(byteString.hashCode());
    }
    public static Hash getHash(Object o){
	return getHash(getBytes(o));
    }


    /** utility function used for testing purposes **/
    public static boolean compareBytes(byte[] one, byte[] two){
        if (one == null || two == null)
            return false;
        return java.util.Arrays.equals(one,two);
    }

    /** returns the current local time in ms since 1970 **/
    public static long getTime(){
        return System.currentTimeMillis();
    }
    
    


    /** makes the indicatd directory if its not already there **/
    public static void mkdir(String dir) throws Exception{
	if (dir == null) return;
        File f = new File(dir);
        if (f.exists())
            return;
        f.mkdirs();
    }


    /** return the first non-null element in the array, or null
        if all elements are null (or the array is a null pointer)
    **/
    public static Object firstNonNull(Object[] array) {
        if (null==array) return null;
        for (int i=0; i<array.length; i++)
            if (null!=array[i]) return array[i];
        return null;
    }

    /** return the first non-null element in the iteration, or null
        if all elements are null (or iter is a null pointer)
    **/
    public static Object firstNonNull(java.util.Iterator iter) {
        if (null==iter) return null;
        while (iter.hasNext()) {
            Object x = iter.next();
            if (null!=x) return x;
        }
        return null;
    }

    /** return the index of the instance of the object in the array,
	or -1 if it's not there. The array does not need to be
	sorted. **/
    public static int indexOf(Object[] array, Object inArray) {
	if (null==array) return -1;
	for (int i=0; i<array.length; i++) {
	    if (array[i]!=null && array[i].equals(inArray)) {
		return i;
	    }
	}
	return -1;
    }

    /** return your IP address **/
    public static String getIP()
    {
	try { return InetAddress.getLocalHost().getHostAddress(); }
	catch (Exception e) { return null; }
    }

    /** sleep without exceptions **/
    public static void sleep(int millis)
    {
	try { Thread.sleep(millis); }
	catch (Exception e) { /* whatevs */ }
    }
}


