package code.security;

import java.io.EOFException;
import java.io.IOException;
import java.util.*;

import code.*;
import code.security.ahs.*;
import code.security.holesync.filter.*;
import code.security.holesync.*;
import code.security.liveness.*;

public class NewCheckpointFilter {

  SecurityFilter securityFilter;

  public static final boolean reqLastPrecise = true;
  public static final boolean verifyLastInval = true;

  private Hashtable<NodeId, TreeSet<Long>> globalSplitPoints;

  NewCheckpointFilter(SecurityFilter securityFilter)
  {
    this.securityFilter = securityFilter;
    globalSplitPoints = new Hashtable<NodeId, TreeSet<Long>>();
  }

  boolean apply(SecureCheckpoint chkPt, NodeId sender, LinkedList<GeneralInv> giList){
    long start =0 , end =0;

    AcceptVV cvv = securityFilter.getCurrentVV();

    assert(SangminConfig.securityLevel > SangminConfig.NONE);
    //System.out.println("applySecureCheckpoint:" + securityFilter.getCurrentVV() + " \n " + chkPt);
    // first check the correctness of secure checkpoint
    // then update the lpVV based on the received information
    if(this.securityFilter.measureTime){
      start = System.currentTimeMillis();
    }
    if(!getMaxDVV(chkPt, sender)){
      return false;
    }else if(verifyAndApplyCheckpoint(chkPt, giList, sender)){
      AcceptVV startVV  = chkPt.getStartVV(); 
      //
      // No lock shoud be held before calling updateState 
      // Refer to OutgoingConnectionMailbox.newUpdate()
      if(this.securityFilter.measureTime){
        end = System.currentTimeMillis();
        LatencyWatcher.put("SecurityFilter.SecureCheckpointVerification", 
            (end - start));
        start = end;
      }

      // now we deal with the 2nd case: chkPt is splittable across curVV when chkPt.getEndVV() > curVV
      DependencyVV dvv = securityFilter.getSplitCVVBetweenStartAndEnd(chkPt.getStartVV(), chkPt.getEndVV(), cvv);
      //if(SecurityFilter.dbg)System.out.println("case 2: ensuring verifiability of " + dvv + " startVV " + chkPt.getStartVV() + " endVV " + chkPt.getEndVV() + " cvv" + securityFilter.getCurrentVV());

      if(!securityFilter.getLivenessFilter().ensureCompatibility(dvv, securityFilter.getAhsMap(), sender)){
        assert false;
        return false;
      }

      if(this.securityFilter.measureTime){
        end = System.currentTimeMillis();
        LatencyWatcher.put("SecurityFilter.SecureCheckpointUpdateState", 
            (end - start));
      }
      return true;   
    }else{
      System.out.println("invalid checkpoint received: " + chkPt);
    }
    return false;
  }

  private boolean verifyAndApplyCheckpoint(SecureCheckpoint chkPt, LinkedList<GeneralInv> giList,  NodeId sender){
    boolean ret = true;
    AcceptVV initialVV = securityFilter.getCurrentVV();
    if(verifyCheckpoint(chkPt, giList, sender, initialVV)){
      ret = true;    
    }else{
      ret = false;
      System.out.println("invalid checkpoint received: " + chkPt);
    }
    return ret;
  }

  public AHSEntry getAHSEntryWithoutSH(AHSEntry ahsEntry){
    if(!SangminConfig.optimizeDVVMapHashes){
      return ahsEntry;
    }
    // add the SHs to the AHSEntry whereever missing
    DVVMap dvvMap = ahsEntry.getMaxDVVMap();
    Hashtable<NodeId, DVVMapEntry> newDVVMap = new Hashtable<NodeId, DVVMapEntry>();
    boolean modified = false;
    for(NodeId nodeId: dvvMap.getNodeSet()){
      DVVMapEntry dme = dvvMap.getEntry(nodeId);
      if(isSplit(nodeId, dme.getTimeStamp())){
        dme = new DVVMapEntry(dme.getTimeStamp());
        modified = true;
      }
      newDVVMap.put(nodeId, dme);
    }
    if(!modified){
      return ahsEntry;
    }else{
      return new AHSEntry(ahsEntry.getLevel(), ahsEntry.getHash(), ahsEntry.getStartTS(), ahsEntry.getEndTS(), ahsEntry.getInvalTarget(), new DVVMap(newDVVMap), null);
    }
  } 

  public AHSEntry getAHSEntryWithoutLocalSH(AHSEntry ahsEntry, NodeId n){
    if(!SangminConfig.optimizeDVVMapHashes){
      return ahsEntry;
    }
    // add the SHs to the AHSEntry whereever missing
    DVVMap dvvMap = ahsEntry.getMaxDVVMap();
    Hashtable<NodeId, DVVMapEntry> newDVVMap = new Hashtable<NodeId, DVVMapEntry>();
    boolean modified = false;
    for(NodeId nodeId: dvvMap.getNodeSet()){
      DVVMapEntry dme = dvvMap.getEntry(nodeId);
      if(n.equals(nodeId) && isSplit(nodeId, dme.getTimeStamp())){
        dme = new DVVMapEntry(dme.getTimeStamp());
        modified = true;
      }
      newDVVMap.put(nodeId, dme);
    }
    if(!modified){
      return ahsEntry;
    }else{
      return new AHSEntry(ahsEntry.getLevel(), ahsEntry.getHash(), ahsEntry.getStartTS(), ahsEntry.getEndTS(), ahsEntry.getInvalTarget(), new DVVMap(newDVVMap), null);
    }
  } 
  
  public AHS getAHSWithoutSH(AHS ahs){
    if(!SangminConfig.optimizeDVVMapHashes){
      return ahs;
    }
    AHS newAHS = new AHS();
    for(AHSEntry ahsEntry: ahs.getSortedListOfEntries()){
      newAHS.add(getAHSEntryWithoutSH(ahsEntry));
    }
    return newAHS;
  } 
  
  
  boolean verifyCheckpoint(SecureCheckpoint chkPt, LinkedList<GeneralInv> giList,  NodeId sender, AcceptVV initialVV){
    HashMap<NodeId, Boolean> lastVerifiedPrecise = new HashMap<NodeId, Boolean>();
    HashMap<NodeId, Long> lastVerifiedTime = new HashMap<NodeId, Long>(); 
    long start=0,end=0,start2=0,end2=0;
    if(this.securityFilter.measureTime){
      start = System.currentTimeMillis();
    }
    if(this.securityFilter.measureTime){
      end = System.currentTimeMillis();

      LatencyWatcher.put("SecurityFilter.SecureCheckpointsecurityFilter.getAhsMap()Clone", 
          (end - start));

    }
    SplitManager.advance = true;


    AcceptVV currentVV = securityFilter.getCurrentVV();
    CounterVV cvv = new CounterVV(currentVV);

    AcceptVV finalCVV = new AcceptVV(cvv).cloneMaxVV(chkPt.getEndVV());
    HashMap<Hole, HoleFiller> hf = chkPt.getHoleFiller();
    TreeSet<Hole> ts = new TreeSet<Hole>(hf.keySet());
    for(Hole h: ts){
      if(!verifyAndApplyHoleFiller(h, hf.get(h), finalCVV, sender)){
        assert false;
        return false;
      }
    }

    //indicates the VV of the last precise update received by the corresponding writer
    //  HashMap<NodeId, AcceptVV> lastPreciseVVMap = new HashMap<NodeId, AcceptVV>();

    //stores the unverified IHs received for a given node
    HashMap<NodeId, Vector<IH>> newIHMap = new HashMap<NodeId, Vector<IH>>();

    // a temp variable for holding the DVV map
    DVVMap tempDVVMap;

    // stores the set of all IHs unverified IHs received since last precise invalidate by any node
    // useful for creating dummy imprecise invalidates for local state modification
    HashMap<NodeId, Vector<IH>> allIHs = new HashMap<NodeId, Vector<IH>>();

    ImpreciseInv ii = null;

    // initialize lastPreciseVVMap to the startVV of the checkpoint
    //  AcceptVV lastPreciseVV = cvv.cloneAcceptVV();
    //  for(VVIterator vvIterator = chkPt.getStartVV().getIterator(); vvIterator.hasMoreElements();){
    //  NodeId nId = vvIterator.getNext();
    //  lastPreciseVVMap.put(nId, lastPreciseVV);
    //  }

    //TODO: ensure that the updates are ordered in the ordered update list in the order of their endTS

    // process all the updates in the checkpoint
    for(Object o: chkPt.orderedUpdateList){
      if(o instanceof SecurePreciseInv){
        // process secure precise invalidate

        // apply a dummy imprecise inval to advance the cvv
        if(!allIHs.isEmpty()){
          // create the insecure imprecise invalidate
//        ii = this.securityFilter.impreciseInvFilter.getInsecureImpreciseInv(allIHs);
//        giList.add(ii);

          // advance the cvv to include the endVV of the insecure imprecise invalidate
          //cvv.advanceTimestamps(ii.getEndVV());
        }
        SecurePreciseInv spi = (SecurePreciseInv)o;

        AcceptStamp as = spi.getAcceptStamp();
        if(!lastVerifiedTime.containsKey(as.getNodeId()) || lastVerifiedTime.get(as.getNodeId()) < as.getLocalClock()){ 
          if(spi.isHasHash()){
            lastVerifiedPrecise.put(spi.getCreator(), true);
            lastVerifiedTime.put(spi.getCreator(), spi.getAcceptStamp().getLocalClock());
          }
        }
	if(SecurityFilter.dbg)System.out.println("applying " + spi);
        if(this.securityFilter.preciseInvFilter.verifyPreciseInv(spi, new AcceptVV(cvv), sender)){

          if(this.securityFilter.measureTime){
            start2 = System.currentTimeMillis();
          }

          if(!spi.isHasHash()){
            spi = securityFilter.getSecurePreciseInvWithSH(spi);
          }
          
          securityFilter.apply(spi, null);
          if(this.securityFilter.measureTime){
            end2 = System.currentTimeMillis();
            LatencyWatcher.put("SecurityFilter.applyInCheckpoint", 
                (end2 - start2));
          }

          // finally reset the state
          if(SangminConfig.attachBodyToCheckpoint){
            giList.add(((SecureBoundInval)spi).getBoundInval());
          } else {
            giList.add(spi);
          }
          allIHs.clear();
          newIHMap.remove(spi.getCreator());
          cvv.advanceTimestamps(spi.getAcceptStamp());

        }else{
          assert false;
          return false;
        }



      }else{
        //process NodeAHSTuple

        assert(o instanceof NodeAHSTuple);
        //simply apply the securityFilter.hashMap and return false if it fails

        NodeAHSTuple ihMap = (NodeAHSTuple)o;
        NodeId nodeId = ihMap.getNodeId();


        ii = this.securityFilter.impreciseInvFilter.getInsecureImpreciseInv(nodeId, ihMap.getAHS().getIHs());
        giList.add(ii);
        cvv.advanceTimestamps(ii.getEndVV());

        //System.out.println("ihMap : " + ihMap);

        // Apply ih tuples for this writer
        try{
          if(this.securityFilter.measureTime){
            start2 = System.currentTimeMillis();
          }
          List<AHSEntry> ahsEntries = ihMap.getAHS().getSortedListOfEntries();

          long lastTS = ahsEntries.get(ahsEntries.size()-1).getEndTS();
          if(!lastVerifiedTime.containsKey(nodeId) || lastVerifiedTime.get(nodeId) < lastTS){ 
            lastVerifiedPrecise.put(nodeId, false);
            lastVerifiedTime.put(nodeId, lastTS);
          }

          AHSMap ahsMap = securityFilter.getAhsMap();
          for(AHSEntry ahsEntry: ahsEntries){
            if(securityFilter.impreciseInvFilter.verifyAHSEntry(nodeId, ahsEntry, sender, finalCVV, !SangminConfig.optimizeDVVMapHashes)){
		if(SecurityFilter.dbg)System.out.println("applying " + ahsEntry);
              ahsMap.applyAHS(nodeId, new AHS(securityFilter.getAHSEntryWithSH(ahsEntry)), null);
            }else{
		assert false: ahsEntry + "\n ahsMap " + ahsMap.toLongString() + " \n checkpoint " + chkPt + "\n ahsMap: " + ahsMap.toString();
              return false;
            }
          }

          if(this.securityFilter.measureTime){
            end2 = System.currentTimeMillis();
            LatencyWatcher.put("SecurityFilter.applyInCheckpoint", 
                (end2 - start2));
          }
        }catch(UnmatchingTreeNodeException e){
          e.printStackTrace();
          assert(false);
          //TODO: call controller
        }
        // put the ih tuples in the allIHs securityFilter.hashMap for creating imprecise invalidate
        Vector<IH> ihVector = ihMap.getAHS().getIHs();
        if(allIHs.containsKey(nodeId)){
          Vector<IH> v = allIHs.get(nodeId);
          v.addAll(ihVector);
        }else{
          allIHs.put(nodeId, (Vector<IH>)ihVector.clone());
        }

        // also put the tuples in each field in newIHMap. newIHMap is used for validating ih tuples on receiving precise invalidates
        if(newIHMap.containsKey(nodeId)){
          Vector<IH> v = newIHMap.get(nodeId);
          v.addAll(ihVector);
        }else{
          newIHMap.put(nodeId, (Vector<IH>)ihVector.clone());
        }

      }
    }

    if(!allIHs.isEmpty()){

      // princem: added the following
      for(NodeId nodeId: allIHs.keySet()){
        ii = this.securityFilter.impreciseInvFilter.getInsecureImpreciseInv(nodeId, allIHs.get(nodeId));
        giList.add(ii);
        cvv.advanceTimestamps(ii.getEndVV());
      }
      // create the insecure imprecise invalidate for the last set of uncovered IHs
//    ii = securityFilter.impreciseInvFilter.getInsecureImpreciseInv(allIHs);
//    giList.add(ii);
//    cvv.advanceTimestamps(ii.getEndVV());
    }
    // now check the last ih tuples

    //check that the lastVerifiedPrecise is true for all nodeIds
    for(NodeId nodeId: lastVerifiedPrecise.keySet()){
      if(reqLastPrecise && !lastVerifiedPrecise.get(nodeId)){
        assert false: "lastVerifiedPrecise:" + nodeId;
      return false;
      }
    }


    if(cvv.includes(chkPt.getEndVV())){
      // update getKnowledge()


      return true;
    }else{
      System.out.println("checkpoint verification failed as cvv inclusion test failed" + cvv + " " + chkPt.getEndVV());
      assert false;
      return false;
    }

  }

  private void updateKnowledge(SecureCheckpoint chkPt){
//  HashMap<Hole, HoleFiller> hf = chkPt.getHoleFiller();
//  for(Hole h: hf.keySet()){
//  securityFilter.getKnowledge().apply(h, hf.get(h).getInvals());
//  }

//  for(Object o: chkPt.orderedUpdateList){
//  if(o instanceof SecurePreciseInv){
//  if(o instanceof SecurePreciseInv){
//  securityFilter.getKnowledge().apply((SecurePreciseInv)o);
//  }else{
//  assert o instanceof NodeAHSTuple;
//  NodeAHSTuple nat = (NodeAHSTuple)o;
//  List<AHSEntry> ahsEntries = nat.getAHS().getSortedListOfEntries();
//  Hole h = new Hole(nat.getNodeId(), new Range(ahsEntries.get(0).getStartTS(), 
//  ahsEntries.get(ahsEntries.size()-1).getEndTS()));
//  securityFilter.getKnowledge().apply(h, ahsEntries);
//  }
//  }
//  }
  }



  


  private boolean verifyAndApplyHoleFiller(Hole hole, HoleFiller hf, AcceptVV cvv, NodeId sender){
    // verify all the components of HoleFiller
    // and ensure that the hole is compatible with our local history
    //
    // Actual approach: We will ensure that we can generate an AHS that starts and ends at the desired hole
    // then we will start applying the components of the HoleFiller assuming compatibility but after verifying
    // them individually. If an incompatibility occurs, the application would be aborted

    AHSMap ahsMap =securityFilter.getAhsMap();
    TreeNode tnStart = ahsMap.getTreeNodeTS(hole.getNodeId(), hole.getRange().getStart());
    TreeNode tnEnd = ahsMap.getTreeNodeTS(hole.getNodeId(), hole.getRange().getEnd());
    if(tnStart == null || tnEnd == null || tnStart.getStartTS() != hole.getRange().getStart() || tnEnd.getEndTS() != hole.getRange().getEnd()){
      assert false;
      return false;
    }

    // continue application
    for(Object o: hf.getInvals()){
      if(o instanceof AHSEntry){
        if(!securityFilter.impreciseInvFilter.verifyAHSEntry(hole.getNodeId(), (AHSEntry)o, sender, cvv, !SangminConfig.optimizeDVVMapHashes)){
          assert false;
          return false;
        }else{
          try{
            AHSEntry ahsEntry = securityFilter.getAHSEntryWithSH((AHSEntry)o);
            ahsMap.applyAHS(hole.getNodeId(), new AHS(ahsEntry), null);
          }catch(UnmatchingTreeNodeException e){
            // TODO Auto-generated catch block
            e.printStackTrace();
            assert false;
            return false;
          }
        }
      }else if(o instanceof SecurePreciseInv){
        if(!this.securityFilter.preciseInvFilter.verifyPreciseInv((SecurePreciseInv)o, cvv, sender)){
          assert false;
          return false;
        }else{
          try{
            SecurePreciseInv spi = (SecurePreciseInv)o;
            if(!spi.isHasHash()){
              spi = securityFilter.getSecurePreciseInvWithSH(spi);
            }
            ahsMap.applySecurePreciseInv(spi);
          }catch(UnmatchingTreeNodeException e){
            // TODO Auto-generated catch block
            e.printStackTrace();
            assert false;
            return false;
          }
        }
      }else{
        return false;
      }
    }

    return true;
  }

////update LPVV state based on the received checkpoint. Assume that the checkpoint is correct.
////startVV denotes the inclusion startVV for the checkpoint based on our local logs

////No lock shoud be held before calling updateState 
////Refer to OutgoingConnectionMailbox.newUpdate()
//private void updateState(AcceptVV startVV, LinkedList<GeneralInv> giList, SecureCheckpoint chkPt, ConnectionState sic){
////create a CatchupStreamStartMsg and call apply GI for each update after converting these updates into a valid update
//AcceptVV cssmStartVV = securityFilter.getAhsMap().getPrceedingVV(chkPt.getStartVV());

//CatchupStreamStartMsg cssm = new CatchupStreamStartMsg(chkPt.ss, cssmStartVV);
////sic.applyCatchupStreamStartMsg(cssm);

//assert sic instanceof SecureIncomingInvalConnection;
//((SecureIncomingInvalConnection)sic).applyCheckpointCatchupStreamStartMsg(cssm, chkPt.getEndVV());
////apply the msgs from the checkpoint now
//for(GeneralInv gi: giList){
////System.out.println("applyCatchupGi called for " + gi);
//sic.applyInsecureGI(gi);
////((SecureIncomingInvalConnection)sic).applyInsecureGI(gi);
//}
//sic.applyCatchupStreamEndMsg(cssm);
//}

  /**
   * New checkpoint update func to update state based on a secure checkpoint
   * @param startVV
   * @param giList
   * @param chkPt
   * @param sic
   */
  void updateState(AcceptVV startVV, LinkedList<GeneralInv> giList, SecureCheckpoint chkPt, SecureConnectionState sic){
    sic.applySecureCheckpointUpdates(startVV, giList, chkPt);
  }

  /**
   * create an iterator from startVV to endVV (inclusive)
   * @param startVV
   * @param endVV
   * @return
   */
  Iterator<NodeTreeNodeTuple> allUpdatesIterator(AcceptVV startVV, AcceptVV endVV){
    SecureReverseTreeNodeIterator rii = new SecureReverseTreeNodeIterator(endVV, securityFilter.getAhsMap());

    LinkedList<NodeTreeNodeTuple> allUpdateList = new LinkedList<NodeTreeNodeTuple>();
    // the counter starts from endVV and goes till startVV
    CounterVV receivedMinVV = new CounterVV(endVV);
    // because we want to use the !includes check
    AcceptVV lastStartVV = AcceptVV.decrementAll(startVV);

    NodeTreeNodeTuple curInvFromLog = null;
    if(SecurityFilter.dbg){
      System.out.println("startVV : " + startVV + " endVV " + endVV);
    }

    // keep iterating until all entries covered or if the log is empty
    while(rii.hasPrev() && 
        !lastStartVV.includes(receivedMinVV)){
      curInvFromLog = rii.prev();
      receivedMinVV.setStampByServer(curInvFromLog.getNodeId(), curInvFromLog.getTreeNode().getStartTS());
      AcceptStamp as = new AcceptStamp(curInvFromLog.getTreeNode().getEndTS(),curInvFromLog.getNodeId());
      //princem: I also think the following checek should be removed to ensure a causally consistent checkpoint
      if(!lastStartVV.includes(as) && endVV.containsNodeId(as.getNodeId())){
        allUpdateList.addFirst(curInvFromLog);
      }else{
        if(SecurityFilter.dbg){
          System.out.println("rejecting : " + as);
        }
      }
    }

    ListIterator<NodeTreeNodeTuple> l = allUpdateList.listIterator();
    NodeTreeNodeTuple ntnt = null;
    if(l.hasNext()){
      ntnt = l.next();
    }
    while(l.hasNext()){
      NodeTreeNodeTuple ntnt2 = l.next();
      assert(ntnt.compareTo(ntnt2) < 0);
      ntnt = ntnt2;
    }


    //LinkedList<NodeTreeNodeTuple> allUpdateList2 = (LinkedList<NodeTreeNodeTuple>)allUpdateList.clone();
    //Collections.sort((List)allUpdateList2);
    //assert(allUpdateList.equals(allUpdateList2)):"original :" + allUpdateList + "\nsorted :" + allUpdateList2;

    //  Iterator<NodeTreeNodeTuple> testIter = allUpdateList.iterator();
    //  Iterator<NodeTreeNodeTuple> optiIter = this.makeSecureTreeNodeIterator(AcceptVV.decrementAll(startVV));
    //  while(testIter.hasNext()){
    //  assert optiIter.hasNext();
    //  NodeTreeNodeTuple testntnt = testIter.next();
    //  NodeTreeNodeTuple optintnt = optiIter.next();
    //  assert testntnt.equals(optintnt):"testIter:"+testntnt+" optiIter:" + optintnt;
    //  if(!testntnt.equals(optintnt)){
    //  System.out.println("testIter:"+testntnt+" optiIter:" + optintnt);
    //  }
    //  }

    //  assert !optiIter.hasNext():"This assertion may fail if concurrent threads are updating and reading from " +
    //  "the AHSMap (i.e. in realisitic settings). However, this must pass in single threaded system. " +
    //  "Comment it in multithreaded systems";
    if(SecurityFilter.dbg){
      System.out.println("allUpdatesiterator: " + allUpdateList);
    }
    return allUpdateList.iterator();
    /*
    //ensure that startVV contains -1 for all entries for which we have an entry in endVV
    CounterVV cvv = new CounterVV(startVV);
    VVIterator vvi = endVV.getIterator();
    while(vvi.hasMoreElements()){
      NodeId n = vvi.getNext();
      if(!cvv.containsNodeId(n)){
        cvv.setStampByServer(n, AcceptVV.BEFORE_TIME_BEGAIN);
      }
    }
    return new SecureTreeNodeIterator(new AcceptVV(cvv), securityFilter.getAhsMap());*/
  }


  private void addSplitPoint(NodeId n, long ts){
    if(!globalSplitPoints.containsKey(n)){
      globalSplitPoints.put(n, new TreeSet<Long>());
    }
    globalSplitPoints.get(n).add(ts);
  }
  
  private boolean isSplit(NodeId n, long ts){
    if(!globalSplitPoints.containsKey(n)){
      return false;
    }else{
      return globalSplitPoints.get(n).contains(ts);
    }
  }
  
  public HashMap<Hole, HoleFiller> getHoleFillers(FilterKnowledge fk){
    HashMap<Hole, HoleFiller> hfs = new HashMap<Hole, HoleFiller>();
    Hashtable<NodeId, TreeSet<Range>> holes = fk.getHoles();
    AHSMap ahsMap = securityFilter.getAhsMap();
    for(NodeId nId:holes.keySet()){
      for(Range r: holes.get(nId)){
        addSplitPoint(nId, r.getEnd());
        boolean precise = false;
        int count = 0;
        TreeNode startTN= ahsMap.getTreeNodeTS(nId, r.getStart());
        if(startTN !=null && startTN.getStartTS() == r.getStart()){
          TreeNode endTN = ahsMap.getTreeNodeTS(nId, r.getEnd());
          if(endTN !=null && endTN.getEndTS() == r.getEnd()){
            HoleFiller hf = new HoleFiller();
            long lastStart = -1;
            long lastEnd = -1;
            while(startTN !=null && startTN.getEndTS()<=r.getEnd()){
              if(fk.getFilter().isPresent(nId, new AHSEntry(startTN))){
                if(lastStart != -1){
                  List<AHSEntry> ahsEntries = ahsMap.createAHS(nId, lastStart, lastEnd).getSortedListOfEntries();
                  for(AHSEntry ahsEntry: ahsEntries){
		      //                    hf.add(ahsEntry);
		    hf.add(getAHSEntryWithoutLocalSH(ahsEntry, nId));
                    addSplitPoint(nId, ahsEntry.getEndTS());
                    count++;
                  }  
                }
                count++;
                if(startTN.getInv() instanceof PreciseInv){
                  precise = true;
                  hf.add(securityFilter.getSecurePreciseInvWithoutSHAndSignature((SecurePreciseInv)startTN.getInv()));
                  addSplitPoint(nId, startTN.getEndTS());
                }else{
                  addSplitPoint(nId, startTN.getEndTS());
		  hf.add(getAHSEntryWithoutLocalSH(new AHSEntry(startTN), nId));
		  //                  hf.add(new AHSEntry(startTN));
                }
                lastStart = -1;
                lastEnd = -1;
              }else{
                if(lastStart == -1){
                  lastStart = startTN.getStartTS();
                }
                lastEnd = startTN.getEndTS();
              }
              startTN = startTN.getNext();
            }

                if(lastStart != -1){
                  List<AHSEntry> ahsEntries = ahsMap.createAHS(nId, lastStart, lastEnd).getSortedListOfEntries();
                  for(AHSEntry ahsEntry: ahsEntries){
		      //                    hf.add(ahsEntry);
                    hf.add(getAHSEntryWithoutLocalSH(ahsEntry, nId));
                    addSplitPoint(nId, ahsEntry.getEndTS());
                    count++;
                  }  
                }
            if(count > 1 || precise){
              hfs.put(new Hole(nId, r), hf);
              long min = Long.MAX_VALUE;
              long max = Long.MIN_VALUE;
              for(Object o: hf.getInvals()){
                if(o instanceof SecurePreciseInv){
                  SecurePreciseInv spi = (SecurePreciseInv)o;
                  min = (min > spi.getEnd())?spi.getEnd():min;
                  max = (max < spi.getEnd())?spi.getEnd():max;
                }else{
                  AHSEntry aEntry = (AHSEntry)o;
                  min = (min > aEntry.getStartTS())?aEntry.getStartTS():min;
                  max = (max < aEntry.getEndTS())?aEntry.getEndTS():max;
                }
              }
              assert hf.getInvals().size() > 1 || precise: hf;
              assert min == r.getStart() && max == r.getEnd(): " min " + min + " max " + max + " range " + r;
            }
          }
        }

      }
    }
    
    return hfs;
  }

  //TODO: princem: Can optimize by not using the SecureInMemLogIterator
  /** 
   *  getSecureCheckpoint() 
   *  startVV: inclusion startVV 
   *  endVV: inclusion endVV 
   **/ 
  SecureCheckpoint getSecureCheckpoint(AcceptVV startVV, 
      boolean withBody,
      AcceptVV endVV, 
      /*      boolean includeAll,*/
      LivenessFilter lf, 
      Filter f, 
      FilterKnowledge fk){

    long startTime=0,endTime=0, startTime2=0, endTime2=0;

    if(this.securityFilter.measureTime){
      startTime = System.currentTimeMillis();
    }

//  assert endVV.includes(startVV):" startVV " + startVV + " endVV" + endVV;
    //System.out.println("startVV : " + startVV);  
    //System.out.println("endVV: " + endVV);  
    //assert !startVV.includes(endVV); // what is the right assertion here?
    //System.out.println("SubscriptionSet: " + ss);  
    SecureCheckpoint chkPt = new SecureCheckpoint(startVV, endVV, SubscriptionSet.makeEmptySet(), securityFilter.core.getMyNodeId());
    this.globalSplitPoints.clear();
    HashMap<Hole, HoleFiller> hf = getHoleFillers(fk);
    chkPt.setHoleFiller(hf);
    HashMap<NodeId, SecurePreciseInv> lastPrecise = new HashMap<NodeId, SecurePreciseInv>();

    LinkedList orderedUpdateList = new LinkedList();

    if(endVV.includes(startVV)){// && !startVV.includes(endVV)){

      LinkedList<PreciseInv> lastUpdateList = getLastUpdateList(startVV, endVV/*, includeAll*/, lf, f);

      if(SecurityFilter.dbg){
	  for(NodeId n: globalSplitPoints.keySet()){
	      System.out.print("NodeId < " + n + " > : ");
	      for(Long l: globalSplitPoints.get(n)){
		  System.out.print(l + " ");
	      }
	      System.out.println();
	  }
      }


      //  if(securityFilter.core.getMyNodeId().equals(new NodeId(4))){
//    System.out.println("LastUpdateList: " + lastUpdateList);
//    }

      if(this.securityFilter.measureTime){
        endTime = System.currentTimeMillis();
        LatencyWatcher.put("SecurityFilter.getLastUpdateList", 
            (endTime - startTime));
      }

      if(SecurityFilter.dbg){
        System.out.println("lastUpdateList.toString()");   
        System.out.println(lastUpdateList.toString());   
      }

      /**
       * the decrement all is to provide the secureInMemLog with the exclusion VV
       */
      //SecureInMemLogIterator sii = this.makeSecureInMemLogIterator(AcceptVV.decrementAll(startVV));
      if(this.securityFilter.measureTime){
        startTime2 = System.currentTimeMillis();
      }
      Iterator<NodeTreeNodeTuple> sii = allUpdatesIterator(startVV, endVV);
      if(this.securityFilter.measureTime){
        endTime2 = System.currentTimeMillis();
        LatencyWatcher.put("SecurityFilter.allUpdatesIterator", 
            (endTime2 - startTime2));
      }
      /*
	     System.out.println("AllUpdatesIterator");   
	     while(sii.hasNext()){
	       System.out.println(sii.next().toString());
	     }
	     sii = this.allUpdatesIterator(startVV, endVV);
       */
      NodeTreeNodeTuple curNodeFromLog = null;

      PreciseInv curStopSignInv = null;
      AcceptStamp curStopSignInvAS = null;
      if(lastUpdateList.size() > 0){
        curStopSignInv = lastUpdateList.removeFirst();
        curStopSignInvAS = curStopSignInv.getAcceptStamp();
      }
      else{
        curStopSignInv = null;
      }
      CounterVV accuIIStartVV = new CounterVV(); // the startVV of the next accumulated II
      CounterVV accuIIEndVV = new CounterVV(); // the endVV of the next accumulated II
      CounterVV curVV = new CounterVV(); // the VV summarizing all the updates we have seen so far 
      CounterVV realStartVV = new CounterVV(); // the VV summarizing all the updates we have seen so far 

      while(sii.hasNext() && !(curVV.includes(endVV))){

        curNodeFromLog = sii.next();
        if(SecurityFilter.dbg){
          System.out.println("### CurrentStopSignInv : " + curStopSignInv);
          System.out.println("### CurrentNodeFromLog : " + curNodeFromLog);
        }
//      if(securityFilter.core.getMyNodeId().equals(new NodeId(4))){
//      System.out.println("### CurrentStopSignInv : " + curStopSignInv);
//      System.out.println("### CurrentNodeFromLog : " + curNodeFromLog);
//      }

        AcceptStamp as = new AcceptStamp(curNodeFromLog.getTreeNode().getEndTS(), curNodeFromLog.getNodeId());
        if(!realStartVV.containsNodeId(curNodeFromLog.getNodeId())){
          realStartVV.setStamp(new AcceptStamp(curNodeFromLog.getTreeNode().getStartTS(), curNodeFromLog.getNodeId()));
        }
        // the following is just trying to extract the accept stamp for the creator
        //curVV.advanceTimestamps(curNodeFromLog.getEndVV());
        curVV.advanceTimestamps(as);

        if(curStopSignInv != null &&
            curVV.includes(curStopSignInvAS)){
          // main algorithm
          //if(curStopSignInv instanceof DummyPreciseInv){
          // set the IH start and end ts fields appropriately
          //   accuIIStartVV.addMinVV(curNodeFromLog.getStartVV());
          //   accuIIEndVV.addMaxVV(curNodeFromLog.getEndVV());
          //add the AHS for the portion from accuIIStartVV[curNodeFromLog.getCreator]-accuIIEndVV[curNodeFromLog.getCreator]
          // reseet accuIIStartVV[creator], accuIIEndVV[creator]
          //}else{
          // precise inv
          //add the AHS for portion from accuIIStartVV[curNodeFromLog.getCreator]-accuIIEndVV[curNodeFromLog.getCreator]
          //reset accuIIStartVV[creator], accuIIEndVV[creator]
          // add the pi to the checkpoint
          //}

//        if(securityFilter.core.getMyNodeId().equals(new NodeId(4))){
//        System.out.println("found split point: " + curStopSignInv);
//        }

          // set the IH start and end ts fields appropriately
          if(curStopSignInv instanceof DummyPreciseInv){
            //	accuIIStartVV.addMinVV(curNodeFromLog.getStartVV());
            //	accuIIEndVV.addMaxVV(curNodeFromLog.getEndVV());
            accuIIStartVV.addMinAS(new AcceptStamp(curNodeFromLog.getTreeNode().getStartTS(), curNodeFromLog.getNodeId()));
            accuIIEndVV.addMaxAS(new AcceptStamp(curNodeFromLog.getTreeNode().getEndTS(), curNodeFromLog.getNodeId()));
          }

          NodeId nodeId = curStopSignInv.getNodeId();

          //send the IH tuples
          if(accuIIEndVV.getSize() != 0 && accuIIEndVV.containsNodeId(nodeId)){
            assert accuIIEndVV.includes(accuIIStartVV);


            try{
//            if(securityFilter.core.getMyNodeId().equals(new NodeId(4))){
//            System.out.println("taking care of split point: " + curStopSignInv + " " + accuIIEndVV + " " + accuIIStartVV);
//            }
              if(SecurityFilter.dbg){
                System.out.println("taking care of split point: " + curStopSignInv + " " + accuIIEndVV + " " + accuIIStartVV);
              }

              long start = accuIIStartVV.getStampByServer(nodeId);
              long end =  accuIIEndVV.getStampByServer(nodeId);
              assert curStopSignInv.getEnd() == curNodeFromLog.getTreeNode().getEndTS():curStopSignInv + " doesn't agree with " + curNodeFromLog;
              AHS ahs = getAHSWithoutSH(securityFilter.getAhsMap().createAHS(nodeId, start, end));
              NodeAHSTuple accumulatedII = new NodeAHSTuple(nodeId, ahs);
              List<AHSEntry> ahsList = ahs.getSortedListOfEntries();
              assert ahsList.size() > 0:"ahslist of size 0 obtaied " + start + " " + end + " " + nodeId;
              assert ahsList.get(0).getStartTS() == start && ahsList.get(ahsList.size() - 1).getEndTS() == end:ahsList + 
              " doesn't agree with " + start + " "+ end+ " "+ nodeId;

              orderedUpdateList.addLast(accumulatedII);
              
              accuIIStartVV.dropNode(nodeId);
              accuIIEndVV.dropNode(nodeId);
            }catch(NoSuchEntryException e){
              assert false;
            }

          }else{
//          if(securityFilter.core.getMyNodeId().equals(new NodeId(4))){
//          System.out.println("strange condition: found split point: " + curStopSignInv + " " + accuIIEndVV + " " + accuIIStartVV);
//          }
            // can only happen when no other updates received between two consecutive precise invalidates
            assert curStopSignInv instanceof SecurePreciseInv: curStopSignInv;

          }
          //add the split point if its not a dummy update (in other words if it is a precise update)
          if(curStopSignInv != null && !(curStopSignInv instanceof DummyPreciseInv)){
            if(SangminConfig.attachBodyToCheckpoint){
              BodyMsg bodyMsg = null;
              try{
                bodyMsg =
                  securityFilter.core.read(curStopSignInv.getObjId(),
                      curStopSignInv.getOffset(),
                      curStopSignInv.getLength(),
                      false, false, false, 3000);
              }catch(EOFException e){
                e.printStackTrace();
                assert(false);
              }catch(ObjNotFoundException e){
                e.printStackTrace();
                assert(false);
              }catch(IOException e){
                e.printStackTrace();
                assert(false);
              }catch(ReadOfInvalidRangeException e){
                e.printStackTrace();
                assert(false);
              }catch(ReadOfHoleException e){
                e.printStackTrace();
                assert(false);
              }

              curStopSignInv =
                new SecureBoundInval((SecurePreciseInv) curStopSignInv,
                    bodyMsg.getBody());
            }          

            SecurePreciseInv spi =(SecurePreciseInv)curStopSignInv;
            orderedUpdateList.addLast(securityFilter.getSecurePreciseInvWithoutSHAndSignature(spi));
            lastPrecise.put(nodeId, spi);
            
          }

          if(lastUpdateList.size() > 0){
            curStopSignInv = lastUpdateList.removeFirst();
            curStopSignInvAS = curStopSignInv.getAcceptStamp();
          }
          else{
            curStopSignInv = null;
          }
        }else{
          //	accuIIStartVV.addMinVV(curNodeFromLog.getStartVV());
          //	accuIIEndVV.addMaxVV(curNodeFromLog.getEndVV());
          accuIIStartVV.addMinAS(new AcceptStamp(curNodeFromLog.getTreeNode().getStartTS(), curNodeFromLog.getNodeId()));
          accuIIEndVV.addMaxAS(new AcceptStamp(curNodeFromLog.getTreeNode().getEndTS(), curNodeFromLog.getNodeId()));
        }

      }

      if(accuIIEndVV.getSize() != 0){
        Vector<NodeAHSTuple> accumulatedII = securityFilter.getAhsMap().generateAHS(accuIIStartVV, accuIIEndVV);
        for(NodeAHSTuple t: accumulatedII){
          orderedUpdateList.addLast(new NodeAHSTuple(t.getNodeId(), getAHSWithoutSH(t.getAHS())));
        }
      }
      
      for(Object o: orderedUpdateList){
        if(o instanceof SecurePreciseInv){
          SecurePreciseInv spi = (SecurePreciseInv)o;
          if(lastPrecise.get(spi.getNodeId()).getAcceptStamp().equals(spi.getAcceptStamp())){
            chkPt.addUpdate(lastPrecise.get(spi.getNodeId()));
          }else{
            chkPt.addUpdate(spi);
          }
        }else{
          chkPt.addUpdate((NodeAHSTuple)o);
        }
        
      }
      

      AcceptVV acceptCurVV = new AcceptVV(curVV);

      if(!chkPt.getStartVV().includes(startVV)){
        System.out.println("Excess data: required " + startVV + " present " + chkPt.getStartVV());
      }
      if(SecurityFilter.dbg){
        System.out.println("startVV : "  +startVV.dropNegatives() + " endVV " + endVV.dropNegatives() + "realStartVV " + realStartVV + " acceptCurVV " + acceptCurVV.dropNegatives()  + 
            " chkPt: startVV " + chkPt.getStartVV().dropNegatives() + " endVV " +chkPt.getEndVV().dropNegatives());
        System.out.println("checkpoint: " + chkPt);
      }
      assert acceptCurVV.equalsIgnoreNegatives(chkPt.getEndVV()): "AcceptCurVV: " + acceptCurVV + " endVV " + chkPt.getEndVV() + " chkPt " + chkPt;
      assert acceptCurVV.includes(endVV): "acceptCurVV: "  +acceptCurVV + " endVV " + endVV + " chkPt " + chkPt;
      assert realStartVV.equals(chkPt.getStartVV()): " realStartVV " + realStartVV + " startVV " + chkPt.getStartVV();
      //AcceptVV relevantVV = AcceptVV.decrementAll(acceptCurVV.getRealDiff(startVV)).dropNegatives();

      // chkPt.setEndVV(acceptCurVV.project(relevantVV));
      //chkPt.setStartVV(startVV.project(relevantVV));

      //princem: need to fix these
      //chkPt.setEndVV(acceptCurVV);
      //chkPt.setStartVV(new AcceptVV(realStartVV));

      assert (!chkPt.getStartVV().includes(chkPt.getEndVV()) || chkPt.getStartVV().equalsIgnoreNegatives(chkPt.getEndVV())): "startVV"+chkPt.getStartVV()+" "+chkPt.getEndVV() ;
      assert chkPt.getEndVV().includes(chkPt.getStartVV());
      assert securityFilter.getCurrentVV().includes(chkPt.getEndVV()):"can't create checkpoint beyond my cvv" + securityFilter.getCurrentVV() + " " + chkPt.getEndVV();


    }
    if(this.securityFilter.measureTime){
      endTime = System.currentTimeMillis();
      LatencyWatcher.put("SecurityFilter.SecureCheckpointCreation", 
          (endTime - startTime));
    }
    VVIterator vvi = chkPt.getEndVV().getIterator();
    while(vvi.hasMoreElements()){
      NodeId n = vvi.getNext();
      assert lastPrecise.containsKey(n): n + " chkPt " + chkPt + " ahsMap " + securityFilter.getAhsMap().toLongString();
    }
    return chkPt;
  }



  /**
   * Merge precise invalidtes present in ret and the split stamps under the constraint that nothing older than exclusionVV should 
   * be included
   * @param ret
   * @param splitStamps
   * @param exclusionVV
   * @return
   */
  LinkedList<PreciseInv> merge(LinkedList<PreciseInv> ret, TreeSet<PreciseInv> splitStamps, AcceptVV exclusionVV){
    Iterator<PreciseInv> splitIter = splitStamps.iterator();
    Iterator<PreciseInv> preciseIter = ret.iterator();
    LinkedList<PreciseInv> returnSplitPoints = new LinkedList<PreciseInv>();
    PreciseInv curSplit = null;
    PreciseInv curPreciseInv = null;

    // initialize curSplit
    if(splitIter.hasNext()){
      curSplit = splitIter.next();
    }

    //advance until curSplit > exclusionVV: basically throwing away stuff that we don't need
    while(curSplit != null && exclusionVV.includes(curSplit.getAcceptStamp())){
      if(splitIter.hasNext()){
        curSplit = splitIter.next();
      }else{
        curSplit = null;
      }
    }

    if(preciseIter.hasNext()){
      curPreciseInv = preciseIter.next();
      assert !(exclusionVV.includes(curPreciseInv.getAcceptStamp()));
    }

    while(curSplit != null && curPreciseInv != null){
      int comp = curSplit.compareTo(curPreciseInv);
      if(comp < 0){
        returnSplitPoints.add(curSplit);
        if(splitIter.hasNext()){
          curSplit = splitIter.next();
        }else{
          curSplit = null;
        }
      }else{
        returnSplitPoints.add(curPreciseInv);
        if(preciseIter.hasNext()){
          curPreciseInv = preciseIter.next();
        }else{
          curPreciseInv = null;
        }
        if(comp == 0){ // don't need to add splitpoints for ts covered by precise invalidates
          if(splitIter.hasNext()){
            curSplit = splitIter.next();
          }else{
            curSplit = null;
          }
        }
      }
    }

    while(splitIter.hasNext()){
      returnSplitPoints.add(curSplit);
      curSplit = splitIter.next();
    }

    if(curSplit != null){
      returnSplitPoints.add(curSplit);
    }

    while(preciseIter.hasNext()){
      returnSplitPoints.add(curPreciseInv);
      curPreciseInv = preciseIter.next();
    }

    if(curPreciseInv != null){
      returnSplitPoints.add(curPreciseInv);
    }
    if(SecurityFilter.sanityCheck){
    LinkedList test = new LinkedList(returnSplitPoints);
    Collections.sort(test);
    assert test.equals(returnSplitPoints):"Sorted List: " + test + " merged one " + returnSplitPoints;
    }
    return returnSplitPoints;
  }

  /**
   * Can this secureCheckpoint's dvv dependency be satisfied based on current state and potentially by 
   * fetching information from other nodes based on liveness policy?
   * @param secureCheckpoint
   * @param sender
   * @return
   */
  boolean getMaxDVV(SecureCheckpoint secureCheckpoint, NodeId sender){

    AcceptVV cvv = securityFilter.getCurrentVV();
    AcceptVV vv = cvv.cloneMaxVV(secureCheckpoint.getEndVV());
    VV startVV = secureCheckpoint.getStartVV();
    VVIterator vvi = secureCheckpoint.getEndVV().getIterator();
    /*
    HashMap<NodeId, DependencyVV> dvvMapMap = secureCheckpoint.getMaxDVVMap();
    //System.out.println("dvvMap: " + dvvMapMap);
    // For each writer in sii
    while(vvi.hasMoreElements()){
      Object token = vvi.getNext();
      NodeId nodeId = startVV.getServerByIteratorToken(token);
      DependencyVV maxDVV = dvvMapMap.get(nodeId);
      //verify each dvvMap component
      if(!vv.includes(maxDVV)){
        assert false;
        return false;
      }

      //check if the DVV is split for both 1) values included in imprecise inval and 
      // lcoal values. If not for local values, call rmiController
      // if not for included values. return false;
      AcceptVV borderStartVV = cvv;
      vvi = borderStartVV.getIterator();
      try{
        while(vvi.hasMoreElements()){
          nodeId = vvi.getNext();
          if(maxDVV.containsNodeId(nodeId)){
            if(borderStartVV.getStampByIteratorToken(nodeId) < maxDVV.getStampByServer(nodeId)){
              // princem: added the following check as I think this condition should not occur
              //assert false: "maxDVV " + maxDVV + " cvv " + cvv  + " nodeId " + nodeId + " \n " + secureCheckpoint;
              maxDVV.dropNode(nodeId);
            }
          }
        }
      }catch(NoSuchEntryException e){
        // TODO Auto-generated catch block
        e.printStackTrace();
        assert false;
      }

      // check if the maxDVV is split?
      // find tree node containing 


      System.out.println("ensuring verifiability of " + maxDVV);
      if(!securityFilter.getLivenessFilter()(maxDVV, sender)){
        assert false;
        return false;
      }

    }
     */
    // check if accepting this checkpoint implies that i am accepting concurrent overlapping updates 
    // that i can't verify to be from the same history: if yes, then try to break the existing updates/ or the 
    // checkpoint to ensure compatibility

    //approach: 4 possibilities for any nodeId component:
    // 1> startVV >= CVV: Do nothing
    // 2> startVV < CVV and endVV > CVV: ensure that the checkpoint is split across endVV or get a split secure ii
    // 3> endVV = CVV: do nothing
    // 4> CVV > endVV: ensure that current state is split across endVV
    // We deal with 4th case here and deal with the 2nd case after applying the checkpoint
    // so we need to find out all the VV entries where CVV > endVV

    DependencyVV dvv = securityFilter.getSplitEndVVBetweenStartAndCVV(secureCheckpoint.getStartVV(), 
        secureCheckpoint.getEndVV(), securityFilter.getCurrentVV());
    if(SecurityFilter.dbg)System.out.println("case 4: ensuring verifiability of " + dvv + " startVV " + 
        secureCheckpoint.getStartVV() + " endVV " + secureCheckpoint.getEndVV() + " cvv" + securityFilter.getCurrentVV());


    if(!securityFilter.getLivenessFilter().ensureCompatibility(dvv, securityFilter.getAhsMap(), sender)){
      assert false;
      return false;
    }


    //System.out.println("returning true");
    return true;
  }

  /**
   * returns a list of last updates and all the split points in that range
   * the split points are the set of points which must NOT be included in an imprecise invalidate 
   * as validation of some precise or imprecise invalidate depends on those points being split
   * For a range of uniwrites ihs, split points can be calculated by getting a maxDVV/writer's component over that range. 
   * @param startVV inclusion
   * @param ss
   * @param endVV inclusion
   * @return
   */
  LinkedList<PreciseInv> getLastUpdateList(AcceptVV startVV,  
      SubscriptionSet ss, AcceptVV endVV){

    return getLastUpdateList(startVV, endVV/*, false*/, new BlockingLivenessFilter(), new SubscriptionSetFilter(ss));
  }


  //TODO:princem: this function can be highly optimized by embedding the maxDVV computation within the traversal by maintaining running maxDVVs
  /**
   * returns a list of last updates and all the split points in that range
   * the split points are the set of points which must NOT be included in an imprecise invalidate 
   * as validation of some precise or imprecise invalidate depends on those points being split
   * For a range of uniwrites ihs, split points can be calculated by getting a maxDVV/writer's component over that range.
   * Ensure that only the split points belonging to the nodes present in endVV are included 
   * @param startVV inclusion
   * @param ss
   * @param endVV inclusion
   * @return
   */
  LinkedList<PreciseInv> getLastUpdateList(AcceptVV startVV,  
      AcceptVV endVV/*, boolean includeAll*/, LivenessFilter lf, Filter f){

    try{
      assert securityFilter.getAhsMap().isConsistent();
      endVV = endVV.dropNegatives();
      startVV = startVV.dropNegatives();
      SecureReverseTreeNodeIterator rii = new SecureReverseTreeNodeIterator(endVV, securityFilter.getAhsMap());


      HashMap<NodeId, Boolean> lastPrecise = new HashMap<NodeId, Boolean>(); 

      /**
       * this tracks the endVV of last inval to be added as a split point for imprecise holes
       */
      CounterVV prevVV = new CounterVV();

      // the counter starts from endVV and goes till startVV
      CounterVV receivedMinVV = new CounterVV(endVV);
      // because we want to use the !includes check
      AcceptVV lastStartVV = AcceptVV.decrementAll(startVV).dropNegatives();
      // this hash map stores the last updates for each objects
      HashMap<ObjId, TreeNode> lastPreciseUpdates = new HashMap<ObjId, TreeNode>();
      
      //this hash map tracks the DVV for which the split points have not been added yet..i.e. the DVV corresponding to the AHS 
      // entries between consecutive relevant precise invals from the same writer
      HashMap<NodeId, DependencyVV> dvvTracker = new HashMap<NodeId, DependencyVV>();
      HashMap<NodeId, AcceptVV> lastAddedDVVTracker = new HashMap<NodeId, AcceptVV>();
      VVIterator vvi = endVV.getIterator();
      while(vvi.hasMoreElements()){
        NodeId n = vvi.getNext();
        dvvTracker.put(n, new DependencyVV());
        lastAddedDVVTracker.put(n, new AcceptVV());
        if(this.reqLastPrecise){
          lastPrecise.put(n, false);
        }else{
          lastPrecise.put(n, true);
        }
      }

      // stores the precise invalidates to be returned
      LinkedList<PreciseInv> ret = new LinkedList<PreciseInv>();

      //sorted set to maintain the points of splits
      TreeSet<PreciseInv> splitStamps = new TreeSet<PreciseInv>();

      AcceptVV maxDVV = null; 

      NodeTreeNodeTuple curInvFromLog = null;
      NodeId nId;
      ObjInvalTarget it = null;


      // keep iterating until all entries covered or if the log is empty
      while(rii.hasPrev() && 
          !lastStartVV.includes(receivedMinVV)){
        curInvFromLog = rii.prev();

        if(SecurityFilter.dbg){

          System.out.println("### receivedMinVV Before"+receivedMinVV);
          System.out.println("### curInvFromLog : "+ curInvFromLog);
        }
        receivedMinVV.setStampByServer(curInvFromLog.getNodeId(), curInvFromLog.getTreeNode().getStartTS());
        nId = curInvFromLog.getNodeId();

        if(prevVV.containsNodeId(nId)){
	    AcceptStamp as = new AcceptStamp(prevVV.getStampByServer(nId), nId);
            if(!lastStartVV.includes(as) && endVV.containsNodeId(nId)){
		splitStamps.add(new DummyPreciseInv(as));
                addSplitPoint(as.getNodeId(), as.getLocalClock());

	    }
        }

        // if the treenode we are considering is precise
        // and is useful for ss and  
        // we don;t already have an invalidate for this object
        AHSEntry ahsEntry = new AHSEntry(curInvFromLog.getTreeNode());
        if(f.isPresent(nId, ahsEntry) || 
            !lf.isPresent(nId, ahsEntry) ||
            (curInvFromLog.getTreeNode().getInv() instanceof SecurePreciseInv && 
                !lastPrecise.get(nId))){

//        System.out.println(f + "f.isPresent" + f.isPresent(nId, ahsEntry));
//        System.out.println(lf + "lf.isPresent" + lf.isPresent(nId, ahsEntry));
//        System.out.println("lastPrecise " + (!lastPrecise.containsKey(nId) || !lastPrecise.get(nId)));

          // if last preVV is not null then add that splitPoint
          // if precise and interested: do whatever we were doing earlier and remove that splitPoint
          // if imprecise and interested, reset prevVV to the end of the ahsEntry
          // if imprecise and not interested and prevVV contains an entry then clear prevVV and add the end of the imprecise to the splitpoints
          
          
          
          
          if(curInvFromLog.getTreeNode().getInv() instanceof SecurePreciseInv){
//          ((it = (ObjInvalTarget)curInvFromLog.getTreeNode().getInvalTarget()).intersects(ss) &&
//          lf.shouldSummarize(curInvFromLog.getTreeNode().getInv().getEndVV()) &&
//          !lastPreciseUpdates.containsKey(it.getObjId())) || includeAll){

            
            
            prevVV.dropNode(nId);

            lastPrecise.put(nId, true);

            it = (ObjInvalTarget)curInvFromLog.getTreeNode().getInvalTarget();

            assert curInvFromLog.getTreeNode().getLevel() == 0;
            // we found a precise inval for "it"
            // add it to the data structure indicating that it has been covered
            // also add it to the list of stuff to be sent/split
            lastPreciseUpdates.put(it.getObjId(), curInvFromLog.getTreeNode());
            SecurePreciseInv inv = (SecurePreciseInv)(curInvFromLog.getTreeNode().getInv());

            if(!lastStartVV.includes(inv.getAcceptStamp()) && endVV.containsNodeId(nId)){
		if(SecurityFilter.dbg)System.out.println("Adding split point " + inv.getAcceptStamp());
              addSplitPoint(nId, ahsEntry.getEndTS());
            }
            
            // princem: I think the following check should be removed if we want a causal stream
            if( !lastStartVV.includes(inv.getAcceptStamp()) ){
              
              //code related to addition of splitStamps of AHS entries
              //first drop entries also present in acceptVV
              AcceptVV preciseInvDVV = lastAddedDVVTracker.get(nId);
              DependencyVV ahsDVV = dvvTracker.get(nId);
              VVIterator vvIterator = preciseInvDVV.getIterator();
              while(vvIterator.hasMoreElements()){
                ahsDVV.dropNode(vvIterator.getNext());
              }

	      if(SecurityFilter.dbg){
		  //	  System.out.println("Adding splitpoints corresponding to stamps " + ahsDVV);
		  assert securityFilter.splitManager.tryAndSplit(ahsDVV, securityFilter.core.getMyNodeId(), false);
	      }

              //add split stamps corresponding to all remaining entries
              Collection<AcceptStamp> allStamps = new AcceptVV(ahsDVV).getAllStamps();
              for(AcceptStamp s: allStamps){
                if(!lastStartVV.includes(s) && endVV.containsNodeId(s.getNodeId())){
		    //                  if(SecurityFilter.dbg)System.out.println("adding splitpoint : " + s);
                  splitStamps.add(new DummyPreciseInv(s));
                  addSplitPoint(s.getNodeId(), s.getLocalClock());
                }

              }

              dvvTracker.put(nId, new DependencyVV());
              lastAddedDVVTracker.put(nId, new AcceptVV(((SecurePreciseInv)inv).getDVV()));

              if(SecurityFilter.dbg){
                System.out.println("~~~ Adding inv :"  + (SecurePreciseInv)inv);
              }
              ret.addFirst((PreciseInv)(curInvFromLog.getTreeNode().getInv()));

              DependencyVV dvv = ((SecurePreciseInv)inv).getDVV();
              dvv.dropNode(nId);
              maxDVV = new AcceptVV(dvv);

              
	      if(SecurityFilter.dbg){
		  //  System.out.println("Adding splitpoints corresponding to stamps from precise " + maxDVV);
	      }
              allStamps = maxDVV.getAllStamps();
              for(AcceptStamp s: allStamps){
                if(!lastStartVV.includes(s) && endVV.containsNodeId(s.getNodeId())){
		    //                  if(SecurityFilter.dbg)System.out.println("adding splitpoint : " + s);
                  splitStamps.add(new DummyPreciseInv(s));
                  addSplitPoint(s.getNodeId(), s.getLocalClock());

                }

              }

            }else{

            }
          }
          else{
            // add splitpoints to ensure that this AHSEntry is not aggregated
            // if prevVV contains a component for nId then add prevVV[nId] and endVV[nId]
            // also reset prevVV[nId]

            prevVV.setStampByNodeId(nId, ahsEntry.getEndTS());
            dvvTracker.get(nId).extend(ahsEntry.getMaxDVVMap().getDVV());

          }

        }else{
          // neither filter nor liveness filter is interested in this inval
          if(prevVV.containsNodeId(nId)){
            AcceptStamp as = new AcceptStamp(curInvFromLog.getTreeNode().getEndTS(), nId);
            if(!lastStartVV.includes(as) && endVV.containsNodeId(nId)){
		//              if(SecurityFilter.dbg)System.out.println("adding splitpoint because transitioning from interested to not interested "  +as);
              splitStamps.add(new DummyPreciseInv(as));
              addSplitPoint(as.getNodeId(), as.getLocalClock());
            }
            prevVV.dropNode(nId);
          }
          dvvTracker.get(nId).extend(ahsEntry.getMaxDVVMap().getDVV());

        }

      }
      
      

      vvi = endVV.getIterator();
      NodeId nodeId;
      while(vvi.hasMoreElements()){
        nodeId = vvi.getNext();
        
        //adding the remaining splitStamps
        AcceptVV preciseInvDVV = lastAddedDVVTracker.get(nodeId);
        DependencyVV ahsDVV = dvvTracker.get(nodeId);
        VVIterator vvIterator = preciseInvDVV.getIterator();
        while(vvIterator.hasMoreElements()){
	    NodeId n = vvIterator.getNext();
	    assert (!ahsDVV.containsNodeId(n) || ahsDVV.getStampByServer(n) < preciseInvDVV.getStampByServer(n)): ahsDVV + "preciseInvDVV:"+ preciseInvDVV;
          ahsDVV.dropNode(n);
        }
	if(SecurityFilter.dbg){
	    //	    System.out.println("Adding splitpoints corresponding to stamps " + ahsDVV);
	}

        //add split stamps corresponding to all remaining entries
        Collection<AcceptStamp> allStamps = new AcceptVV(ahsDVV).getAllStamps();
        for(AcceptStamp s: allStamps){
          if(!lastStartVV.includes(s) && endVV.containsNodeId(s.getNodeId())){
	      //            if(SecurityFilter.dbg)System.out.println("adding splitpoint : " + s);
            splitStamps.add(new DummyPreciseInv(s));
            addSplitPoint(s.getNodeId(), s.getLocalClock());

          }

        }
        
        
      }


      //finally we have all the split points and precise invals in splitStamps and ret
      // now we have to merge them ensuring that no value included in exclusionVV gets included
      // only splitStamps can have values smaller than exclusionVV
      AcceptVV exclusionVV = AcceptVV.decrementAll(startVV);


      assert securityFilter.getAhsMap().isConsistent();
      return merge(ret, splitStamps, exclusionVV);
    }
    catch(Exception e){
      e.printStackTrace();
      assert false: e;
      return null;
    }
  }

}
