package code.security;


import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.lang.reflect.AccessibleObject;
import java.lang.reflect.Field;
import java.security.PublicKey;
import java.util.*;

import code.*;
import code.branchDetecting.BranchID;
import code.security.holesync.*;
import code.security.ahs.AHS;
import code.security.ahs.AHSEntry;
import code.security.ahs.DVVMap;
import code.security.ahs.DVVMapEntry;
import code.security.ahs.IH;
import code.security.ahs.NodeAHSTuple;
import code.security.ahs.DependencyVV;
import code.security.ahs.UnmatchingTreeNodeException;
/**
 * Represents a secure checkpoint: a collection of startVV, endVV and subscriptionSet
 * @author princem
 *
 */
public class SecureCheckpoint implements Externalizable, Cloneable, SecureInv{
  transient private AcceptVV startVV;
  transient private AcceptVV endVV;
  private SubscriptionSet ss;
  private HashMap<Hole, HoleFiller> holeFiller;
  transient private AcceptVV externalDVV = null;

  /**
   * send SS or not?
   */
  public final static boolean sendSS = false;

  /**
   * contains an ordered list of precise updates and AHS value which should be applied according to their order in the list
   */
  LinkedList orderedUpdateList;
  final NodeId creator;

  public SecureCheckpoint(){
    creator = null;
    startVV = null;
    endVV = null;
    ss = null;
    orderedUpdateList = null;
    holeFiller = null;
  }

  public SecureCheckpoint(VV startVV, VV endVV, SubscriptionSet ss, NodeId creator){
    this.startVV = new AcceptVV(new DependencyVV());
    this.endVV = new AcceptVV(new DependencyVV());

//  if(startVV instanceof AcceptVV){
//  this.startVV = (AcceptVV)startVV;
//  }else{
//  this.startVV = new AcceptVV(startVV);
//  }

//  if(endVV instanceof AcceptVV){
//  this.endVV = (AcceptVV)endVV;
//  }else{
//  endVV = new AcceptVV(endVV);
//  }

    this.ss = ss;
    orderedUpdateList = new LinkedList();
    this.creator = creator;
    holeFiller = new HashMap<Hole, HoleFiller>();
  }

  public void setHoleFiller(HashMap<Hole, HoleFiller> hf){
    this.holeFiller = hf;
  }

  public final HashMap<Hole, HoleFiller> getHoleFiller(){
    return this.holeFiller;
  }

  public void addUpdate(PreciseInv update){
    orderedUpdateList.addLast(update);
    startVV = startVV.cloneMinVV(update.getStartVV());
    endVV = endVV.cloneMaxVV(update.getEndVV());
  }



  public void addUpdate(NodeAHSTuple update){
    orderedUpdateList.addLast(update);
    //princem: hacked code, should be removed
    List<AHSEntry> ahs = update.getAHS().getSortedListOfEntries();
    try{
      if(!startVV.containsNodeId(update.getNodeId()) || startVV.getStampByServer(update.getNodeId()) > ahs.get(0).getStartTS()){
        DependencyVV dvv = new DependencyVV(startVV);
        dvv.put(update.getNodeId(), ahs.get(0).getStartTS());
        startVV = new AcceptVV(dvv);
      }
    }catch(NoSuchEntryException e){
      // TODO Auto-generated catch block
      assert false: e;
    }

    try{
      if(!endVV.containsNodeId(update.getNodeId()) || endVV.getStampByServer(update.getNodeId()) < ahs.get(ahs.size()-1).getEndTS()){
        DependencyVV dvv = new DependencyVV(endVV);
        dvv.put(update.getNodeId(), ahs.get(ahs.size()-1).getEndTS());
        endVV = new AcceptVV(dvv);
      }
    }catch(NoSuchEntryException e){
      // TODO Auto-generated catch block
      assert false: e;
    }
  }

  public void addHoleFiller(Hole h, HoleFiller hf){
    holeFiller.put(h, hf);
  }
  /**
   * initialize externalIDVV
   */
  public void initExternalDVV(){
    CounterVV _externalDVV = new CounterVV();
    CounterVV _minVV = new CounterVV();

    HashMap<NodeId, NodeId> nodeIdList = new HashMap<NodeId, NodeId>();
    for(Object o: orderedUpdateList){
      if(o instanceof SecurePreciseInv){
        // process secure precise invalidate

        SecurePreciseInv spi = (SecurePreciseInv)o;
        nodeIdList.put(spi.getCreator(), spi.getCreator());
        _externalDVV.addMaxVV(spi.getExternalDVV());
        if(spi.getExternalDVV().containsNodeId(spi.getCreator())){
          try {
            _minVV.addMinAS(new AcceptStamp(spi.getExternalDVV().getStampByServer(spi.getCreator()), spi.getCreator()));
          } catch (NoSuchEntryException e) {
            // TODO Auto-generated catch block
            assert false: e;
          }
        }else{
          _minVV.addMinAS(new AcceptStamp(AcceptStamp.BEFORE_TIME_BEGAN, spi.getCreator()));
        }
      }else{
        //process NodeAHSTuple

        //assert false;
        assert(o instanceof NodeAHSTuple);
        NodeAHSTuple nodeAHSTuple = (NodeAHSTuple)o;
        NodeId nodeId = nodeAHSTuple.getNodeId();
        nodeIdList.put(nodeAHSTuple.getNodeId(), nodeId);
        DVVMap _maxDVVMap = SecurityFilter.getMaxDVVMap(nodeAHSTuple.getAHS().getIHs(), nodeId);  
        DependencyVV _dvv = _maxDVVMap.getDVV();
        _externalDVV.addMaxVV(_dvv);

        if(_dvv.containsNodeId(nodeId)){
          try {
            _minVV.addMinAS(new AcceptStamp(_dvv.getStampByServer(nodeId), nodeId));
          } catch (NoSuchEntryException e1) {
            // TODO Auto-generated catch block
            assert false: e1;
          }
        }else{
          _minVV.addMinAS(new AcceptStamp(AcceptStamp.BEFORE_TIME_BEGAN, nodeId));
        }

      }
    }
    for(NodeId nodeId: nodeIdList.keySet()){
      try{
        _externalDVV.setStampByServer(nodeId, _minVV.getStampByServer(nodeId));
      }catch(NoSuchEntryException e){
        // TODO Auto-generated catch block
        assert false: e;
      }
    }
    this.externalDVV = _externalDVV.cloneAcceptVV();
  }

  public int size(CostPredicate cp){
    int size = 0;
    int numInvals = 0;
    int preCVVPreciseInvals = 0;
    int preCVVSummaryInvals = 0;
    int preCVVSummarySize = 0 ;
    int postCVVPreciseSize = 0;
    int postCVVSummarySize = 0;
    int logicalOnDiskSize = 0;
    int physicalOnDiskSize = 0; 
    int logicalPredicateCost = 0;
    int physicalPredicateCost = 0;
    int preSSCost = 0;
    int impreSSCost = 0;

    int bw = 0;
    int dvv = 0;
    //System.out.println("CVV:" + cvv);

    size += 8; // for creator nodeId
    size += 4; // for the size
    for(Object o: orderedUpdateList){
      if(o instanceof SecurePreciseInv){
        // process secure precise invalidate
        size++;
        numInvals++;
        bw +=2;
        SecurePreciseInv spi = (SecurePreciseInv)o;
        dvv += spi.getDVV().getSize();
        Hashtable<String, Integer> spiStatTable = spi.getSize();
        logicalOnDiskSize+=spiStatTable.get("LogicalOnDiskSize");
        physicalOnDiskSize += spiStatTable.get("PhysicalOnDiskSize");
	preSSCost += spiStatTable.get("SubscriptionSet");
        if(cp.includes(spi.getAcceptStamp())){
          preCVVPreciseInvals++;
          logicalPredicateCost+=spiStatTable.get("LogicalOnDiskSize");
          physicalPredicateCost += spiStatTable.get("PhysicalOnDiskSize");
        }else{
          postCVVPreciseSize++;
        }

      }else{
        //process NodeAHSTuple

        //assert false;
        assert(o instanceof NodeAHSTuple);
        NodeAHSTuple nodeAHSTuple = (NodeAHSTuple)o;
        for(AHSEntry ahsEntry: nodeAHSTuple.getAHS().getSortedListOfEntries()){
          numInvals+= Math.pow(2, ahsEntry.getLevel());
          size+= 1;
          DependencyVV dVV = ahsEntry.getMaxDVVMap().getDVV();
          bw+= dVV.getSize();
          dvv += dVV.getSize();
          Hashtable<String, Integer> ahsStatTable = ahsEntry.getSize();
          logicalOnDiskSize+=ahsStatTable.get("LogicalOnDiskSize");
          physicalOnDiskSize += ahsStatTable.get("PhysicalOnDiskSize");
	  impreSSCost += ahsStatTable.get("SubscriptionSet");
          if(cp.includes(new AcceptStamp(ahsEntry.getStartTS(), nodeAHSTuple.getNodeId()), 
              new AcceptStamp(ahsEntry.getEndTS(), nodeAHSTuple.getNodeId()))){
            preCVVSummaryInvals+=Math.pow(2, ahsEntry.getLevel());
            preCVVSummarySize+=1;
            logicalPredicateCost+=ahsStatTable.get("LogicalOnDiskSize");
            physicalPredicateCost += ahsStatTable.get("PhysicalOnDiskSize");
          }else{
            postCVVSummarySize++;
          }
        }
      }
    }
    
    size += 4; // for the size
    Set<NodeId> nodeIds = new TreeSet<NodeId>();
    for(Hole h: this.holeFiller.keySet()){
      HoleFiller hf = holeFiller.get(h);
      nodeIds.add(h.getNodeId());
      for(Object o: hf.getInvals()){
        if(o instanceof SecurePreciseInv){
          // process secure precise invalidate
          size++;
          numInvals++;
          bw +=2;
          SecurePreciseInv spi = (SecurePreciseInv)o;
          dvv += spi.getDVV().getSize();
          Hashtable<String, Integer> spiStatTable = spi.getSize();
          logicalOnDiskSize+=spiStatTable.get("LogicalOnDiskSize");
          physicalOnDiskSize += spiStatTable.get("PhysicalOnDiskSize");
          preSSCost += spiStatTable.get("SubscriptionSet");
          if(cp.includes(spi.getAcceptStamp())){
            preCVVPreciseInvals++;
            logicalPredicateCost+=spiStatTable.get("LogicalOnDiskSize");
            physicalPredicateCost += spiStatTable.get("PhysicalOnDiskSize");
          }else{
            postCVVPreciseSize++;
          }

        }else{
          //assert false;
          assert(o instanceof AHSEntry);
          AHSEntry ahsEntry = (AHSEntry)o;
          numInvals+= Math.pow(2, ahsEntry.getLevel());
            size+= 1;
            DependencyVV dVV = ahsEntry.getMaxDVVMap().getDVV();
            bw+= dVV.getSize();
            dvv += dVV.getSize();
            Hashtable<String, Integer> ahsStatTable = ahsEntry.getSize();
            logicalOnDiskSize+=ahsStatTable.get("LogicalOnDiskSize");
            physicalOnDiskSize += ahsStatTable.get("PhysicalOnDiskSize");
            impreSSCost += ahsStatTable.get("SubscriptionSet");
            if(cp.includes(new AcceptStamp(ahsEntry.getStartTS(), h.getNodeId()), 
                new AcceptStamp(ahsEntry.getEndTS(), h.getNodeId()))){
              preCVVSummaryInvals+=Math.pow(2, ahsEntry.getLevel());
              preCVVSummarySize+=1;
              logicalPredicateCost+=ahsStatTable.get("LogicalOnDiskSize");
              physicalPredicateCost += ahsStatTable.get("PhysicalOnDiskSize");
            }else{
              postCVVSummarySize++;
            }
          
        }
      }
    }
    
    size += nodeIds.size() * (8 + 4); // 8 for nodeIds and 4 for the size of vector
    
    if(SecurityFilter.measureTime)System.out.println("DVV " + dvv + " bw " +bw + " invals " + numInvals + " preCVVSummaryInvals " + preCVVSummaryInvals + " preCVVPreciseInvals " + preCVVPreciseInvals + " preCVVSummarySize " + preCVVSummarySize + 
        " preCVVPreciseSize " + preCVVPreciseInvals + " postCVVPreciseSize " + postCVVPreciseSize + " postCVVSummarySize " + postCVVSummarySize + 
        " logicalBW " + logicalOnDiskSize + " physicalBW " + physicalOnDiskSize + 
        " logicalPreCVV " + logicalPredicateCost + " physicalPreCost " + physicalPredicateCost + 
        " logicalNPreCVV " + (logicalOnDiskSize - logicalPredicateCost) + " physicalNPreCost " + (physicalOnDiskSize-physicalPredicateCost) + " CostPredicateCost " + cp.size() + " preSSCost " + preSSCost + " impreSSCost " + impreSSCost);
    return size;
  }

  public Hashtable<String, Integer> stats(){
    Hashtable<String, Integer> statTable = new Hashtable<String, Integer>();
    int bw = 0; 
    int dvv = 0; 
    int size = 0;
    int logicalOnDiskSize = 0;
    int physicalOnDiskSize = 0; 
    int subscriptionSet = 0;
    int timeStamps = 0;
    int nodeId = 0;
    int signature = 0;
    int other = 0;
    int serializationCost = 0;
    int summarySerCost = 0;
    int preciseSerCost = 0;
    int summaries = 0;
    int preciseInvs = 0;
    for(Object o: orderedUpdateList){
      if(o instanceof SecurePreciseInv){
        // process secure precise invalidate
        size++;
        SecurePreciseInv spi = ((SecurePreciseInv)o);
        Hashtable<String, Integer> spiStatTable = spi.getSize();
        bw+= spiStatTable.get("BW (total number of hashes)");
        dvv += spiStatTable.get("DVV (total number of DVV Entries)");
        logicalOnDiskSize+=spiStatTable.get("LogicalOnDiskSize");
        physicalOnDiskSize += spiStatTable.get("PhysicalOnDiskSize");
        subscriptionSet += spiStatTable.get("SubscriptionSet");
        timeStamps += spiStatTable.get("TimeStamps");
        nodeId += spiStatTable.get("NodeId");
        signature += spiStatTable.get("Signature");
        other += spiStatTable.get("Other");
        serializationCost += spiStatTable.get("SerializationCost");
        preciseSerCost += spiStatTable.get("SerializationCost");
        preciseInvs++;

      }else{
        //process NodeAHSTuple

        //assert false;
        assert(o instanceof NodeAHSTuple);
        NodeAHSTuple nodeAHSTuple = (NodeAHSTuple)o;
        for(AHSEntry ahsEntry: nodeAHSTuple.getAHS().getSortedListOfEntries()){
          Hashtable<String, Integer> ahsStatTable = ahsEntry.getSize();
          bw+= ahsStatTable.get("BW (total number of hashes)");
          dvv += ahsStatTable.get("DVV (total number of DVV Entries)");
          logicalOnDiskSize+=ahsStatTable.get("LogicalOnDiskSize");
          physicalOnDiskSize += ahsStatTable.get("PhysicalOnDiskSize");
          subscriptionSet += ahsStatTable.get("SubscriptionSet");
          timeStamps += ahsStatTable.get("TimeStamps");
          nodeId += ahsStatTable.get("NodeId");
          signature += ahsStatTable.get("Signature");
          other += ahsStatTable.get("Other");
          serializationCost += ahsStatTable.get("SerializationCost");
          summarySerCost += ahsStatTable.get("SerializationCost"); 
          summaries++;
        }
      }
    }
    Set<NodeId> nodeIds = new TreeSet<NodeId>();
    for(Hole h: this.holeFiller.keySet()){
      HoleFiller hf = holeFiller.get(h);
      nodeIds.add(h.getNodeId());
      for(Object o: hf.getInvals()){
        if(o instanceof SecurePreciseInv){
          // process secure precise invalidate
          size++;
          SecurePreciseInv spi = ((SecurePreciseInv)o);
          Hashtable<String, Integer> spiStatTable = spi.getSize();
          bw+= spiStatTable.get("BW (total number of hashes)");
          dvv += spiStatTable.get("DVV (total number of DVV Entries)");
          logicalOnDiskSize+=spiStatTable.get("LogicalOnDiskSize");
          physicalOnDiskSize += spiStatTable.get("PhysicalOnDiskSize");
          subscriptionSet += spiStatTable.get("SubscriptionSet");
          timeStamps += spiStatTable.get("TimeStamps");
          nodeId += spiStatTable.get("NodeId");
          signature += spiStatTable.get("Signature");
          other += spiStatTable.get("Other");
          serializationCost += spiStatTable.get("SerializationCost");
          preciseSerCost += spiStatTable.get("SerializationCost");
          preciseInvs++;

        }else{
          //process NodeAHSTuple

          //assert false;
          assert(o instanceof AHSEntry);
          AHSEntry ahsEntry = (AHSEntry)o;
          Hashtable<String, Integer> ahsStatTable = ahsEntry.getSize();
          bw+= ahsStatTable.get("BW (total number of hashes)");
          dvv += ahsStatTable.get("DVV (total number of DVV Entries)");
          logicalOnDiskSize+=ahsStatTable.get("LogicalOnDiskSize");
          physicalOnDiskSize += ahsStatTable.get("PhysicalOnDiskSize");
          subscriptionSet += ahsStatTable.get("SubscriptionSet");
          timeStamps += ahsStatTable.get("TimeStamps");
          nodeId += ahsStatTable.get("NodeId");
          signature += ahsStatTable.get("Signature");
          other += ahsStatTable.get("Other");
          serializationCost += ahsStatTable.get("SerializationCost");
          summarySerCost += ahsStatTable.get("SerializationCost"); 
          summaries++;

        }
      }
    }

    statTable.put("LogicalUpdates", size);
    statTable.put("DVV (total number of DVV Entries)", dvv);
    statTable.put("BW (total number of hashes)", bw);
    statTable.put("LogicalOnDiskSize", logicalOnDiskSize);
    statTable.put("PhysicalOnDiskSize", physicalOnDiskSize);
    statTable.put("SubscriptionSet", subscriptionSet);
    statTable.put("TimeStamps", timeStamps);
    statTable.put("NodeId", nodeId);
    statTable.put("Signature", signature);
    statTable.put("Other", other);
    statTable.put("SerializationCost", serializationCost);
    statTable.put("SummarySerializationCost", summarySerCost);
    statTable.put("Summaries", summaries);
    statTable.put("PreciseInv", preciseInvs);
    statTable.put("PreciseSerializationCost", preciseSerCost);

    return statTable;
  }

  /**
   * Get a maxDVVMap for each node whose updates are present in the checkpoint
   * @return
   */
  /*
  public HashMap<NodeId, DependencyVV> getMaxDVVMap(){
    HashMap<NodeId, DependencyVV> dvvMapMap = new HashMap<NodeId, DependencyVV>();

    CounterVV minVV = new CounterVV(); // for keeping track of the min time stamp 

    //indicates the VV of the last precise update received by the corresponding writer
    //  HashMap<NodeId, AcceptVV> lastPreciseVVMap = new HashMap<NodeId, AcceptVV>();

    // process all the updates in the checkpoint
    for(Object o: orderedUpdateList){
      if(o instanceof SecurePreciseInv){
        // process secure precise invalidate

        SecurePreciseInv spi = (SecurePreciseInv)o;

        // also check if the hashes for the imprecise hashes received since last precise invalidate match or not
        // basically excluding components for DVV[spi], check all other components for the sh and the corresponding value
        // the following checks only take place if some updates received since last precise inval
        if(dvvMapMap.containsKey(spi.getCreator())){
          dvvMapMap.get(spi.getCreator()).takeMax(spi.getDVV());
        }else{
          dvvMapMap.put(spi.getCreator(), spi.getDVV().clone());
        }
        if(spi.getDVV().containsNodeId(spi.getCreator())){
          try{
            minVV.addMinAS(new AcceptStamp(spi.getDVV().getStampByServer(spi.getCreator()), spi.getCreator()));
          }catch(NoSuchEntryException e){
            // TODO Auto-generated catch block
            e.printStackTrace();
            assert false;
          }
        }else{
          minVV.addMinAS(new AcceptStamp(AcceptStamp.BEFORE_TIME_BEGAN, spi.getCreator()));
        }

      }else{
        //process NodeAHSTuple

        assert(o instanceof NodeAHSTuple);
        //simply apply the hashMap and return false if it fails

        NodeAHSTuple ihMap = (NodeAHSTuple)o;
        NodeId nodeId = ihMap.getNodeId();

        // put the ih tuples in the allIHs hashMap for creating imprecise invalidate
        Vector<IH> ihVector = ihMap.getAHS().getIHs();
        DependencyVV dvv = new DependencyVV();
        for(IH ih: ihVector){
          dvv.takeMax(ih.getAHSEntry().getMaxDVVMap().getDVV());
        }
        // also put the tuples in each field in newIHMap. newIHMap is used for validating ih tuples on receiving precise invalidates
        if(dvvMapMap.containsKey(nodeId)){

          dvvMapMap.get(nodeId).takeMax(dvv);
        }else{
          dvvMapMap.put(nodeId, dvv);
        }

        DependencyVV firstDVV = ihVector.get(0).getAHSEntry().getMaxDVVMap().getDVV();
        if(firstDVV.containsNodeId(nodeId)){
          try{
            minVV.addMinAS(new AcceptStamp(firstDVV.getStampByServer(nodeId), nodeId));
          }catch(NoSuchEntryException e){
            // TODO Auto-generated catch block
            e.printStackTrace();
            assert false;
          }
        }else{
          minVV.addMinAS(new AcceptStamp(AcceptStamp.BEFORE_TIME_BEGAN, nodeId));
        }

      }
    }

    // reset the entries to min value for nodes whose updates are present in the checkpoint
    for(NodeId nid: dvvMapMap.keySet()){
      if(minVV.containsNodeId(nid)){
        try{
          if(minVV.getStampByServer(nid) != AcceptStamp.BEFORE_TIME_BEGAN){
            dvvMapMap.get(nid).put(nid, minVV.getStampByServer(nid));
          }else{
            dvvMapMap.get(nid).dropNode(nid);    
          }
        }catch(NoSuchEntryException e){
          // TODO Auto-generated catch block
          e.printStackTrace();
          assert false;
        }
      }else{
        assert false;
      }
    }

    assert dvvMapMap.size() == endVV.size():dvvMapMap + " " + this;

    return dvvMapMap;
  }

   */ 

  public AcceptVV getStartVV(){
    return startVV;
  }

//public void setStartVV(AcceptVV startVV){
//this.startVV = startVV;
//}

  public AcceptVV getEndVV(){
    return endVV;
  }

//public void setEndVV(VV endVV){
//if(endVV instanceof AcceptVV){
//this.endVV = (AcceptVV)endVV;
//}else{
//this.endVV = new AcceptVV(endVV);
//}
//}

  public SubscriptionSet getSs(){
    return ss;
  }

  public void setSs(SubscriptionSet ss){
    this.ss = ss;
  }

  public final NodeId getCreator(){
    return creator;
  }

  public boolean verify(PublicKey pubkey){
    if(SangminConfig.disableSignatureVerfication){
      return true; 
    }
    return true;
  }

  public byte[] getSignature(){
    return new byte[]{};
  }

  public AcceptVV getExternalDVV(){
    assert(externalDVV != null);
    return externalDVV;
  }

  public String toString(){
    String str = "Checkpoint: startVV:" + startVV + " endVV:" + endVV + " creator:" + creator + " updateList: ";
    for(Object o : this.orderedUpdateList){
      str += "\n"+o;
    }

    str += " holes";
    for(Object o : holeFiller.keySet()){
      str += "\n"+o + " -> " + holeFiller.get(o);
    }
    return str;
  }

  public LinkedList getOrderedUpdateList(){
    return orderedUpdateList;
  }

  public void setOrderedUpdateList(LinkedList orderedUpdateList){
    this.orderedUpdateList = orderedUpdateList;
  }

  public void setEndVV(AcceptVV endVV){
    this.endVV = endVV;
  }

//public void setExternalDVV(AcceptVV externalDVV){
//this.externalDVV = externalDVV;
//}

  public boolean isEmpty(){
    return orderedUpdateList.isEmpty() && holeFiller.isEmpty();
  }

  public void writeExternal(ObjectOutput out) throws IOException{
    long start, end;

    start = System.currentTimeMillis();

    if(SangminConfig.forkjoin){
      out.writeObject(creator);
    } else {
      if(SangminConfig.compressNodeId){
        out.writeByte((int)creator.getIDint());
      }else{
        out.writeLong(creator.getIDint());
      }
    }
    //     System.out.println("WRITING CHECKPOINT");
    //    System.out.println("WRITING CHECKPOINT" + creator);
    boolean isEmp = isEmpty();
    //    System.out.println("WRITING CHECKPOINT" + isEmp);
    out.writeBoolean(isEmp);
    if(!isEmp){
      //startVV.writeExternal(out);
      //endVV.writeExternal(out);
//    out.writeObject(startVV);
//    out.writeObject(endVV);


      if(sendSS)
      {
        out.writeObject(ss);
      }


      int size = this.orderedUpdateList.size();
      out.writeInt(size);
      //      System.out.println("WRITING CHECKPOINT" + size);

      for(Object o : this.orderedUpdateList){
        if( o instanceof SecurePreciseInv ){
          SecurePreciseInv spi = (SecurePreciseInv)o;

	  	  //System.out.println("WRITING SPI");
          out.writeBoolean(true); // true -> SecurePreciseInv
          spi.writeExternal(out);
          //out.writeObject(spi);
        }else{
          assert(o instanceof NodeAHSTuple);
          //simply apply the hashMap and return false if it fails

	  	  //System.out.println("WRITING AHS");
          NodeAHSTuple ihMap = (NodeAHSTuple)o;

          out.writeBoolean(false);

          NodeId nodeId = ihMap.getNodeId();
          List<AHSEntry> l = ihMap.getAHS().getSortedListOfEntries();

          if(SangminConfig.forkjoin){
            assert nodeId instanceof BranchID;
            out.writeObject(nodeId);
            //System.out.println("Secure Checkpoint BranchID SENT : " +nodeId);
            out.writeInt(l.size());
          } else {
          
            if(SangminConfig.compressNodeId){
              out.writeByte((int)nodeId.getIDint());  
              out.writeByte(l.size());
            }else{
              out.writeLong(nodeId.getIDint());
              out.writeInt(l.size());
            }
          }
	  //	    System.out.println("WRITING CHECKPOINT" + l.size());
          //assert l.size() == 1;

          for(AHSEntry ahsEntry : l){
            ahsEntry.writeExternal(out);
          }


        } // end of else (end of sending NodeAHSTuple)

      } // end of orderedUpdateList

      writeHoleFiller(out);
    }
    end = System.currentTimeMillis();
    LatencyWatcher.put("SecureCheckpoint.writeExternal", 
        (end - start));
  }

  public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException{
    try{
      long start,end;
      start = System.currentTimeMillis();
      /*
      long creatorIdLong;
      if(SangminConfig.compressNodeId){
        creatorIdLong = in.readByte();
      }else{
        creatorIdLong = in.readLong();
      }*/
      
      NodeId creatorId;
      if(SangminConfig.forkjoin){
        creatorId = (BranchID)in.readObject();
        //System.out.println("SecureCheckpoint Creator received :" + creatorId);
      } else {
        if(SangminConfig.compressNodeId){
          creatorId = new NodeId(in.readByte());
        } else {
          creatorId = new NodeId(in.readLong());
        }
      }
      
      //      System.out.println("READING CHECKPOINT" + creatorIdLong);
      boolean isEmpty = in.readBoolean();
      //         System.out.println("READING CHECKPOINT" + isEmpty);
      CounterVV _startVV = new CounterVV();
      //_startVV.readExternal(in);
//    AcceptVV _startVV = (AcceptVV)in.readObject();
      CounterVV _endVV = new CounterVV();
      //_endVV.readExternal(in);
      // (AcceptVV)in.readObject();
      SubscriptionSet tmpss = SubscriptionSet.makeSubscriptionSet("/*");
      LinkedList list = new LinkedList();
      SubscriptionSet _ss;
      HashMap<Hole, HoleFiller> _hf;

      if(!isEmpty){
        if(sendSS)
        {
          tmpss = (SubscriptionSet)in.readObject();
        }


        InvalTarget hit = new HierInvalTarget();

        int orderedListSize = in.readInt();
	//	    System.out.println("READING CHECKPOINT" + orderedListSize);
        for(int i=0; i<orderedListSize; i++){
          boolean isPreciseInv = in.readBoolean();
	  //	      System.out.println("READING CHECKPOINT" + isPreciseInv);
          if(isPreciseInv){
            //System.out.println("READING SPI");
            SecurePreciseInv spi = null;
            try{
              spi = new SecurePreciseInv();
              spi.readExternal(in);
//            spi = (SecurePreciseInv)in.readObject();
              _startVV.addMinAS(spi.getAcceptStamp());
              _endVV.addMaxAS(spi.getAcceptStamp());
            }catch(IOException e){
              System.out.println(e.toString());

              System.out.println("creator" + creatorId + " List" + list + " orderedListSize " + orderedListSize + " isPreciseInv " + isPreciseInv + "current orderedlist " + i );
              throw e;
            }

            if(!sendSS)
            {
              hit = hit.getUnion(spi.getInvalTarget(), tmpss);
            }
            list.addLast(spi);
          }else{
            //System.out.println("READING AHS");
            NodeId nodeId;
            int ahssize;
            
            if(SangminConfig.forkjoin){
              nodeId = (BranchID)in.readObject();
              //System.out.println("SecureCheckpoint BranchID received :" + nodeId);
              ahssize = in.readInt();
            } else {

              if(SangminConfig.compressNodeId){
                nodeId = new NodeId(in.readByte());
                ahssize = in.readByte();
              }else{
                nodeId = new NodeId(in.readLong());
                ahssize = in.readInt();
              }
            }

            AHS ahs = new AHS();
            //int ahssize = 1;
	    //	        System.out.println("READING CHECKPOINT" + ahssize);
            for(int j=0; j < ahssize; j++){

              AHSEntry _ahsEntry = new AHSEntry();
              _ahsEntry.readExternal(in);
              ahs.add(_ahsEntry);
              _startVV.addMinAS(new AcceptStamp(_ahsEntry.getStartTS(), nodeId));
              _endVV.addMaxAS(new AcceptStamp(_ahsEntry.getEndTS(), nodeId));
            }

            NodeAHSTuple tuple = new NodeAHSTuple(nodeId, ahs);

            list.addLast(tuple);               

          }            

        }

        _hf = readHoleFiller(in);




        if(sendSS)
        {
          _ss = tmpss;
        }
        else
        {
          _ss = new SubscriptionSet((HierInvalTarget)hit);
        }

      }else{
        _hf = new HashMap<Hole, HoleFiller>();
        _ss = SubscriptionSet.makeSubscriptionSet("/*");
      }
      Field[] f = new Field[6];

      assert(in.available() == 0);

      try{

        f[0] = SecureCheckpoint.class.getDeclaredField("creator");
        f[1] = SecureCheckpoint.class.getDeclaredField("startVV");
        f[2] = SecureCheckpoint.class.getDeclaredField("endVV");
        f[3] = SecureCheckpoint.class.getDeclaredField("ss");
        f[4] = SecureCheckpoint.class.getDeclaredField("orderedUpdateList");
        f[5] = SecureCheckpoint.class.getDeclaredField("holeFiller");


      }catch(NoSuchFieldException ne){
        System.err.println(ne.toString());
        ne.printStackTrace();
        System.exit(-1);

      }

      try{
        AccessibleObject.setAccessible(f, true);
      } catch (SecurityException se){
        System.err.println(se.toString());
        se.printStackTrace();
        System.exit(-1);
      }
      try{
        f[0].set(this, creatorId);
        f[1].set(this, new AcceptVV(_startVV));
        f[2].set(this, new AcceptVV(_endVV));
        f[3].set(this, _ss);
        f[4].set(this, list);
        f[5].set(this, _hf);

      }catch(IllegalArgumentException ie){
        System.err.println(ie.toString());
        ie.printStackTrace();
        System.exit(-1);
      }catch(IllegalAccessException iae){
        System.err.println(iae.toString());
        iae.printStackTrace();
        System.exit(-1);
      }

      try{
        AccessibleObject.setAccessible(f, false);
      } catch (SecurityException se){
        System.err.println(se.toString());
        se.printStackTrace();
        System.exit(-1);
      }

      initExternalDVV();

      end = System.currentTimeMillis();
      LatencyWatcher.put("SecureCheckpoint.readExternal", 
          (end - start));

    }catch(IOException e){
      System.out.println(e);
      e.printStackTrace();
      throw e;
    }

  }

  private HashMap<Hole, HoleFiller> readHoleFiller(ObjectInput in) throws IOException, ClassNotFoundException{
    HashMap<Hole, HoleFiller> _hf = new HashMap<Hole, HoleFiller>();
    int size;
    if(SangminConfig.compressNodeId){
      size = in.readByte();
    }else{
      size = in.readInt();
    }
    for(int i = 0; i < size; i++){
      NodeId n;
      int othersize;
      if(SangminConfig.forkjoin){
        n = (BranchID)in.readObject();
        othersize = in.readInt();
      } else {
        if(SangminConfig.compressNodeId){
          n = new NodeId(in.readByte());
          othersize = in.readByte();
        }else{
          n = new NodeId(in.readLong());
          othersize = in.readInt();
        }
      }
      for(int j = 0; j < othersize; j++){
        HoleFiller hf = new HoleFiller();
        Range r = hf.specialReadExternal(in);
        _hf.put(new Hole(n, r), hf);
      }

    }
    return _hf;
  }

  private void writeHoleFiller(ObjectOutput out) throws IOException{
    // no need to send the actual hole; just send the nodeId and associated list of holeFillers. Holes can be 
    // inferred from that 
    HashMap<NodeId, Vector<HoleFiller>> rearrangedHoleFillers = new HashMap<NodeId, Vector<HoleFiller>>();
    for(Hole h: holeFiller.keySet()){
      if(!rearrangedHoleFillers.containsKey(h.getNodeId())){
        rearrangedHoleFillers.put(h.getNodeId(), new Vector<HoleFiller>());
      }
      rearrangedHoleFillers.get(h.getNodeId()).add(holeFiller.get(h));
    }
    if(SangminConfig.compressNodeId){
      out.writeByte(rearrangedHoleFillers.size());
    }else{
      out.writeInt(rearrangedHoleFillers.size());
    }
    for(NodeId n:rearrangedHoleFillers.keySet()){
      if(SangminConfig.forkjoin){
       out.writeObject(n);
       out.writeInt(rearrangedHoleFillers.get(n).size());
      } else {
        if(SangminConfig.compressNodeId){
          out.writeByte((int)n.getIDint());
          out.writeByte(rearrangedHoleFillers.get(n).size());
        }else{
          out.writeLong(n.getIDint());
          out.writeInt(rearrangedHoleFillers.get(n).size());
        }
      }
      for(HoleFiller hf: rearrangedHoleFillers.get(n)){
        hf.writeExternal(out);
      }
    }
  }

  private HashMap<Hole, HoleFiller> readHoleFiller1(ObjectInput in) throws IOException, ClassNotFoundException{
    HashMap<Hole, HoleFiller> _hf = new HashMap<Hole, HoleFiller>();
    int size = in.readInt();
    for(int i = 0; i < size; i++){
      Hole h = new Hole();
      h.readExternal(in);
      HoleFiller hf = new HoleFiller();
      hf.readExternal(in);
      _hf.put(h, hf);
    }
    return _hf;
  }

  private void writeHoleFiller1(ObjectOutput out) throws IOException{
    out.writeInt(holeFiller.size());
    for(Hole h: holeFiller.keySet()){
      h.writeExternal(out);
      holeFiller.get(h).writeExternal(out);
    }
  }

}
