package code.security;
import java.io.EOFException;
import java.io.IOException;
import java.io.InvalidClassException;
import java.net.Socket;

import code.AcceptVV;
import code.CPStartMsg;
import code.CatchupStreamEndMsg;
import code.CatchupStreamStartMsg;
import code.DebargoMsg;
import code.Env;
import code.GeneralInv;
import code.NodeId;
import code.VVIterator;
import code.security.SecureIncomingInvalConnection;
import code.IncommingInvalConnectionWorker;
import code.LatencyWatcher;
import code.PreciseInv;
import code.TaggedInputStream;
import code.TerminateMsg;
import code.UnbindMsg;

public class SecureIncomingInvalConnectionWorker extends
IncommingInvalConnectionWorker{

 /** 
 * Constructor for worker thread 
 **/ 
  public SecureIncomingInvalConnectionWorker(TaggedInputStream tis,
      Socket underlyingSocket,
      SecureIncomingInvalConnection ic){
    super(tis, underlyingSocket, ic);
  }

 /** 
 *  Process an incoming stream consisting of 
 *   
 *  [INVAL_SOCKET_MAGIC, senderId, intial prevVV,  
 *   {(CatchupStreamStartMsg - (GerneralInv)* - CatchupStreamEndMsg)| 
 *   (GeneralInv|UnbindMsg|DebargoMsg)*| 
 *   (CPStartMsg-cvv-LPVVRecord*- (PerObjStateForSend|PerRangeStateForSend|BodyMsg)*-RAS_SHIP_DONE)}* 
 *   TerminateMsg] 
 **/ 
  public void
  run(){

    if(SecureIncomingInvalConnection.dbgProgress){
      System.out.println("IncommingInvalConnectionWorker starts");
    }

    //
    // handshake
    //
    if(!handshake()){
      Env.remoteAssert(false);
      this.closeStreamAndSocket("SecureIncomingInvalConnection cannot handshake with sender ");
      closeFile();
      return;
    }

    //tbd from InvalRecvWorker.java
    if(!tbdPrinted){
      Env.tbd("TBD: subclass Socket to be a heartbeat Socket \n"
          + "sender sends a msg every 5 seconds \n "
          + "and receiver expects to recv message every 20 seconds or\n"
          + "throws IO Exception OR USE KEEPALIVE OPTION?");
      tbdPrinted = true;
    }





    long handleInvalStart=0;
    long handleInvalEnd=0;

    //
    // Keep catchupstreamstart so that when a catchupstream
    // finishes, we know what we were working on...
    //
    CatchupStreamStartMsg lastCatchupStreamStart = null;

    //
    // Read the sequence of invalidations/CP from stream
    //
    try{

      Object next = s.readTaggedObject();
      while(! (next instanceof TerminateMsg)){

        if(measureTime){
          handleInvalStart = System.currentTimeMillis();
        }
        if(SecureIncomingInvalConnection.dbg){
          Env.dprintln(SecureIncomingInvalConnection.dbg, "Read next : " + next);
        }


        if(next instanceof UnbindMsg){
          ic.applyUnbind((UnbindMsg)next);
        }else if(next instanceof DebargoMsg){
          ic.applyDebargo((DebargoMsg)next);



        }else if(next instanceof CatchupStreamStartMsg){
          // 
          // Right now we do not allow nested catchupstreams.
          // If we want to do that. Receiver needs to keep
          // a stack of start messages...
          //
          Env.remoteAssert(lastCatchupStreamStart == null);
          if(lastCatchupStreamStart != null){
            throw new IOException("Nested catchup stream");
          }
          if(SecureIncomingInvalConnection.dbgPerformance){
            long start = System.currentTimeMillis();
            Env.dprintln(SecureIncomingInvalConnection.dbgPerformance, "@ "+ start 
                + " SecureIncomingInvalConnection -- catchupStreamStartMsg arrives");
          }
          lastCatchupStreamStart = (CatchupStreamStartMsg)next;
          ic.applyCatchupStreamStartMsg(lastCatchupStreamStart);

          if(measureTime){
            handleInvalEnd = System.currentTimeMillis();

            LatencyWatcher.put("SecureIncomingInvalConnection.ApplyCatchupStreamStartMsg", 
                (handleInvalEnd - handleInvalStart));

          }
        }else if(next instanceof CatchupStreamEndMsg){

          Env.remoteAssert(lastCatchupStreamStart != null);
          if(lastCatchupStreamStart == null){
            throw new IOException("Mismatched catchup stream");
          }
          if(SecureIncomingInvalConnection.dbgPerformance){
            long start = System.currentTimeMillis();
            Env.dprintln(SecureIncomingInvalConnection.dbgPerformance, "@ "+ start 
                + " SecureIncomingInvalConnection -- catchupStreamEndMsg arrives");
          }
          ic.applyCatchupStreamEndMsg( 
              lastCatchupStreamStart);
          lastCatchupStreamStart = null;

          if(measureTime){
            handleInvalEnd = System.currentTimeMillis();

            LatencyWatcher.put("SecureIncomingInvalConnection.ApplyCatchupStreamEndMsg", 
                (handleInvalEnd - handleInvalStart));
          }


        }else if(next instanceof CPStartMsg){
          ic.applyCP(this.s, (CPStartMsg)next);
          if(measureTime){
            handleInvalEnd = System.currentTimeMillis();
            writeToFile((handleInvalEnd - handleInvalStart));
            LatencyWatcher.put("SecureIncomingInvalConnection.ApplyCPStartMsg", (handleInvalEnd - handleInvalStart));
          }

        }else if(next instanceof SecureCheckpoint){
          if(SecureIncomingInvalConnection.dbgProgress){
            System.out.println("Received checkpoint" + ((SecureCheckpoint)next));
          }
          ((SecureIncomingInvalConnection)ic).applySecureCheckpoint((SecureCheckpoint)next);
          if(SecureIncomingInvalConnection.dbgProgress){
            System.out.println("Finished processing checkpoint");
          }
          if(measureTime){
            handleInvalEnd = System.currentTimeMillis();
            writeToFile((handleInvalEnd - handleInvalStart));
            LatencyWatcher.put("SecureIncomingInvalConnection.applySecureCheckpoint", (handleInvalEnd - handleInvalStart));
            ((SecureIncomingInvalConnection)ic).getController().informCheckpointApplied(ic.getSenderId(), true);
          }
        
        }else {
          if(!(next instanceof GeneralInv)){
            throw(new InvalidClassException("" + next.getClass()));
          }

          if(SecureIncomingInvalConnection.dbgPerformance && (next instanceof PreciseInv)){
            long start = System.currentTimeMillis();
            Env.dprintln(SecureIncomingInvalConnection.dbgPerformance, "IncommingReceive @ "+ start 
                + " " + ((PreciseInv)next).getAcceptStamp().toString());
          }
          ic.applyGI((GeneralInv)next);
          if(measureTime){
            handleInvalEnd = System.currentTimeMillis();
            writeToFile((handleInvalEnd - handleInvalStart));
            LatencyWatcher.put("SecureIncomingInvalConnection.ApplyInval", (handleInvalEnd - handleInvalStart));
            AcceptVV startVV = ((GeneralInv)next).getStartVV();
            AcceptVV endVV = ((GeneralInv)next).getEndVV();
            VVIterator vvi = startVV.getIterator();
            int n = 0;
            while(vvi.hasMoreElements()){
              NodeId nodeId = vvi.getNext();
              long start = startVV.getStampByServer(nodeId);
              long end = endVV.getStampByServer(nodeId);
              n += end-start+1;
            }            
            LatencyWatcher.put("SecureIncomingInvalConnection.AveragedApplyInval",
                (handleInvalEnd-handleInvalStart),n);
          }
        }



        if(SecureIncomingInvalConnection.dbgPerformance){
          long start = System.currentTimeMillis();
          Env.dprintln(SecureIncomingInvalConnection.dbgPerformance, "@ "+ start 
              + " SecureIncomingInvalConnection -- about to read next message");
        }


        next = s.readTaggedObject();


        if(SecureIncomingInvalConnection.dbgPerformance){
          long start = System.currentTimeMillis();
          Env.dprintln(SecureIncomingInvalConnection.dbgPerformance, "@ "+ start 
              + " SecureIncomingInvalConnection -- done reading next message");
        }
      }// while loop for next object

    }catch(EOFException i){

      this.closeStreamAndSocket("terminated because of EOFException " + i.toString());
      closeFile();

      return; // expected case thread exit
    }catch(IOException io){
      if(SecureIncomingInvalConnection.dbg){
        io.printStackTrace();
      }
      this.closeStreamAndSocket("terminated because of IOException " + io.toString());
      closeFile();
      return; // expected case thread exit
    }catch(Exception i3){
      i3.printStackTrace();
      assert false; // Unexpected cases
      this.closeStreamAndSocket("terminated unexcpectedly because of " + i3.toString());
      closeFile();
      return; // thread exit
    }

    this.closeStreamAndSocket("terminated normally");

  }
}
