package code.security;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.io.ObjectOutputStream;
import java.lang.reflect.AccessibleObject;
import java.lang.reflect.Field;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Signature;
import java.util.Collections;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.List;
import java.util.ListIterator;
import java.util.Random;

import code.AcceptStamp;
import code.AcceptVV;
import code.Config;
import code.NodeId;
import code.ObjId;
import code.ObjInvalTarget;
import code.PreciseInv;
import code.ResultStats;
import code.SummaryHash;
import code.branchDetecting.BranchID;
import code.security.ahs.AHSEntry;
import code.security.ahs.DependencyVV;
import code.serialization.SerializationHelper;
import code.simulator.Hash;



public class SecurePreciseInv extends PreciseInv implements SecureInv{

  protected final DependencyVV  dvv;
  protected final SummaryHash   sh;
  protected final DataHash	dh;
  protected byte[]	signature;
  protected final boolean hasHash;

  final static int classCode = 0x5b;


  public static long encryptionTime = 0;
  int hops;

  // For unit test
  public SecurePreciseInv(){
    dvv = null;
    sh = null;
    dh = null;
    signature = null;
    hasHash = false;
  }
  public SecurePreciseInv(final ObjInvalTarget obj_,
      final AcceptStamp acceptStamp_,
      final AcceptStamp rtAcceptStamp_,
      boolean embargoed_,
      final DependencyVV dvv_,
      final SummaryHash sh_,
      final DataHash dh_,
      byte[] sig){

    super(obj_, acceptStamp_, rtAcceptStamp_, embargoed_);
    dvv = dvv_.clone();
    sh = sh_;
    dh = dh_;
    signature = sig;
    hasHash = true;
    hops = 0;

  }

  public SecurePreciseInv(final ObjInvalTarget obj_,
      final AcceptStamp acceptStamp_,
      final AcceptStamp rtAcceptStamp_,
      boolean embargoed_,
      final DependencyVV dvv_,
      final DataHash dh_){

    super(obj_, acceptStamp_, rtAcceptStamp_, embargoed_);
    dvv = dvv_.clone();
    sh = null;
    dh = dh_;
    signature = null;
    hasHash = false;
    hops = 0;

  }

  public SecurePreciseInv(final ObjInvalTarget obj_,
      final AcceptStamp acceptStamp_,
      final AcceptStamp rtAcceptStamp_,
      boolean embargoed_,
      final DependencyVV dvv_,
      final SummaryHash sh_,
      final DataHash dh_,
      PrivateKey privKey){

    super(obj_, acceptStamp_, rtAcceptStamp_, embargoed_);
    dvv = dvv_.clone();
    sh = sh_;
    dh = dh_;
    hasHash = true;
    signature = sign(privKey);
    hops = 0;

  }

  protected SecurePreciseInv(final ObjInvalTarget obj_,
      final AcceptStamp acceptStamp_,
      final AcceptStamp rtAcceptStamp_,
      final DependencyVV dvv_,
      final SummaryHash sh_,
      PrivateKey privKey){

    super(obj_, acceptStamp_, rtAcceptStamp_, false);
    dvv = dvv_.clone();
    sh = sh_;
    dh = new DataHash(new byte[]{0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19}, false);
    signature = null;
    hasHash = true;
    hops = 0;

  }

  public SecurePreciseInv(PreciseInv inv,
      final DependencyVV dvv_,
      final SummaryHash sh_,
      final DataHash dh_,
      PrivateKey privKey){

    super(inv.getObjInvalTarget(),
        inv.getAcceptStamp(),
        inv.getRTAcceptStamp(),
        inv.isEmbargoed());

    if(SangminConfig.securityLevel == SangminConfig.COMPLETE){
      dvv = dvv_.clone();
      sh = sh_;    
    }else{
      dvv = null;
      sh = null;
    }
    hasHash = true;

    dh = dh_;
    signature = sign(privKey);
    hops = 0;

  }

  public SecurePreciseInv setHasHash(SummaryHash _sh){
    return new SecurePreciseInv(this.getObjInvalTarget(), this.getAcceptStamp(), 
        this.getRTAcceptStamp(), this.embargoed, this.getDVV(), _sh, this.getDH(), new byte[]{0});
  }

  public SecurePreciseInv resetHasHashAndSignature(){
    return new SecurePreciseInv(this.getObjInvalTarget(), this.getAcceptStamp(), 
        this.getRTAcceptStamp(), this.embargoed, this.getDVV(), this.getDH());
  }

  public SecurePreciseInv cloneWithNewNodeId(NodeId newId){
    return new SecurePreciseInv(this.obj,new AcceptStamp(super.acceptStamp.getLocalClock(),newId),
        new AcceptStamp(super.rtAcceptStamp.getLocalClock(),newId), this.embargoed, this.dvv,
        this.sh, this.dh, this.signature);    
  }

  public DataHash getDH(){
    return this.dh;
  }

  public SummaryHash getSH(){
    return this.sh;
  }

  public DependencyVV getDVV(){
    return this.dvv.clone();
  }

  /**
   * Update oldId of dvv to newId if the oldId's time stamp is greater 
   * than or equal to timestamp 
   * @param oldId NodeId to be updated
   * @param timestmp
   * @param newId new NodeId
   * @return true if this dvvMap has oldId component to be updated,
   *  false otherwise
   */
  public boolean updateDVV(NodeId oldId, long ts, NodeId newId){
    return dvv.updateNodeId(oldId, ts, newId);
  }

  public AcceptVV getExternalDVV(){
    return new AcceptVV(dvv);
    //return this.dvv.makeVVexcludeNode(super.getNodeId());
  }

  public NodeId getCreator(){
    return super.getNodeId();
  }
  public boolean verify(PublicKey pubKey){
    if(SangminConfig.disableSignatureVerfication || !hasHash){
      return true; 
    }
    long start = 0;
    if(SecurityFilter.measureTime){
      start = System.currentTimeMillis();
    }
    try{
      Signature sig = Crypto.getSignature();
      //Signature sig = Signature.getInstance(pubKey.getAlgorithm());
      sig.initVerify(pubKey);

      SerializationHelper sh = this.obj2BytesExcludingSignature();
      sh.close();
      sig.update(sh.toBytes());
      boolean ret =sig.verify(signature);
      Crypto.returnSignature(sig);
      return ret;
    }catch(Exception e){
      System.out.println(e.toString());
      e.printStackTrace();
      e.printStackTrace(System.out);
      System.exit(-1);
    }
    finally{
      if(SecurityFilter.measureTime){
        this.encryptionTime += System.currentTimeMillis() - start;
      }
    }
    return false;
  }

  protected byte[] sign(PrivateKey privKey){
    assert (privKey != null);

    if(SangminConfig.disableSignatureVerfication){
      byte[] b =new byte[1];
      b[0] = (byte)0;
      return b;
    }else{
      try{
        Signature sig = Crypto.getSignature();
        //      if(SangminConfig.useRSA){
        //    	  sig = Signature.getInstance(SangminConfig.RSAscheme,SangminConfig.RSAProvider);
        //      } else {
        //    	  sig = Signature.getInstance(privKey.getAlgorithm());
        //      }
        sig.initSign(privKey);

        SerializationHelper sh = this.obj2BytesExcludingSignature();
        sh.close();
        sig.update(sh.toBytes());
        byte[] ret = sig.sign();
        Crypto.returnSignature(sig);
        return ret;
      }catch(Exception e){
        System.out.println(e.toString());
        e.printStackTrace();
        e.printStackTrace(System.out);
        System.exit(-1);
      }
    }

    return null;
  }

  public byte [] obj2Bytes(){
    SerializationHelper sh = obj2BytesInternal();
    sh.close();
    return sh.toBytes();
  }

  public Hashtable<String, Integer> getSize(){
    Hashtable<String, Integer> statTable = new Hashtable<String, Integer>();
    int size = obj2Bytes().length+4; 
    try{
      SerializationHelper sh = new SerializationHelper();
      this.writeExternal(sh.getObjectOutputStream());
      if(SecurityFilter.dbg)System.out.println("serializaed object size: " + sh.toBytes().length + " obj2Bytes "+ size + " serialization cost " +  statTable.get("SerializationCost"));
      size = sh.toBytes().length;
    }catch(IOException e){
      // TODO Auto-generated catch block
      e.printStackTrace();
    }catch(Exception e1){
      e1.printStackTrace();
    }
    int ss = this.obj.onDiskSize();
    int sigSize = (hasHash?this.signature.length+4:0);
    statTable.put("DVV (total number of DVV Entries)", dvv.getSize()*16+(SangminConfig.compressNodeId?1:4));
    statTable.put("BW (total number of hashes)", 20);
    statTable.put("LogicalOnDiskSize", 1);
    statTable.put("PhysicalOnDiskSize", size);
    statTable.put("SubscriptionSet", ss);
    statTable.put("TimeStamps", 16);
    statTable.put("NodeId", (SangminConfig.compressNodeId?1:8));
    statTable.put("Signature", sigSize);
    statTable.put("Other", 1);
    statTable.put("SerializationCost", (size - dvv.getSize()*16 - 20 -ss - 16 - (SangminConfig.compressNodeId?1:8) - 1 - sigSize - (SangminConfig.compressNodeId?1:4)));

    return statTable;
  }

  public int getSignatureSize(){
    return this.signature.length;
  }

  protected SerializationHelper obj2BytesInternal(){
    try{

      SerializationHelper serHelper = this.obj2BytesExcludingSignature(); //new SerializationHelper();
      ObjectOutput oos = serHelper.getObjectOutputStream();
      //      
      //      oos.writeObject(this.getObjId().getPath());
      //      oos.writeLong(this.getOffset());
      //      oos.writeLong(this.getLength());
      //      oos.writeLong(this.getAcceptStamp().getLocalClock());
      //      oos.writeLong(this.getAcceptStamp().getNodeId().getIDint());
      //
      //      // realtime AcceptStamp & embargoed flag for tact/embargoed write
      //      oos.writeLong(this.getRTAcceptStamp().getLocalClock());
      //      oos.writeBoolean(this.isEmbargoed());
      //
      //      if(SangminConfig.securityLevel >= SangminConfig.COMPLETE){
      //        
      //        Enumeration<NodeId> e = dvv.getEnumeratedNodes();
      //        List<NodeId> l = Collections.list(e);
      //        Collections.sort(l);
      //        ListIterator<NodeId> iter;
      //        
      //        for(iter = l.listIterator(); iter.hasNext();){
      //          try{
      //            NodeId nodeId = iter.next();
      //            oos.writeLong(nodeId.getIDint());
      //            oos.writeLong(dvv.getStampByServer(nodeId));				  
      //          }catch(Exception e1){
      //            e1.printStackTrace();
      //            System.exit(-1);
      //          }
      //        }
      //
      //        assert(sh.getValue().length == 20);
      //        oos.write(sh.getValue());
      //      }
      //      assert(dh.getHashVal().length == 20);
      //      oos.write(dh.getHashVal());

      if(hasHash){
        oos.writeInt(signature.length);
        oos.write(signature);
      }
      oos.flush();

      return serHelper;
    }catch(Exception e){
      System.out.println(e.toString());
      e.printStackTrace();
      System.exit(1);
    }
    return null;
  }

  protected SerializationHelper obj2BytesExcludingSignature(){

    //return new byte[]{0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19};

    try{
      SerializationHelper serHelper = new SerializationHelper();
      ObjectOutput oos = serHelper.getObjectOutputStream();

      oos.writeObject(this.getObjId().getPath());
      oos.writeLong(this.getOffset());
      oos.writeLong(this.getLength());
      oos.writeLong(this.getAcceptStamp().getLocalClock());
      oos.writeLong(this.getAcceptStamp().getNodeId().getIDint());

      // realtime AcceptStamp & embargoed flag for tact/embargoed write
      oos.writeLong(this.getRTAcceptStamp().getLocalClock());
      oos.writeBoolean(this.isEmbargoed());
      oos.writeBoolean(hasHash);

      if(SangminConfig.securityLevel >= SangminConfig.COMPLETE){

        //        Enumeration<NodeId> e = dvv.getEnumeratedNodes();
        //        List<NodeId> l = Collections.list(e);
        //        Collections.sort(l);
        //        ListIterator<NodeId> iter;
        //        
        //        for(iter = l.listIterator(); iter.hasNext();){
        //          try{
        //            NodeId nodeId = iter.next();
        //            oos.writeLong(nodeId.getIDint());
        //            oos.writeLong(dvv.getStampByServer(nodeId));                                  
        //          }catch(Exception e1){
        //            e1.printStackTrace();
        //            System.exit(-1);
        //          }
        //        }

        if(!SangminConfig.forkjoin){
          serHelper.writeVV(dvv);
        }

        if(hasHash){
          assert(sh.getValue().length == 20);
          oos.write(sh.getValue());
        }
      }
      assert(dh.getHashVal().length == 20);
      oos.write(dh.getHashVal());

      oos.flush();
      //      byte [] obj = bs.toByteArray();
      //      oos.close();
      //      bs.close();
      //      return obj;
      return serHelper;

    }catch(Exception e){
      System.out.println(e.toString());
      e.printStackTrace();
      System.exit(1);
    }

    return null;
  }





  public void writeExternal(ObjectOutput out) throws IOException{
    if(SecurityFilter.debugSerialization){
      out.writeByte(SecurePreciseInv.classCode);
    }

    assert(this.verify((PublicKey)Config.publicKeys.get(new Long(this.getCreator().getIDint())))) : "writeExternal" + this;
    //out.write(this.obj2Bytes());
    out.writeObject(this.getObjId().getPath());
    out.writeLong(this.getOffset());
    out.writeLong(this.getLength());
    out.writeLong(this.getAcceptStamp().getLocalClock());
    //System.out.println("WRITE getRTAcceptStamp : " + this.getAcceptStamp().getLocalClock());
    if(SangminConfig.forkjoin){
      out.writeObject(getAcceptStamp().getNodeId());
    } else {
      if(SangminConfig.compressNodeId){
        out.writeByte((int)getAcceptStamp().getNodeId().getIDint());
      }else{
        out.writeLong(getAcceptStamp().getNodeId().getIDint());
      }
    }
    // realtime AcceptStamp & embargoed flag for tact/embargoed write
    out.writeLong(this.getRTAcceptStamp().getLocalClock());
    //System.out.println("WRITE getRTAcceptStamp : " + this.getRTAcceptStamp().getLocalClock());
    out.writeBoolean(this.isEmbargoed());
    out.writeBoolean(hasHash);

    // security components
    if(SangminConfig.securityLevel >= SangminConfig.COMPLETE){
      if(SangminConfig.compressNodeId){
        out.writeByte(dvv.getSize());
      }else{
        out.writeInt(dvv.getSize());
      }
      //System.out.println("WRITE numNodesInDVV : " + dvv.getSize());
      for( Enumeration<NodeId> enu = dvv.getEnumeratedNodes(); enu.hasMoreElements();){
        try{
          NodeId nodeId = (NodeId)enu.nextElement();
          if(SangminConfig.forkjoin){            
            out.writeObject(nodeId);
          } else {
            if(SangminConfig.compressNodeId){
              out.writeByte((int)nodeId.getIDint());
            }else{
              out.writeLong(nodeId.getIDint());
            }       
          }
          out.writeLong(dvv.getStampByServer(nodeId));                               
        }catch(Exception e){
          e.printStackTrace();
          System.exit(-1);
        }
      }

      if(hasHash){
        assert(sh.getValue().length == 20);
        out.write(sh.getValue());
      }
    }
    assert(dh.getHashVal().length == 20);
    out.write(dh.getHashVal());
    if(hasHash){
      assert(signature.length>=0):signature.length;
      out.writeInt(signature.length);
      out.write(signature);
    }
    out.writeInt(hops);
    if(SecurityFilter.debugSerialization){
      out.writeByte(SecurePreciseInv.classCode);
    }
  }


  protected void readExternalHelper(ObjectInput in)
  throws IOException, ClassNotFoundException{
    String objIdString = (String)(in.readObject());
    long  offset = in.readLong();
    long  length = in.readLong();
    long  localClock = in.readLong();
    //System.out.println("READ localClock : " + localClock);
    NodeId nId;
    if(SangminConfig.forkjoin){
      nId = (BranchID)in.readObject();
    } else {
      if(SangminConfig.compressNodeId){
        nId = new NodeId(in.readByte());
      } else {
        nId = new NodeId(in.readLong());
      }
    }
    /*
    long  nodeIdInt;
    if(SangminConfig.compressNodeId){
      nodeIdInt = in.readByte();
    }else{
      nodeIdInt = in.readLong();
    }*/

    long realTime = in.readLong();
    //System.out.println("READ realTime : " + realTime);
    boolean  myembargoed = in.readBoolean();
    boolean  _hasHash = in.readBoolean();

    DependencyVV _dvv = null;
    SummaryHash _sh = null;

    if(SangminConfig.securityLevel >= SangminConfig.COMPLETE){
      int  numNodesInDVV;
      if(SangminConfig.compressNodeId){
        numNodesInDVV = in.readByte();
      }else{
        numNodesInDVV = in.readInt();
      }
      _dvv = new DependencyVV();
      //System.out.println("READ numNodesInDVV : " + numNodesInDVV);
      for(int i=0; i < numNodesInDVV; i++){
        NodeId nodeId;
        if(SangminConfig.forkjoin){
          nodeId = (BranchID)in.readObject();
        } else {
          if(SangminConfig.compressNodeId){
            nodeId = new NodeId(in.readByte());
          }else{
            nodeId = new NodeId(in.readLong());
          }
        }
        _dvv.put(nodeId, in.readLong());
      }


      byte[] shBuffer = new byte[20];
      if(_hasHash){
        SecureCore.readBytes(in, 20, shBuffer, 0);
        //      int readBB = in.read(shBuffer,0,20);
        //assert readBB == 20: readBB + " " + 20;
        _sh = new SummaryHash(shBuffer, false);
      }else{
        _sh = null;
      }
    }

    byte[] dhBuffer = new byte[20];
    //    int readB = in.read(dhBuffer,0,20);
    SecureCore.readBytes(in, 20, dhBuffer, 0);
    DataHash _dh = new DataHash(dhBuffer, false);
    //    assert readB == 20: readB + " " + 20;

    byte[] _signature;
    if(_hasHash){
      int signatureLength = in.readInt();
      _signature= new byte[signatureLength];
      SecureCore.readBytes(in, signatureLength, _signature, 0);
      //    readB = in.read(_signature,0,signatureLength);
      //assert readB == signatureLength: readB + " " + signatureLength;
    }else{
      _signature = null;
    }
    int _hops = in.readInt();

    Field[] f = new Field[10];

    try{

      f[0] = PreciseInv.class.getDeclaredField("obj");
      f[1] = PreciseInv.class.getDeclaredField("acceptStamp");
      f[2] = PreciseInv.class.getDeclaredField("rtAcceptStamp");
      f[3] = PreciseInv.class.getDeclaredField("embargoed");
      f[4] = SecurePreciseInv.class.getDeclaredField("dvv");
      f[5] = SecurePreciseInv.class.getDeclaredField("sh");
      f[6] = SecurePreciseInv.class.getDeclaredField("dh");
      f[7] = SecurePreciseInv.class.getDeclaredField("signature");
      f[8] = SecurePreciseInv.class.getDeclaredField("hops");
      f[8] = SecurePreciseInv.class.getDeclaredField("hops");
      f[9] = SecurePreciseInv.class.getDeclaredField("hasHash");

    }catch(NoSuchFieldException ne){
      System.err.println(ne.toString());
      ne.printStackTrace();
      System.exit(-1);

    }
    try{
      AccessibleObject.setAccessible(f, true);
    } catch (SecurityException se){
      System.err.println(se.toString());
      se.printStackTrace();
      System.exit(-1);
    }
    try{
      f[0].set(this, new ObjInvalTarget(new ObjId(objIdString),
          offset,
          length));
      f[1].set(this, new AcceptStamp(localClock, nId));
      f[2].set(this, new AcceptStamp(realTime, nId));
      f[3].setBoolean(this, myembargoed);
      f[4].set(this, _dvv);
      f[5].set(this, _sh);
      f[6].set(this, _dh);
      f[7].set(this, _signature);
      f[8].setInt(this, _hops);
      f[9].setBoolean(this, _hasHash);

    }catch(IllegalArgumentException ie){
      System.err.println(ie.toString());
      ie.printStackTrace();
      System.exit(-1);
    }catch(IllegalAccessException iae){
      System.err.println(iae.toString());
      iae.printStackTrace();
      System.exit(-1);
    }

    try{
      AccessibleObject.setAccessible(f, false);
    } catch (SecurityException se){
      System.err.println(se.toString());
      se.printStackTrace();
      System.exit(-1);
    }


  }

  public void readExternal(ObjectInput in) 
  throws IOException, ClassNotFoundException{
    if(SecurityFilter.dbg) System.out.println("reading SecurePreciseInv");
    if(SecurityFilter.debugSerialization){
      byte b = in.readByte();
      assert b == SecurePreciseInv.classCode: b;
    }
    readExternalHelper(in);
    if(SecurityFilter.debugSerialization){
      byte b = in.readByte();
      assert b == SecurePreciseInv.classCode: b;
    }
    assert(this.verify((PublicKey)Config.publicKeys.get(new Long(this.getCreator().getIDint())))) : "readExternal" + this + " available " + in.available();
    //   assert(in.available() == 0):this;

  }

  /** 
   *  Return true if the passed-in object is "equal" to this PreciseInv 
   **/ 
  public boolean
  equals(Object obj){
    boolean result = false;
    SecurePreciseInv spi = null;

    if(obj instanceof SecurePreciseInv){
      spi = (SecurePreciseInv)obj;

      if(SangminConfig.securityLevel >= SangminConfig.COMPLETE){
        result = (super.equals(obj) && 
            this.sh.equals(spi.sh) &&
            this.dh.equals(spi.dh));
      }else{
        result = (super.equals(obj) && 
            this.dh.equals(spi.dh));
      }
      return(result);
    }else{
      return false;
    }
  }

  public int hashCode(){
    return this.getAcceptStamp().hashCode();
  }

  public String toString(){
    String str = super.toString() + " DVV"  + this.dvv + "SH " + sh; // + dh + signature.toString();
    // str += "Security Info <sh: " + sh.toString() +">";

    return str;
  }

  public int getHops(){
    return hops;
  }

  public void increaseHops(){
    hops++;
  }


  public static void main(String[] argv) throws IOException{
    
    int datasize = 1024;
    if(argv.length > 0){
      datasize = Integer.parseInt(argv[0]);
    }
    int mode = 0;
    int All = 0, Hash = 1, SigGen = 2, SigVer =3;
    if(argv.length > 1){
      if(argv[1].contains("All")){
        mode = All;
      }else if(argv[1].contains("Hash")){
        mode = Hash;
      }else if(argv[1].contains("SigGen")){
        mode = SigGen;
      }else if(argv[1].contains("SigVer")){
        mode = SigVer;
      }
    }
    
    try{
      Config.readKeys();
    } catch (Exception e){
      e.printStackTrace();
      System.exit(-1);
    }
    System.out.println("Key reading completed.");
    PrivateKey privKey = (PrivateKey)Config.privateKeys.get(new Long(1));
    PublicKey pubKey = (PublicKey)Config.publicKeys.get(new Long(1));

    byte[] buffer = new byte[]{0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19};

    SummaryHash sh = new SummaryHash(buffer, true);
    //sh.putValue(buffer);
    DataHash dh = new DataHash(buffer, true);
    //dh.putHashVal(buffer);

    SecurePreciseInv spi = new SecurePreciseInv(new ObjInvalTarget(new ObjId("/rand1"), 5, 10),
        new AcceptStamp(1, new NodeId(100)),
        new AcceptStamp(2, new NodeId(100)),
        false,
        new DependencyVV(),
        sh,
        dh,
        privKey);
    assert(spi.verify(pubKey));

    ResultStats stats = new ResultStats();
    ResultStats hstats = new ResultStats();
    ResultStats stats2 = new ResultStats();
    long start = 0,end = 0;

    
//    for(int i=0; i<1000; i++){
//      start = System.currentTimeMillis();
//      spi.sign(privKey);
//      end = System.currentTimeMillis();
//      stats.enter((double)end-start);
//      spi = new SecurePreciseInv(new ObjInvalTarget(new ObjId("/rand1"), 5, 10),
//          new AcceptStamp(1, new NodeId(i)),
//          new AcceptStamp(2, new NodeId(i+5)),
//          false,
//          new DependencyVV(),
//          sh,
//          dh,
//          privKey);
//    }
    
//    ResultStats disk = new ResultStats();
//    char[] cbuf = new char[datasize];
//    long t1,t2;
//    for(int i=0; i<1000; i++){
//      FileWriter f = new FileWriter("/tmp/diskiotest");
//      t1 = System.currentTimeMillis();
//      f.write(cbuf);
//      f.flush();
//      f.close();
//      t2 = System.currentTimeMillis();
//      disk.enter((double)(t2-t1));
//      (new File("/tmp/diskiotest")).delete();
//    }
//    System.out.println("###### Disk write time!!");
//    System.out.println(disk.getSummary(1));
//    System.out.println(disk.getAverage90());
    
    
    //Security.addProvider(new BouncyCastleProvider());
    Signature sig = Crypto.getSignature();
    for(int i=0; i<1000; i++){
      byte[] b = new byte[datasize];
      byte[] signature;
      
      Random r = new Random(System.currentTimeMillis());
      r.nextBytes(b);    

      try{
        start = System.currentTimeMillis();
        Hash h = new Hash(b,true);
        end = System.currentTimeMillis();
        hstats.enter(end-start);
        //sig = Signature.getInstance(privKey.getAlgorithm());
        sig.initSign(privKey);
//        spi.obj2BytesExcludingSignature();
        sig.update(h.getHashVal());
        start = System.currentTimeMillis();
        signature = sig.sign();
        end = System.currentTimeMillis();
        stats.enter((double)end-start);
        //sig = Signature.getInstance(pubKey.getAlgorithm());
        sig.initVerify(pubKey);
        sig.update(h.getHashVal());
        start = System.currentTimeMillis();
        if(!sig.verify(signature)){
          System.out.println("Verification failed. " + i);
          System.exit(-1);
        }
        end = System.currentTimeMillis();
        stats2.enter((double)end-start);
      }catch(Exception e){
        System.out.println(e.toString());
        e.printStackTrace();
        System.exit(-1);
      }      
      //stats.enter((double)end-start);
    }
    Crypto.returnSignature(sig);

    System.out.println("###### hash time!!");
    System.out.println("Summary1:"  + hstats.getSummary(1));
    System.out.println("Summary0:" + hstats.getSummary(0));
    System.out.println(hstats.getAverage90());
    System.out.println("###### Signning time!!");
    System.out.println("Summary1:"  + stats.getSummary(1));
    System.out.println("Summary0:" + stats.getSummary(0));
    System.out.println(stats.getAverage90());
    System.out.println("###### Verifying time!!");
    System.out.println("Summary1:"  + stats2.getSummary(1));
    System.out.println("Summary0:" + stats2.getSummary(0));
    System.out.println(stats2.getSummary(1));
    System.out.println(stats2.getAverage90());


    /** 
    /*    
    DSASigner signer = new DSASigner();
    SecureRandom sr = null;

    try{
      //sr = SecureRandom.getInstance("SHA1PRNG");
      sr = new SecureRandom();
      sr.setSeed("Hello".getBytes());
    }catch(Exception nsa){
      nsa.printStackTrace();
      System.exit(-1);
    }

    DSAKeyGenerationParameters kgp = new DSAKeyGenerationParameters(sr,DESedeParameters.DES_EDE_KEY_LENGTH*8);
    DSAKeyPairGenerator kpg = new DSAKeyPairGenerator();
    kpg.init(kgp);

    AsymmetricCipherKeyPair keyPair = kpg.generateKeyPair();

    CipherParameters bcPriKey = keyPair.getPrivate();
    CipherParameters bcPubKey = keyPair.getPublic();

    SHA1Digest sha1Digest = new SHA1Digest();

    byte [] ba = spi.obj2BytesExcludingSignature();
    System.out.println("Byte length :" + ba.length);


    sha1Digest.update(spi.obj2BytesExcludingSignature(), 0, ba.length);

    byte[] digest = new byte[sha1Digest.getDigestSize()];    
    sha1Digest.doFinal(digest, 0);

    signer.init(true, bcPriKey);

    BigInteger[] bcSignature = signer.generateSignature(digest);

    signer.init(false, bcPubKey);
    if(!signer.verifySignature(digest, bcSignature[0], bcSignature[1])){
      assert(false);
    }else{
      System.out.println("Yeah!");
    }

     */  

    /*    

    for(int i=0; i<1000; i++){
      byte[] b = new byte[150];
      byte[] signature;
      Signature sig;
      Random r = new Random(System.currentTimeMillis());
      r.nextBytes(b);    

      try{
        sig = Signature.getInstance(privKey.getAlgorithm());
        sig.initSign(privKey);
        spi.obj2BytesExcludingSignature();
        sig.update(b);
        signature = sig.sign();

        start = System.currentTimeMillis();
        sig = Signature.getInstance(pubKey.getAlgorithm());
        sig.initVerify(pubKey);

        sig.update(b);

        sig.verify(signature);
        end = System.currentTimeMillis();
      }catch(Exception e){
        System.out.println(e.toString());
        System.out.println(e.getStackTrace());
        System.exit(-1);
      }

      stats.enter((double)end-start);
    }*/




  }
  public boolean isHasHash(){
    return hasHash;
  }


}

