package code.security.ahs;
import code.*;
import code.security.*;
import code.security.holesync.Range;
import code.serialization.SerializationHelper;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Externalizable;
import java.lang.reflect.AccessibleObject;
import java.lang.reflect.Field;
import java.util.Arrays;
import java.util.Hashtable;
import java.io.*;
import java.util.Vector;

/**
 * An single AHSEntry contains all the relevant information for AHS: level, superhash, startTS, endTS, invalTarget and maxDVVMap. 
 * Collection of such entries (upto log in number) forms an AHS value
 * @author princem
 *
 */
public class AHSEntry implements Externalizable, Comparable, Immutable{
  final int level;
  final byte[] hash;
  final long startTS;
  final long endTS;
  final InvalTarget invalTarget;
  final DVVMap maxDVVMap;
  
  final static int classCode = 0x5a;
  final byte[] superhash;

  public AHSEntry()
  {
    level = 0; hash = null; startTS = -1; endTS = -1; invalTarget = null; maxDVVMap = null; 
    superhash = null;
  }
  
  public AHSEntry(int level_, byte[] hash_, long startTS_,
      long endTS_, InvalTarget invalTarget_, DVVMap maxDVVMap_,
      byte[] superhash_){
    level = level_;
    hash = hash_;
    startTS = startTS_;
    endTS = endTS_;
    invalTarget = invalTarget_;
    maxDVVMap = maxDVVMap_;
    superhash = superhash_;
  }
  
  
  
  public AHSEntry(TreeNode treeNode){
    this.level = treeNode.level;
    this.hash = treeNode.hash;
    this.startTS = treeNode.startTS;
    this.endTS = treeNode.endTS;
    this.invalTarget = treeNode.invalTarget;
    this.maxDVVMap = treeNode.maxDVVMap.clone();
    this.superhash = treeNode.superhash;    
  }
  
  public DVVMap getMaxDVVMap(){
    return maxDVVMap;
  }
  
  public long getStartTS(){
    return startTS;
  }
  
  public long getEndTS(){
    return endTS;
  }
  
  public int getLevel(){
    return level;
  }
  
  public byte[] getHash(){
    return hash;
  }
  
  public Range getRange(){
    return new Range(startTS, endTS);
  }
  
  public byte[] getSuperHash(){
  	if(SangminConfig.forkjoin){
  		return superhash;
  	}
    return new TreeNode(this).getSuperHash();
  }
  
  public InvalTarget getInvalTarget(){
    return invalTarget;
  }
  
  public String toString(){
      return "AHSEntry: (level:"+level+", startTS:"+startTS+", endTS:"+endTS+",invalTarget:"+invalTarget+",maxDVV:" + maxDVVMap + (SangminConfig.printDebugHashes?" hash " + DVVMapEntry.byteString(hash):"") + ")";
      //	  return "AHSEntry: (level:"+level+", startTS:"+startTS+", endTS:"+endTS+",invalTarget:"+invalTarget+",maxDVV:" + maxDVVMap.getDVV() + ")";
  }

  public String toLongString(){
      //return "AHS: (level:"+level+", startTS:"+startTS+", endTS:"+endTS+",invalTarget:"+invalTarget+",maxDVV:" + maxDVVMap + " super hash " + DVVMapEntry.byteString(superhash) + " hash " + DVVMapEntry.byteString(hash)+ ")";
	  return "AHSEntry: (level:"+level+", startTS:"+startTS+", endTS:"+endTS+",invalTarget:"+invalTarget+",maxDVV:" + maxDVVMap.getDVV() + ")";
  }

  public boolean equals(Object o){
    if(!(o instanceof AHSEntry)){
      return false;
    }
    
    AHSEntry ae = (AHSEntry)o;
    return level == ae.level &&
           Arrays.equals(hash, ae.hash) &&
           startTS == ae.startTS &&
           endTS == ae.endTS &&
           invalTarget.equals(ae.invalTarget) &&
           maxDVVMap.equals(ae.maxDVVMap) 
//           &&
//           Arrays.equals(superhash, ae.superhash)
           ;
    
  }
  
  public byte[] obj2Bytes(){
    return obj2Bytes(false);
  }
  
  public byte[] obj2Bytes(boolean print){
    byte[] ret = null;
    try{
      ByteArrayOutputStream bs = new ByteArrayOutputStream();
      ObjectOutputStream oos = new ObjectOutputStream(bs);
      int prevLength = bs.toByteArray().length;
      if(print)System.out.println("writing byte 1");
      oos.writeByte(level);
      oos.flush();
      if(print)System.out.println("stream length " + bs.toByteArray().length);

      int newLength = bs.toByteArray().length;
     // assert newLength == 4:newLength + " " + prevLength;
//      assert (newLength - prevLength) == 4:(newLength-prevLength) + " " + prevLength;
     
      if(print)System.out.println("writing long 8");
      oos.writeLong(startTS);
      oos.flush();
      newLength = bs.toByteArray().length;
      //assert (newLength - prevLength) == 8: (newLength-prevLength);
      if(print)System.out.println("stream length " + bs.toByteArray().length);
      
      if(level != 0){
        if(print)System.out.println("writing long 8");
        oos.writeLong(endTS);
        oos.flush();
        if(print)System.out.println("writing hash 20");
        oos.write(hash);
        oos.flush();
        prevLength = bs.toByteArray().length;
        //assert (prevLength - newLength) == 20;
        if(print)System.out.println("stream length " + bs.toByteArray().length);
        
        if(print)System.out.println("stream length " + bs.toByteArray().length);
        
      }else{
        assert invalTarget instanceof ObjInvalTarget: invalTarget;
      }
      
      prevLength = bs.toByteArray().length;
      //assert (prevLength - newLength) == 8: (prevLength-newLength);
//      if(print)System.out.println("writing hash 20");
//      oos.write(superhash);
      oos.flush();
      if(print)System.out.println("stream length " + bs.toByteArray().length);

      newLength = bs.toByteArray().length;
      //assert (-prevLength + newLength) == 8: (-prevLength+newLength);
      oos.write(maxDVVMap.obj2Bytes(print));
      oos.flush();
      if(print)System.out.println("stream length " + bs.toByteArray().length);
      
      oos.writeObject(invalTarget.toString()); // princem modified this code because i think this is correct
//      if(print)System.out.println("writing invalTarget " + invalTarget.toString().length() + " " + invalTarget.toString() );
//      ((HierInvalTarget)invalTarget).writeExternal(oos);
      
//      oos.writeObject(invalTarget);
      oos.flush();
      if(print)System.out.println("stream length " + bs.toByteArray().length);
      
      oos.flush();
      ret = bs.toByteArray();
      oos.close();
      bs.close();
    }catch(Exception e){
      e.printStackTrace();
      System.exit(-1);
    }
    
    return ret;
  }
  
  public Hashtable<String, Integer> getSize(){
    Hashtable<String, Integer> statTable = new Hashtable<String, Integer>();
    
    SerializationHelper sh;
    int size = 0;
    try{
      sh = new SerializationHelper();
      this.writeExternal(sh.getObjectOutputStream());
      size = sh.toBytes().length; //obj2Bytes(true).length;
    }catch(IOException e){
      // TODO Auto-generated catch block
      e.printStackTrace();
    }catch(Exception e){
      // TODO Auto-generated catch block
      e.printStackTrace();
    }
    
    DependencyVV dVV = getMaxDVVMap().getDVV();
    
    int ss;
    
    if(invalTarget instanceof ObjInvalTarget){
      ss = ((ObjInvalTarget)invalTarget).getObjId().getPath().length() + 6;
    }else{
      ss = invalTarget.onDiskSize() + 6; // 6 for the additional unaccounted overhead
    }
    statTable.put("DVV (total number of DVV Entries)", dVV.getSize()*16 + (SangminConfig.compressNodeId?1:4));
    statTable.put("BW (total number of hashes)", dVV.getSize()*20+20);
    statTable.put("LogicalOnDiskSize", 1);
    statTable.put("PhysicalOnDiskSize", size);
    statTable.put("SubscriptionSet", ss);
    statTable.put("TimeStamps", 8 + (level==0?0:8));
    statTable.put("NodeId", 0);
    statTable.put("Signature", 0);
    statTable.put("Other", 4);
    statTable.put("SerializationCost", (size - dVV.getSize()*16 - (SangminConfig.compressNodeId?1:4) - dVV.getSize()*20 - 20 -ss - 8 - (level==0?0:8) - 4));
    if(SecurityFilter.dbg)System.out.println("serializaed object " + size + " serializationCost " + (size - dVV.getSize()*16 - 4 - dVV.getSize()*20 - 20 -ss - 16 - 4));
   // System.out.println("obj2Bytes" + this.obj2Bytes().length);
//    try{
//      SerializationHelper sh = new SerializationHelper();
//      this.writeExternal(sh.getObjectOutputStream());
//      System.out.println("serializaed object size: " + sh.toBytes().length + " obj2Bytes " + size + " serializationCost " + (size - dVV.getSize()*16 - 4 - dVV.getSize()*20 - 40 -ss - 16 - 4));
//    }catch(IOException e){
//      // TODO Auto-generated catch block
//      e.printStackTrace();
//    }catch(Exception e1){
//      e1.printStackTrace();
//    }
    return statTable;
  }
  
  public int compareTo(Object o){
    
    AHSEntry c = (AHSEntry)o;
    
    if(this.startTS < c.startTS){
      return -1;
    }else if(this.startTS > c.startTS){
      return 1;
    }    
    
    return 0;
  }
  
  public void readExternal(ObjectInput in) 
  throws IOException, ClassNotFoundException{
    if(SecurityFilter.debugSerialization){
      byte b = in.readByte();
      assert b == AHSEntry.classCode: b;
    }
    
    int _level = in.readByte();
    byte[] _hash = new byte[20];
    //in.read(_hash, 0, 20);
    
//    System.out.println("Level: " + _level);
    long _startTS = in.readLong();
    long _endTS;
    if(_level != 0){
      _endTS = in.readLong();
      SecureCore.readBytes(in, 20, _hash, 0);
    }else{
      _endTS = _startTS;
      for(int i = 0; i < 20; i++){
        _hash[i] = 0;
      }
    }

    byte code = in.readByte();
    System.out.println("code: " + code);
    InvalTarget _invalTarget = null;
    if(HierInvalTarget.matchClassCode(code)){
      _invalTarget = new HierInvalTarget();
      ((HierInvalTarget)_invalTarget).readExternal(in);
    }else if(ObjInvalTarget.matchClassCode(code)){
      String path = in.readUTF();
      _invalTarget = new ObjInvalTarget(new ObjId(path), 0, 1);
    }else{
      _invalTarget = (InvalTarget)in.readObject();
    }
    DVVMap _maxDVVMap = new DVVMap();
    _maxDVVMap.readExternal(in);
    if(SecurityFilter.debugSerialization){
      byte b = in.readByte();
      assert b == AHSEntry.classCode: b;
    }
//    byte[] _superhash = new byte[20];
//    //in.read(_superhash, 0, 20);
//    SecureCore.readBytes(in, 20, _superhash, 0);
    
    Field[] f = new Field[6];
    try{
      f[0] = AHSEntry.class.getDeclaredField("level");
      f[1] = AHSEntry.class.getDeclaredField("hash");
      f[2] = AHSEntry.class.getDeclaredField("startTS");
      f[3] = AHSEntry.class.getDeclaredField("endTS");
      f[4] = AHSEntry.class.getDeclaredField("invalTarget");
      f[5] = AHSEntry.class.getDeclaredField("maxDVVMap");
//      f[6] = AHSEntry.class.getDeclaredField("superhash");
      
    }catch(NoSuchFieldException ne){
      System.err.println(ne.toString());
      ne.printStackTrace();
      System.exit(-1);
    }
    
    try{
      AccessibleObject.setAccessible(f, true);
    } catch (SecurityException se){
      System.err.println(se.toString());
      se.printStackTrace();
      System.exit(-1);
    }
    
    try{
      f[0].set(this, _level);
      f[1].set(this, _hash);
      f[2].set(this, _startTS);
      f[3].set(this, _endTS);
      f[4].set(this, _invalTarget);
      f[5].set(this, _maxDVVMap);
//      f[6].set(this, _superhash);
    }catch(IllegalArgumentException ie){
      System.err.println(ie.toString());
      ie.printStackTrace();
      System.exit(-1);
    }catch(IllegalAccessException iae){
      System.err.println(iae.toString());
      iae.printStackTrace();
      System.exit(-1);
    }
    
  }
  
  
  public void writeExternal(ObjectOutput out) throws IOException{
    if(SecurityFilter.debugSerialization){
      out.writeByte(AHSEntry.classCode);
    }
    
    
    out.writeByte(level);
    //System.out.println("WRITING LEVEL: " + level);
    
    out.writeLong(startTS);
    if(level != 0){
      out.writeLong(endTS);
      assert hash.length == 20;
      out.write(hash);
    }else{
      assert invalTarget instanceof ObjInvalTarget: invalTarget;
    }
//    ((HierInvalTarget)invalTarget).copySelfOntoOOS(out);

    if(invalTarget instanceof HierInvalTarget){
	//	System.out.println("WRITING CODE: " + HierInvalTarget.getClassCode());
      out.writeByte(HierInvalTarget.getClassCode());
      ((HierInvalTarget)invalTarget).writeExternal(out);
    }else if(invalTarget instanceof ObjInvalTarget){
	//	System.out.println("WRITING CODE: " + ObjInvalTarget.getClassCode());
      out.writeByte(ObjInvalTarget.getClassCode());
      ObjInvalTarget oit = (ObjInvalTarget)invalTarget;
      out.writeUTF(oit.getObjId().getPath());
    }else{
	//	System.out.println("WRITING CODE: 00");
      out.writeByte(0x00);
      out.writeObject(invalTarget);
    }
    
    maxDVVMap.writeExternal(out);
    
    if(SecurityFilter.debugSerialization){
      out.writeByte(AHSEntry.classCode);
    }
    
//    assert superhash.length == 20;
//    out.write(superhash);
    
  }
}
