package code.security.ahs;

import code.*;
import code.security.SecureImpreciseInv;
import code.security.SecureInv;
import code.security.SecurePreciseInv;
import code.security.ahs.*;
import code.security.holesync.*;
import code.security.holesync.filter.*;

import java.util.Enumeration;
import java.util.Vector;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Collection;

public class AHSMap{
  protected Hashtable<NodeId,RootList>     ahsMap;
  public static long inclusionTime = 0;
  public static long linkCheck = 0;
  protected Knowledge knowledge;
  
  public AHSMap(){
    ahsMap = new Hashtable<NodeId,RootList>();
    knowledge = new Knowledge();
  }
  
  public void addFilter(Filter f){
    knowledge.addFilter(f);
  }
  
  public synchronized void close(){
    ahsMap.clear();
  }
  
  public synchronized Knowledge getKnowledge(){
    return knowledge;
  }
  
  public synchronized RootList getRootList(NodeId nodeId){
    if(ahsMap.containsKey(nodeId)){
      return ahsMap.get(nodeId);
    }else{
      RootList ret = new RootList(nodeId);
      ahsMap.put(nodeId, ret);
      return ret;    
    }
  }
  
  public void setFilter(Collection<Filter> f){
    knowledge.setFilter(f);
  }

  public Vector<IH> generateIHs(NodeId nodeId, long startTS, long endTS){
    Vector<IH> v = new Vector<IH>();
    RootList rootList = getRootList(nodeId);
    AHS ahs = rootList.createAHS(startTS, endTS);
    while(ahs.size()>0){
      v.add(new IH(ahs.removeFirst()));
    }
    return v;
  }
  
  public Vector<NodeAHSTuple> generateAHS(VV startVV, VV endVV){
    Vector<NodeAHSTuple> ihtuples = new Vector<NodeAHSTuple>();
    VVIterator vii = endVV.getIterator();
    while(vii.hasMoreElements()){
      Object token = vii.getNext();
      NodeId nodeId = startVV.getServerByIteratorToken(token);
      long startTS = startVV.getStampByIteratorToken(token);
      long endTS = endVV.getStampByIteratorToken(token);
      RootList rootList = getRootList(nodeId);
      ihtuples.add(new NodeAHSTuple(nodeId, rootList.createAHS(startTS, endTS)));
    }
    return ihtuples;
  }
  
  public SecureImpreciseInv generateSII(AcceptVV startVV, AcceptVV endVV, NodeId creator){

    SecureImpreciseInv sii;

    VVIterator vii = startVV.getIterator();

    InvalTarget invTarget = new HierInvalTarget();

    Hashtable<NodeId,Vector<IH>> ihtuples = new Hashtable<NodeId,Vector<IH>>();

    SubscriptionSet ss = SubscriptionSet.makeSubscriptionSet("/*");
    
    while(vii.hasMoreElements()){
      Object token = vii.getNext();
      NodeId nodeId = startVV.getServerByIteratorToken(token);
      long startTS = startVV.getStampByIteratorToken(token);
      long endTS = endVV.getStampByIteratorToken(token);

      ihtuples.put(nodeId, generateIHs(nodeId,startTS,endTS));

      Vector<IH> v = generateIHs(nodeId,startTS,endTS);
      Iterator<IH> i = v.iterator();

      while(i.hasNext()){
        IH ih = i.next();
        invTarget = invTarget.getUnion(ih.getAHSEntry().getInvalTarget(), ss);
      }

    }
    sii = new SecureImpreciseInv(invTarget, startVV, endVV, creator,ihtuples);

    return sii;
  }

  public String toString(){
    String ret = "";

    Enumeration<NodeId> e = ahsMap.keys();
    while(e.hasMoreElements()){
      NodeId nodeId = e.nextElement();
      ret += "NodeId:" + nodeId + " "+ ahsMap.get(nodeId).toString() + "\n";
    }
    return ret;
  }
  
  public String toLongString(){
    String ret = "";

    Enumeration<NodeId> e = ahsMap.keys();
    while(e.hasMoreElements()){
      NodeId nodeId = e.nextElement();
      ret += "NodeId:" + nodeId + " "+ ahsMap.get(nodeId).toLongString() + "\n";
    }
    return ret;
  }
  
  public AHSMap clone(){
    AHSMap ret = new AHSMap();

    Enumeration<NodeId> e = ahsMap.keys();
    while(e.hasMoreElements()){
      NodeId nodeId = e.nextElement();
      ret.ahsMap.put(nodeId, this.ahsMap.get(nodeId).clone());
    }
    ret.knowledge = knowledge.clone();
    return ret;
  }

  // TreeNodePointers are a collection of TreeNodes
  // getNext() on returned a TreeNode returns a TreeNode that has
  // greater startTS than "excludedStartVV"
  public TreeNodePointers getSentPointers(AcceptVV excludedStartVV){
    TreeNodePointers ret = new TreeNodePointers();
    VVIterator vvi = excludedStartVV.getIterator();
    while(vvi.hasMoreElements()){
      NodeId nodeId = (NodeId)vvi.getNext();
      try{
        long ts = excludedStartVV.getStampByServer(nodeId);
        RootList rootList = getRootList(nodeId);
        if(ts != -1){
          TreeNode t = rootList.getTreeNodeTS(ts);
          if(t != null){
            ret.add(nodeId, t);
          }else{
            ret.add(nodeId, rootList.getLastTreeNodePriorTo(ts));
          } 
        }else{
          ret.add(nodeId, rootList.getDummyNode());
        }
      }catch(Exception e){
        System.err.println(e.toString());
        e.printStackTrace();
        System.exit(-1);
      }
      
    }
    
    return ret;
  }
  
  // TreeNodePointers are a collection of TreeNodes
  // getNext() on returned a TreeNode returns a TreeNode that has
  // greater startTS than "excludedStartVV"
  public TreeNodePointers getIncludedReverseSentPointers(AcceptVV includedEndVV){
    TreeNodePointers ret = new TreeNodePointers();
    VVIterator vvi = includedEndVV.getIterator();
    while(vvi.hasMoreElements()){
      NodeId nodeId = (NodeId)vvi.getNext();
      try{
        long ts = includedEndVV.getStampByServer(nodeId);
        RootList rootList = getRootList(nodeId);
        TreeNode t = rootList.getTreeNodeTS(ts);
        if(t == null){
          t = rootList.getLastTreeNodePriorTo(ts);
        }  
        if(//t.getEndTS() <= ts && 
            t.getEndTS() != -1){
          ret.add(nodeId, t);
        }
      }catch(Exception e){
        System.err.println(e.toString());
        e.printStackTrace();
        System.exit(-1);
      }
      
    }
    return ret;
  }

  
  public AcceptVV getPrceedingVV(AcceptVV excludedVV){
    VVIterator vvi = excludedVV.getIterator();
    
    Vector<AcceptStamp> v = new Vector<AcceptStamp>();
    
    while(vvi.hasMoreElements()){
      Object token = vvi.getNext();
      NodeId nodeId = excludedVV.getServerByIteratorToken(token);
      long ts = excludedVV.getStampByIteratorToken(token);
      
      RootList r = this.getRootList(nodeId);
      
      TreeNode t = r.getLastTreeNodePriorTo(ts);
      
      v.add(new AcceptStamp(t.getEndTS(), nodeId));
      
    }
    AcceptStamp[] a = new AcceptStamp[v.size()];
    return new AcceptVV(v.toArray(a));
    
    
  }
  
  public AcceptVV getCurrentVV(){        
    AcceptStamp[] acceptStamps = new AcceptStamp[ahsMap.size()];
    Enumeration<NodeId> e = this.ahsMap.keys();
    int i = 0;
    while(e.hasMoreElements()){
      NodeId nodeId = e.nextElement();
      RootList r = ahsMap.get(nodeId);
      acceptStamps[i++] = new AcceptStamp(r.getEndTS(), nodeId);
    }
    
    return new AcceptVV(acceptStamps);    
  }
  
  public boolean isConsistent(){
	  //for(NodeId nId: ahsMap.keySet()){
		//  ahsMap.get(nId).isConsistent();
	  //}
	  return true;
  }
  
  public boolean isLinear(){
    for(NodeId nId: this.ahsMap.keySet()){
      assert(ahsMap.get(nId).isLinear()): nId + " is not linear \n" + ahsMap.get(nId).toLongString();
    }
    return true;
  }
  
  public Vector<Integer> stats(){
    int size = 0;
    int logicalSize = 0;
    for(NodeId nId: this.ahsMap.keySet()){
      Vector<Integer> s = ahsMap.get(nId).onDiskSize();
      size+= s.get(0);
      logicalSize += s.get(1);
    }
    Vector<Integer> v = new Vector<Integer>();
    
    v.add(size);
    v.add(logicalSize);

    return v;
  }
  
  /**
   * the following set of functions are added to remove references to the RootList in the code
   */
  
  /**
   * Returns an AHS that spans from beginning to as
   * @param as
   * @return
   */
  public synchronized AHS asAHS(NodeId nodeId, long as){
    return getRootList(nodeId).asAHS(as);
  }
  
  public synchronized byte[] getHash(NodeId nodeId){
    return getRootList(nodeId).getHash();
  }
  
  public synchronized void applyAHS(NodeId nodeId, AHS ahs, SecureInv sii) throws UnmatchingTreeNodeException{
    for(AHSEntry ahsEntry: ahs.getSortedListOfEntries()){
      knowledge.apply(nodeId, ahsEntry);
      Range r = getRootList(nodeId).applyAHSEntry(ahsEntry, sii);
      if(r != null){
        updateKnowledge(nodeId, r);
      }
    }
  }
  
  public synchronized void applySecurePreciseInv(SecurePreciseInv spi) throws UnmatchingTreeNodeException{
    knowledge.apply(spi);
    Range r = getRootList(spi.getCreator()).applySecurePreciseInv(spi, this);
    if(r != null){
      updateKnowledge(spi.getCreator(), r);
    }
  }
  
  public synchronized AHS createAHS(NodeId nodeId, long startTS, long endTS){
    return getRootList(nodeId).createAHS(startTS, endTS);
  }
  
  public synchronized TreeNode getLastTreeNodePriorTo(NodeId nodeId, long ts){
    return getRootList(nodeId).getLastTreeNodePriorTo(ts);
  }
  
  public synchronized TreeNode getTreeNodeAfter(NodeId nodeId, long ts){
    return getRootList(nodeId).getTreeNodeAfter(ts);
  }
  
  public synchronized TreeNode getTreeNodeTS(NodeId nodeId, long ts){
    return getRootList(nodeId).getTreeNodeTS(ts);
  }
  
  public synchronized TreeNode getLeafNodeTS(NodeId nodeId, long ts){
    return getRootList(nodeId).getLeafNodeTS(ts);
  }
  
  public synchronized DVVMap getUnverifiedIDVV(NodeId nodeId, long ts){
    return getRootList(nodeId).getUnverifiedIDVV(ts);
  }
  
  public synchronized boolean isEmpty(NodeId nodeId){
    return getRootList(nodeId).isEmpty();
  }
  
  /**
   * update knowledge because a hole corresponding to Range r and NodeId nodeId has been filled
   * @param r
   */
  private void updateKnowledge(NodeId nodeId, Range r){
    //create a list of invals for this range
    LinkedList hf = new LinkedList();
    TreeNode tn = getRootList(nodeId).getTreeNodeTS(r.getStart());
    if(tn.getInv() instanceof SecurePreciseInv){
      hf.add(tn.getInv());
    }else{
      hf.add(new AHSEntry(tn));
    }
    knowledge.apply(new Hole(nodeId, r), hf);
  }
}
