package code.security.ahs;

import code.*;
import code.branchDetecting.BranchID;
import code.security.*;

import java.io.*;
import java.lang.reflect.AccessibleObject;
import java.lang.reflect.Field;

import java.util.Arrays;
import java.util.Hashtable;
import java.util.Enumeration;
import java.util.Map;
import java.util.Collections;
import java.util.List;
import java.util.ListIterator;
import java.util.Set;

/**
 * An extension of VV that also has hash values: a map from nodeId to timestamp and hash value
 * @author princem
 *
 */
public class DVVMap implements Externalizable, Immutable{
  private Hashtable<NodeId,DVVMapEntry>     dvvMap; //key:NodeId , value:DVVMapEntry

  public DVVMap(){
    dvvMap = new Hashtable<NodeId,DVVMapEntry>();
  }

  public DVVMap(Hashtable<NodeId,DVVMapEntry> h){
    dvvMap = new Hashtable<NodeId,DVVMapEntry>();
    for(NodeId n: h.keySet()){
      dvvMap.put(n, h.get(n));
    }
  }

  /**
   * create DVVMap for a parent with the given left and right children
   * @param leftChild
   * @param rightChild
   */
  public DVVMap(DVVMap leftChild, DVVMap rightChild, NodeId nId){
    dvvMap = (Hashtable<NodeId, DVVMapEntry>)leftChild.dvvMap.clone();
    for(NodeId nid: rightChild.dvvMap.keySet()){
      if(!dvvMap.containsKey(nid) || (dvvMap.get(nid).getTimeStamp() < rightChild.dvvMap.get(nid).getTimeStamp())){
        dvvMap.put(nid, rightChild.dvvMap.get(nid));
      }
    }
    if(leftChild.dvvMap.containsKey(nId)){
      dvvMap.put(nId, leftChild.getEntry(nId)); 
    }else{
      dvvMap.remove(nId);
    }
  }
  
  public DVVMap retain(NodeId nodeId){
    DVVMap dvvMap1 = new DVVMap();
    if(dvvMap.containsKey(nodeId)){
      DVVMapEntry dvvMapEntry = dvvMap.get(nodeId);
      dvvMap1.dvvMap.put(nodeId, dvvMapEntry);
    }
    return dvvMap1;
  }

  /**
   * Creates a dvvMap using the given DVV and AHSMap
   * @param dvv
   * @param ahsMap
   */
  public DVVMap(DependencyVV dvv, AHSMap ahsMap){
    dvvMap = new Hashtable<NodeId,DVVMapEntry>();
    Enumeration<NodeId> e = dvv.getEnumeratedNodes();
    while(e.hasMoreElements()){
      NodeId nodeId = e.nextElement();
      try{
        long ts = dvv.getStampByServer(nodeId);
        if(ts < 0 && !SecurityFilter.useDVV){
          continue;
        }else{
          assert(ts>=0) : "NodeId: " + nodeId +",ts: " + ts;
        }
        dvvMap.put(nodeId, new DVVMapEntry(ts,ahsMap.asAHS(nodeId, ts).getHash()));
      }catch(NoSuchEntryException ne){
        assert(false);
      }      
    }    
  }
  
  public boolean contains(NodeId nodeId){
    return dvvMap.containsKey(nodeId);
  }

  public DVVMapEntry getEntry(NodeId nodeId){
    return (DVVMapEntry)dvvMap.get(nodeId);
  }

  public Enumeration<NodeId> getNodes(){
    return dvvMap.keys();
  }

  public Set<NodeId> getNodeSet(){
    return dvvMap.keySet();
  }

  public int size(){
    return dvvMap.size();
  }

  public byte[] obj2Bytes(){
    //System.out.println("will not print");
    return obj2Bytes(false);
  }
  public byte[] obj2Bytes(boolean print){

    if(print)System.out.println(" in DVVMap");

    try{
      ByteArrayOutputStream bs = new ByteArrayOutputStream();
      ObjectOutputStream oos = new ObjectOutputStream(bs);

      Enumeration<NodeId> e = dvvMap.keys();
      List<NodeId> l = Collections.list(e);
      Collections.sort(l);
      ListIterator<NodeId> iter;

      if(print)System.out.println("writing int 4");
      oos.writeInt(dvvMap.size());
      oos.flush();
      if(print)System.out.println("stream length " + bs.toByteArray().length);

      for(iter = l.listIterator(); iter.hasNext();){
        NodeId nodeId = (NodeId)iter.next();
        DVVMapEntry dvvMapEntry = (DVVMapEntry)dvvMap.get(nodeId);
        if(print)System.out.println("writing long 8");
        oos.writeLong(nodeId.getIDint());
        oos.flush();
        if(print)System.out.println("stream length " + bs.toByteArray().length);

        if(print)System.out.println("writing long 8");
        oos.writeLong(dvvMapEntry.getTimeStamp());
        oos.flush();
        if(print)System.out.println("stream length " + bs.toByteArray().length);

        if(print)System.out.println("writing hash 20");
        oos.write(dvvMapEntry.getHashValue());
        oos.flush();
        if(print)System.out.println("stream length " + bs.toByteArray().length);

      }
      oos.flush();
      byte [] obj = bs.toByteArray();
      oos.close();
      bs.close();
      return obj;
    }catch(Exception e){
      e.printStackTrace();
      System.exit(-1);
    }
    return null;
  }

  // if dm has a component that this has,
  // then timestamp of that component should be larger 
  /**
   * extends a DVVMap to the one passed: assumes that the one passed is larger or equal to the original
   */
  public DVVMap extend(DVVMap dm){

    DVVMap dvvMap = this.clone();
    if(dm == null)
    {
      return dvvMap;
    }
    for(NodeId nid: dm.dvvMap.keySet()){
      if(!dvvMap.dvvMap.containsKey(nid) || (dvvMap.dvvMap.get(nid).getTimeStamp() < dm.dvvMap.get(nid).getTimeStamp())){
        dvvMap.dvvMap.put(nid, dm.dvvMap.get(nid));
      }
    }

    return dvvMap;

  }

  public DVVMap clone(){
    Hashtable<NodeId,DVVMapEntry> h = new Hashtable<NodeId,DVVMapEntry>();
    Enumeration<NodeId> e = dvvMap.keys();
    while(e.hasMoreElements()){
      NodeId n = e.nextElement();
      h.put(n, dvvMap.get(n));
    }
    return new DVVMap(h);
  }

  public DependencyVV getDVV(){
    DependencyVV ret = new DependencyVV();
    Enumeration<NodeId> e = dvvMap.keys();
    while(e.hasMoreElements()){
      NodeId nodeId = (NodeId)e.nextElement();
      long ts = ((DVVMapEntry)dvvMap.get(nodeId)).getTimeStamp();
      ret.put(nodeId, ts);
    }
    return ret;
  }


  public String toString(){
    String str = "";
    Enumeration<NodeId> e = dvvMap.keys();
    while(e.hasMoreElements()){
      NodeId nodeId = (NodeId)e.nextElement();
      str += " <" + nodeId + "," + ((DVVMapEntry)dvvMap.get(nodeId)).toString() + "> " ;
    }
    return str;
  }

  public CounterVV getCounterDVV(){
    CounterVV ret = new CounterVV();
    Enumeration<NodeId> e = dvvMap.keys();
    while(e.hasMoreElements()){
      NodeId nodeId = (NodeId)e.nextElement();
      long ts = ((DVVMapEntry)dvvMap.get(nodeId)).getTimeStamp();
      ret.setStampByNodeId(nodeId, ts);
    }
    return ret;
  }

  public boolean equals(Object o){
    if(! (o instanceof DVVMap)){
      return false;
    }
    DVVMap dvvmap = (DVVMap)o;
    if (dvvMap.size() != dvvmap.dvvMap.size()){
      return false;
    }
    Enumeration<NodeId> e = dvvMap.keys();
    while(e.hasMoreElements()){
      NodeId n = e.nextElement();
      if(!dvvmap.dvvMap.containsKey(n)){
        return false;
      }
      DVVMapEntry dme1 = dvvMap.get(n);
      DVVMapEntry dme2 = dvvmap.dvvMap.get(n);
      if(!dme1.equals(dme2)){
        return false;
      }
    }

    return true;
  }

  /**
   * Remove entries also present in vv (nodeId)
   * @param vv
   */
  public DVVMap mask(VV vv){
    DVVMap dvvMap = this.clone();
    for(VVIterator vvi = vv.getIterator(); vvi.hasMoreElements();){
      NodeId nId = vvi.getNext();
      if(dvvMap.dvvMap.containsKey(nId) && (vv.getStampByIteratorToken(nId) >= dvvMap.dvvMap.get(nId).getTimeStamp())){
        dvvMap.dvvMap.remove(nId);
      }
    }

    return dvvMap;
  }

  public boolean isPresentInVV(VV vv){
    if(vv == null){
      assert false;
      return false;
    }
    for(NodeId nId: dvvMap.keySet()){
      if(!vv.containsNodeId(nId)){
        assert false : "vv : " + vv +"\nnid : " + nId;
      return false;
      }
      if(!vv.includes(new AcceptStamp(dvvMap.get(nId).getTimeStamp(), nId))){
        assert false : "vv : " + vv +"\nnid : " + nId + "\ndvv" + getCounterDVV();
      return false;
      }
    }
    return true;
  }

///**
//* Don't know why is it called hackedTryAndSplit and how does it differ from tryAndSplit
//* @param ahsMap
//* @param secureRMIClient
//* @param filter
//* @param ts
//* @param splitNode
//* @param sender
//* @return
//*/
//private boolean hackedTryAndSplit(AHSMap ahsMap, SecureRMIClient secureRMIClient, SecurityFilter filter, long ts, NodeId splitNode, NodeId sender){
//NodeId nodeId = splitNode;  
//TreeNode splitTreeNode = ahsMap.getRootList(nodeId).getTreeNodeTS(ts);
//assert splitTreeNode.getEndTS() >= ts;
//if(splitTreeNode.getEndTS() > ts){
//// make the call
//// apply the returned val
////System.out.println( maxDVV + " " + nodeId + " "  +sender + " splitTreeNode " + splitTreeNode);
//try{
//System.out.println("hackedtryAndSplit contacting remote node" + ts + " nodeId " + nodeId + " sender " + sender);
//ImpreciseInv ii = secureRMIClient.getSplitSecureImpreciseInv(splitTreeNode.getStartTS(), ts, splitTreeNode.getEndTS(), sender, nodeId);
//if(!(ii instanceof SecureImpreciseInv)){
//assert false:ii;
//return false;
//}
//SecureImpreciseInv si = (SecureImpreciseInv)ii;
//if(si instanceof SplitSecureImpreciseInv){
//SplitSecureImpreciseInv ssii = (SplitSecureImpreciseInv)si;
//if(ssii != null){
//// check if the ssii is split at the desired split point
//if(ssii.isSplit(nodeId, ts) && filter.applySplitSecureImpreciseInv(ssii)){
//// do nothing
//}else{
//assert false;
//return false;
//}
//}else{
//assert false;
//return false;
//}
//}else{
//try{
//if(!si.getEndVV().containsNodeId(nodeId) || 
//si.getEndVV().getStampByServer(nodeId) < ts || 
//!si.isSplit(nodeId, ts)){
//assert false;
//return false;
//}else if(!filter.verifyAndApply(si, sender)){
//assert false;
//return false;
//}
//}catch(NoSuchEntryException e){
//// TODO Auto-generated catch block
//e.printStackTrace();
//assert false;
//return false;
//}

//}
//}catch(RMINetworkException e){
//// TODO Auto-generated catch block
//e.printStackTrace();
//assert false;

//}catch(RMIApplicationException e){
//// TODO Auto-generated catch block
//e.printStackTrace();
//assert false;
//return false;
//}
//}
//System.out.println("HackedtryAndSplit split" + ts + " " + splitNode + " returning true");
//return true;
//}
//public boolean verifyDVVMap(AHSMap ahsMap){
////verify each dvvMap component
//Enumeration<NodeId> e = dvvMap.keys();
//while(e.hasMoreElements()){
//NodeId n = e.nextElement();
//DVVMapEntry dvvMapEntry = getEntry(n);
//long ts = dvvMapEntry.getTimeStamp();
//byte[] hash = dvvMapEntry.getHashValue();
//AHS ahs = ahsMap.asAHS(n, ts);
//if(! Arrays.equals(ahs.getHash(),hash)){
//// TODO: rollback and call controller

////RootList r = ahsMap.getRootList(n);
////for(int i=0; i<r.size(); i++){
////TreeNode t = r.NodeAt(i);
////System.out.println("@@@ StartTS: " + t.getStartTS() +" , EndTS: " +t.getEndTS());
////}
////TreeNode t = r.getTreeNodeTS(ts);
////System.out.println(this.getDVV() + " t " + t + " " + ahs);
//return false;
//}
//}
//return true;
//}


  public boolean verifyDVVMap(AHSMap ahsMap, SecureRMIClient rc, NodeId sender, SecurityFilter filter){
    long start;
    if(SecurityFilter.measureTime){
      start = System.currentTimeMillis();
    }
    try{
      //verify each dvvMap component
      Enumeration<NodeId> e = dvvMap.keys();
      while(e.hasMoreElements()){
        NodeId n = e.nextElement();
        DVVMapEntry dvvMapEntry = getEntry(n);
        long ts = dvvMapEntry.getTimeStamp();
        byte[] hash = dvvMapEntry.getHashValue();
        AHS ahs = ahsMap.asAHS(n, ts);
        // ensure that ahsMap is split across ts:
        TreeNode tn = ahsMap.getTreeNodeTS(n, ts);

//      TreeNode tn = ahsMap.getLeafNodeTS(n, ts);
        if(tn.getEndTS() != ts){
          System.out.println("AHS not split across received values" + tn + " ts " +ts);
          return false;
        }
        else if(! Arrays.equals(ahs.getHash(),hash)){
          // TODO: rollback and call controller

//        RootList r = ahsMap.getRootList(n);
//        for(int i=0; i<r.size(); i++){
//        TreeNode t = r.NodeAt(i);
//        System.out.println("@@@ StartTS: " + t.getStartTS() +" , EndTS: " +t.getEndTS());
//        }
//        TreeNode t = r.getTreeNodeTS(ts);
//        System.out.println(this.getDVV() + " t " + t + " " + ahs);

//        princem: commented this as this is not allowed under the current model
//        if(this.hackedTryAndSplit(ahsMap, rc, filter, ts, n, sender)){
//        continue;
//        }else{
//        System.out.println("weird scenario: shouldn't occur");
//        return false;
//        }
	    System.out.println("AHS: " + ahs + " hash " + DVVMapEntry.byteString(ahs.getHash()) + " actual hash " + DVVMapEntry.byteString(hash));
          return false;
        }
      }
      return true;
    }
    finally{
      if(SecurityFilter.measureTime){
        AHSMap.inclusionTime += (System.currentTimeMillis() - start);
      }
    }

  }

  public void readExternal(ObjectInput in) 
  throws IOException, ClassNotFoundException{
    Hashtable<NodeId, DVVMapEntry> _dvvMap = new Hashtable<NodeId, DVVMapEntry>();
    int numKeys;
    if(SangminConfig.compressNodeId){
      numKeys = in.readByte();
    }else{
      numKeys = in.readInt();
    }
    
    for(int i=0; i<numKeys; i++){
      NodeId nodeId;
      if(SangminConfig.forkjoin){
       nodeId = (BranchID)in.readObject(); 
      } else {
        if(SangminConfig.compressNodeId){
          nodeId = new NodeId(in.readByte());
        }else{
          nodeId = new NodeId(in.readLong());
        }
      }
      DVVMapEntry dvvMapEntry = new DVVMapEntry();
      dvvMapEntry.readExternal(in);
      _dvvMap.put(nodeId, dvvMapEntry);
    }

    Field[] f = new Field[1];
    try{
      f[0] = DVVMap.class.getDeclaredField("dvvMap");   
    }catch(NoSuchFieldException ne){
      System.err.println(ne.toString());
      ne.printStackTrace();
      System.exit(-1);
    }

    try{
      AccessibleObject.setAccessible(f, true);
    } catch (SecurityException se){
      System.err.println(se.toString());
      se.printStackTrace();
      System.exit(-1);
    }

    try{
      f[0].set(this, _dvvMap); 
    }catch(IllegalArgumentException ie){
      System.err.println(ie.toString());
      ie.printStackTrace();
      System.exit(-1);
    }catch(IllegalAccessException iae){
      System.err.println(iae.toString());
      iae.printStackTrace();
      System.exit(-1);
    }
  }
  

  public void writeExternal(ObjectOutput out) throws IOException{
    if(SangminConfig.compressNodeId){
      out.writeByte(dvvMap.size());
    }else{
      out.writeInt(dvvMap.size());
    }
    for(NodeId nodeId : dvvMap.keySet()){
      if(SangminConfig.forkjoin){
        out.writeObject(nodeId);
      } else {
        if(SangminConfig.compressNodeId){
          out.writeByte((int)nodeId.getIDint());
        }else{
          out.writeLong(nodeId.getIDint());
        }
      }
      
      dvvMap.get(nodeId).writeExternal(out);
    }
  }
  
  /**
   * Update oldId to newId if the oldId's time stamp is greater than 
   * or equal to timestamp 
   * @param oldId NodeId to be updated
   * @param timestmp
   * @param newId new NodeId
   * @return true if this dvvMap has oldId component to be updated, 
   * false otherwise
   */
  public boolean updateNodeId(NodeId oldId, long timestamp, NodeId newId){
    if(!dvvMap.containsKey(oldId))
      return false;
    DVVMapEntry val = dvvMap.remove(oldId);
    if(val.getTimeStamp() >= timestamp){    
      dvvMap.put(newId, val);
      return true;
    } else {
      dvvMap.put(oldId, val);
      return false;
    }
  }

}
