package code.security.application.MapShare.trace;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Random;
import java.util.SortedSet;
import java.util.TreeSet;

import code.security.application.MapShare.MapManager;

public class TraceGenerator{

  static void debug(String str){
    // System.out.println("[DEBUG] "+str);
  }

  enum Policy{
    SIMPLE, NONSIMPLE
  }

  Policy policy;
  double interestSetPortion;
  double intraGroupSyncPortion;

  int numNodes;
  int numObj;
  int numGroups;
  
  int numObjGroups;

  ArrayList<String> objects;
  //ArrayList<ObjGroup> objGroups;
  ArrayList<TraceNode> nodes;
  ArrayList<NodeGroup> groups;

  SortedSet<Operation> ops;
  long randomSeed;

  Random r;
  
  boolean useGlobalServer;

  TraceGenerator(){
    numNodes = 100;
    numObj = 100;
    //numObjGroups = numObj/10;
    numGroups = 1;

    objects = new ArrayList<String>();
    nodes = new ArrayList<TraceNode>();
    groups = new ArrayList<NodeGroup>();

    ops = new TreeSet<Operation>();
    interestSetPortion = 1.0; // default
    intraGroupSyncPortion = 1.0; // default
    randomSeed = System.currentTimeMillis(); // default
    policy = Policy.NONSIMPLE;
    useGlobalServer = false;
  }

  void init(){

    r = new Random(randomSeed);
    // populate objects
    for(int i = 0; i < numObj; i++){
      objects.add(new String("" + i));
    }
    /*
    for(int i=0; i < numObjGroups; i++){
      
    }
     */
    
    TraceNode globalServer = null;
    
    // create node groups
    for(int j = 0; j < numGroups; j++){
      groups.add(new NodeGroup(j));
    }

    int approximatedGroupSize = numNodes / numGroups;
    // populate nodes
    for(int i = 0; i < numNodes; i++){
      TraceNode n;

      int groupId = i % numGroups;
      NodeGroup g = groups.get(groupId);

      n = new TraceNode(i, groupId);
      g.addNode(n);

      if (useGlobalServer && globalServer == null){
        n.setIS(objects);
        globalServer = n;
      }
      
      // set trusted server
      if(g.trustedServer == null){
        if(useGlobalServer){
          g.setTrustedServer(globalServer);  
        }else{                            
        n.setIS(objects);
        debug("Setting server of group " + g.id + " to node " + n.id);
        g.setTrustedServer(n);
        }
      }

      nodes.add(n);
    }
  }
  
  /*
  private class ObjGroup{
    private ArrayList<String> objs;
    private int id;
    
    public ObjGroup(int id){
      objs = new ArrayList<String>();
    }
    
    public void addObj(String obj){
      objs.add(obj);
    }
    
    public String getRandomObj(){
      return objs.get(r.nextInt(objs.size()));
    }
  }
  */
  private class NodeGroup{
    final int id;

    private TraceNode trustedServer;
    private ArrayList<TraceNode> nodeList;
    private ArrayList<String> objList;
    
    public NodeGroup(int id){
      this.id = id;
      nodeList = new ArrayList<TraceNode>();
    }

    public void setTrustedServer(TraceNode node){
      trustedServer = node;
    }

    public boolean addNode(TraceNode node){
      return nodeList.add(node);
    }

    public boolean isMember(TraceNode node){
      return nodeList.contains(node);
    }

    public TraceNode getRandomNode(){
      return nodeList.get((r.nextInt(nodeList.size())));
    }

  }

  private class TraceNode{
    final int id;
    final int group;
    private ArrayList<String> interestSet;
    //private ArrayList<ObjGroups> interestSet;

    public TraceNode(int id, int group){
      this.id = id;
      this.group = group;
      int sizeOfSet = (int) (numObj * interestSetPortion);
      interestSet = new ArrayList<String>(sizeOfSet);

      while(interestSet.size() < sizeOfSet){
        String obj;
        
        obj = objects.get(r.nextInt(numObj));
        /*
        if(id < numNodes/2){
          if(r.nextInt(10) < 9){
            obj = objects.get(r.nextInt(numObj/2));
          } else {
            obj = objects.get(r.nextInt(numObj/2)+numObj/2);
          }
        }else {
          if(r.nextInt(10) < 9){
            obj = objects.get(r.nextInt(numObj/2)+ (numObj /2));
          } else {
            obj = objects.get(r.nextInt(numObj/2));
          }
        }*/
        if(!interestSet.contains(obj)){
          interestSet.add(obj);
        }

      }
    }

    public void setIS(ArrayList<String> IS){
      interestSet = IS;
    }

    public String getIS(){
      String ret = "";
      for(String obj : interestSet){
        ret += "/" + obj + ":";
      }
      ret = ret.substring(0, ret.length() - 1);
      return ret;
    }

  }

  abstract class Operation implements Comparable{
    final String opName;
    long num;
    final long startTime;
    final long interval;
    long nextTime;

    public Operation(String opName, long num, long startTime, long interval){
      this.opName = opName;
      this.num = num;
      this.startTime = startTime;
      this.interval = interval;

      nextTime = startTime;
    }

    abstract String action();

    public String perform(){
      if(num <= 0)
        return null;
      debug("perform " + this.opName);
      String ret = action();
      nextTime += interval;
      num--;
      return ret;
    }

    public int compareTo(Object arg0){
      Operation op = (Operation) arg0;

      int ret = this.nextTime < op.nextTime ? -1
          : this.nextTime == op.nextTime ? 0 : 1;
      if(ret == 0){
        return this.opName.compareToIgnoreCase(op.opName);
      }else{
        return ret;
      }
    }

    public boolean equals(Object o){
      if(o instanceof Operation){
        return this.opName.equals(((Operation) o).opName);
      }

      return false;
    }
  }

  class UpdateOperation extends Operation{

    public UpdateOperation(long num, long startTime, long interval){
      super("update", num, startTime, interval);

    }

    @Override
    String action(){
      String cmd = null;

      TraceNode node;
      boolean isServer;

      do{
        isServer = false;
        node = nodes.get(r.nextInt(numNodes));
        debug("picked " + node.id);
        for(NodeGroup g : groups){
          if(node.id == g.trustedServer.id){
            isServer = true;
            break;
          }
        }
      }while(isServer);// && policy == Policy.SIMPLE);

      String obj = node.interestSet.get(r.nextInt(node.interestSet.size()));
      cmd = TraceExecutor.UPDATE + " " + node.id + " " + obj + " "
          + r.nextInt(10);

      if(policy == Policy.SIMPLE){
        // tell server to receive data from the updating node
        NodeGroup g = groups.get(node.group);
	//        cmd += "\n" + TraceExecutor.STARTCONNECTION + " " + g.trustedServer.id
	//  + " " + node.id;
      }

      return cmd;
    }
  }

  class SyncOperation extends Operation{
    int l;
    
    public SyncOperation(long num, long startTime, long interval){
      super("sync", num, startTime, interval);
      l = (int)(num/10);
    }

    @Override
    String action(){
      String cmd;
      NodeGroup g = groups.get(r.nextInt(groups.size()));
      TraceNode src = g.getRandomNode();

      while(policy == Policy.SIMPLE && src.id == g.trustedServer.id){
        src = g.getRandomNode();
      }

      TraceNode dst = null;
      if(r.nextDouble() < intraGroupSyncPortion || groups.size() <= 1){
        // intra-group sync
        dst = g.getRandomNode();
        // no sync to itself
        while(src.id == dst.id
            || (policy == Policy.SIMPLE && dst.id == g.trustedServer.id)){
          dst = g.getRandomNode();          
          /*
          if(src.id < numNodes/2){
            if(r.nextInt(10)<9){
              dst = nodes.get(r.nextInt(numNodes/2));
            } else {
              dst = nodes.get(r.nextInt(numNodes/2)+ (numNodes /2));
            }
          } else {
            if(r.nextInt(10)<9){
              dst = nodes.get(r.nextInt(numNodes/2)+ (numNodes /2));
            } else {
              dst = nodes.get(r.nextInt(numNodes/2));
            }
          }*/
        }
      }else{
        // inter-group sync
        NodeGroup dstGrp = groups.get(r.nextInt(groups.size()));
        while(g.id == dstGrp.id){
          dstGrp = groups.get(r.nextInt(groups.size()));
        }

        dst = dstGrp.getRandomNode();
        while(policy == Policy.SIMPLE && dst.id == dstGrp.trustedServer.id){
          dst = g.getRandomNode();
        }
      }

      cmd = TraceExecutor.STARTCONNECTION + " " + src.id + " " + dst.id;      
      
      if( groups.size() >= 2 && policy == Policy.NONSIMPLE && num % l == 0){
        cmd += "\n"+ TraceExecutor.STARTCONNECTION + " "
        + groups.get(0).trustedServer.id + " "
        + groups.get(1).trustedServer.id;
        cmd += "\n"+ TraceExecutor.STARTCONNECTION + " "
        + groups.get(1).trustedServer.id + " "
        + groups.get(0).trustedServer.id;
      }
      
      return cmd;
    }

  }

  void readConfig(File configFile) throws IOException{

    BufferedReader reader = new BufferedReader(new FileReader(configFile));

    for(;;){
      String str = reader.readLine();
      if(str == null){
        return;
      }

      if(str.startsWith("#")){
        outCmd("# " + str);
        continue;
      }
      
      outCmd("# " + str);
      
      
      String[] line = str.split("\\s+");

      if(line[0].equalsIgnoreCase("Policy")){
        if(line[1].equalsIgnoreCase("simple"))
          policy = Policy.SIMPLE;
        else
          policy = Policy.NONSIMPLE;
        debug("Setting Policy to " + policy);
      }

      if(line[0].equalsIgnoreCase("numNodes")){
        numNodes = Integer.parseInt(line[1]);
        debug("Setting numNodes to " + numNodes);
      }
      if(line[0].equalsIgnoreCase("numGroups")){
        numGroups = Integer.parseInt(line[1]);
        debug("Setting numGroups to " + numGroups);
      }

      if(line[0].equalsIgnoreCase("numObjects")){
        numObj = Integer.parseInt(line[1]);
        debug("Setting numObjects to " + numObj);
      }
      
      if(line[0].equalsIgnoreCase("numObjGroups")){
        numObjGroups = Integer.parseInt(line[1]);
        debug("Setting numObjects to " + numObjGroups);
      }
     
      if(line[0].equalsIgnoreCase("singleServer")){
        
        if ( Integer.parseInt(line[1]) == 1 ) {
          useGlobalServer = true;
          debug("Use global server");
        } else {
          useGlobalServer = false;
          debug("Do not use global server");
        }
        
      }
      
      

      if(line[0].equalsIgnoreCase("ISPortion")){
        interestSetPortion = Double.parseDouble(line[1]);
        debug("Setting ISPortion to " + interestSetPortion);
      }

      if(line[0].equalsIgnoreCase("IntraGroupPortion")){
        intraGroupSyncPortion = Double.parseDouble(line[1]);
        debug("Setting IntraGroupPortion to " + intraGroupSyncPortion);
      }

      if(line[0].equalsIgnoreCase("seed")){
        randomSeed = Long.parseLong(line[1]);
        debug("Setting randomSeed to " + randomSeed);
      }

      if(line[0].equalsIgnoreCase("update")){
        debug("Adding update op");
        ops.add(new UpdateOperation(Long.parseLong(line[1]), Long
            .parseLong(line[2]), Long.parseLong(line[3])));
      }

      if(line[0].equalsIgnoreCase("sync")){
        debug("Adding sync op");
        ops.add(new SyncOperation(Long.parseLong(line[1]), Long
            .parseLong(line[2]), Long.parseLong(line[3])));
      }

    }

  }

  private void outCmd(String cmd){
    System.out.println(cmd);
  }

  public void generateTrace(){
    String cmd;

    for(TraceNode node : nodes){
      NodeGroup g = groups.get(node.group);
      debug("## node " + node.id);
      debug("## group " + g.id);
      debug("## server " + g.trustedServer.id);
      cmd = TraceExecutor.SETTRUSTEDNODE + " " + node.id + " "
          + g.trustedServer.id;
      outCmd(cmd);
      cmd = TraceExecutor.SETLIVENESSFILTER + " " + node.id + " ";
      if(policy == Policy.SIMPLE){
        cmd += MapManager.LIVENESS_TRACE_FILTER;
      } else {
        cmd += MapManager.LIVENESS_NORMAL_FILTER;
      }
      outCmd(cmd);
      
      cmd = TraceExecutor.STARTNODE + " " + node.id;
      outCmd(cmd);
      
      cmd = TraceExecutor.SETINTERESTSET + " " + node.id + " " + node.getIS();
      outCmd(cmd);
    }

    while(ops.size() > 0){
      Operation op = ops.first();
      ops.remove(op);
      cmd = op.perform();
      if(cmd != null){
        outCmd(cmd);
        ops.add(op);
      }
    }

  }

  public static void main(String[] args){

    TraceGenerator gen = new TraceGenerator();

    try{
      gen.readConfig(new File(args[0]));
    }catch(IOException e){
      System.err.println(e.getLocalizedMessage());
      System.exit(-1);
    }

    gen.init();

    gen.generateTrace();

  }

}

/*
 * Example of config file
 * 
 * 
 * # Policy (simple or not) 
 * # all other value will be considered non-simple
 * Policy simple
 * 
 * # number of nodes 
 * numNodes 10
 * 
 * # number of groups 
 * numGroups 2
 * 
 * # use single global server (single server regardless # of groups)
 * # 1 : yes , otherwise server per each group (default : no)
 * singleServer 1
 * 
 * # number of Objects 
 * numObjects 10
 * 
 * ------------ ( not implemented )
 * # Object Group info 
 * # the following example indicates three groups
 * # first group consists of 100 objs and the other two consist of
 * # 50 objs each 
 * objectGroup 100 50 50
 * 
 * # Node Group info ( # of nodes for each node group )
 * nodeGroup 10 10 10
 * 
 * # specifying to which obj group and with what proportion 
 * # each node group will write
 * # (in this e.g., first node grp write to the first object group
 * # with 50% of chance, to the second object group with 0%
 * # and to the third group with 50%) 
 * nodeGrp2objGrp 0.5 0 0.5 1.0 0 0 0 1.0 0
 * --------------
 * 
 * # portion of interest set for each node 
 * ISPortion 0.5
 * 
 * # portion of intra group sync 
 * IntraGroupPortion 0.5
 * 
 * # random seed (if not given, use system time) 
 * seed 7
 * 
 * #opname      N_Ops   startTime       interval 
 * update       10         0               10 
 * sync         5          2               15
 */
