package code.security.application.MapShare.trace;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Random;
import java.util.SortedSet;
import java.util.TreeSet;

import code.security.application.MapShare.MapManager;

public class TraceGenerator2{

  static void debug(String str){
    //System.out.println("[DEBUG] "+str);
  }

  enum Policy{
    SIMPLE, NONSIMPLE, FORKJOIN
  }

  Policy policy;
  //double interestSetPortion;
  //double intraGroupSyncPortion;

  int numTotalNodes;
  //int numObj;
  //int numGroups;

  //int numObjGroups;

  //ArrayList<String> objects;
  ArrayList<ObjGroup> objGroups;
  //ArrayList<TraceNode> nodes;
  ArrayList<NodeGroup> nodeGroups;

  // key:node group id, value: array of proportion of objects for ecah obj group
  HashMap <Integer, Double[]> ISPortion;
  // key: node group id, value: array of proportion with which nodes 
  // in the node group sync to each node group  
  HashMap <Integer, Double[]> syncTo;

  SortedSet<Operation> ops;
  long randomSeed;

  Random r;
  Random r2;
  boolean useGlobalServer;

  int numRepeats;

  // map from parent node id to children node id
  HashMap <Integer, Integer[]> forkedNodes;
  
  boolean useTreeTopology = false;
  int branchFactor;
  int numParentsPerNode;
  boolean mirrorImage = false;

  TraceGenerator2(){
    //numNodes = 100;
    //numObj = 100;
    //numObjGroups = numObj/10;
    //numGroups = 1;

    //objects = new ArrayList<String>();
    //nodes = new ArrayList<TraceNode>();
    nodeGroups = new ArrayList<NodeGroup>();
    objGroups = new ArrayList<ObjGroup>();
    ISPortion = new HashMap<Integer, Double[]>();
    syncTo = new HashMap<Integer, Double[]>();

    ops = new TreeSet<Operation>();
    //interestSetPortion = 1.0; // default
    //intraGroupSyncPortion = 1.0; // default
    randomSeed = System.currentTimeMillis(); // default
    policy = Policy.NONSIMPLE;
    useGlobalServer = false;
    numRepeats = 0;

    forkedNodes = new HashMap<Integer, Integer[]>();
  }

  void init(){

    r = new Random(randomSeed);
    r2 = new Random(randomSeed);
    TraceNode globalServer = null;

    // initialize node groups
    int nodeid = 0;
    for(NodeGroup g : nodeGroups){
      int gid = g.id;
      debug("initialzed node group " + gid);
      Double[] isPortion = ISPortion.get(gid);
      if(isPortion == null){
        System.err.println("ISportion for node group " + gid + " not found");
        System.exit(-1);
      }
      if(policy==Policy.FORKJOIN){
        g.setTrustedServer(new TraceNode(-1,-1));
      }
      // create nodes for this group
      for(int i=0; i < g.numNodes; i++){
        TraceNode n = new TraceNode(nodeid++, gid);
        g.addNode(n);

        // populate interest set of this node
        for(int objGrp=0; objGrp < objGroups.size(); objGrp++){
          ObjGroup og = objGroups.get(objGrp);
          if(og == null){
            System.err.println("ISportion for obj group " + objGrp + " not found");
            System.exit(-1);
          }
          int numObjsToAdd = (int) (og.objs.size() * isPortion[objGrp]);
          int numObjsAdded = 0;
          debug("Adding " + numObjsToAdd + " objs to node "+n.id);
          debug("" + og.objs.size() + " * " + isPortion[objGrp]);
          while(numObjsAdded < numObjsToAdd){
            String obj = og.getRandomObj();
            if(!n.interestSet.contains(obj)){
              n.interestSet.add(obj);
              numObjsAdded++;
            }
          }
        }

        if(policy != Policy.FORKJOIN){
          // Set global server
          if (useGlobalServer && globalServer == null){
            globalServer = n;
            n.interestSet.clear();
            for(ObjGroup og : objGroups){
              for(String obj : og.objs){
                if(!n.interestSet.contains(obj)){
                  n.interestSet.add(obj);
                }
              }
            }
          }
          // set trusted server for this node group
          if (g.trustedServer == null){
            if (useGlobalServer){
              g.setTrustedServer(globalServer);
            } else {
              n.interestSet.clear();
              for(ObjGroup og : objGroups){
                for(String obj : og.objs){
                  if(!n.interestSet.contains(obj)){
                    n.interestSet.add(obj);
                  }
                }
              }
              g.setTrustedServer(n);
            }
          }



        }

      }
      if(useTreeTopology){
        buildTree(g);
      }
      
    }

  }

  
  private void buildTree(NodeGroup g){
    int branchFactor = this.branchFactor;
    // Tree topology
    Random random = new Random();
    ArrayList<TraceNode> list = new ArrayList<TraceNode>();
    ArrayList<TraceNode> parents = new ArrayList<TraceNode>();
    ArrayList<TraceNode> children = new ArrayList<TraceNode>();
    list.addAll(g.nodeList);
    //int nIntialParents = 1; // number of parents
    int nParentsPerChild = this.numParentsPerNode;
    ArrayList<TraceNode> roots = new ArrayList<TraceNode>();
    while(list.size()>0){
      
      if(parents.isEmpty()){
        for(int i=0; i < nParentsPerChild; i++){
          TraceNode n = list.remove(random.nextInt(list.size()));
          parents.add(n);
          roots.add(n);
        }
      }
      
      for(int j=0; j < parents.size()*branchFactor; j++){
        if(list.size() <=0){
          break;
        }
        TraceNode n = list.remove(random.nextInt(list.size()));
        children.add(n);
        for(int k=0; k < nParentsPerChild; k++){
          TraceNode p = parents.get((j+k)%parents.size());
          p.children.add(n);
          n.parents.add(p);
        }
      }
      
      parents = children;
      children = new ArrayList<TraceNode>();
      
    }
    
    if(mirrorImage){
      List<TraceNode> clones = cloneTree(g);
      for(TraceNode n : clones){
        g.addNode(n);
      }
    }
    
    System.out.println("# Tree....");
    for(TraceNode n : g.nodeList){
      String s = "Parents: ";
      for(TraceNode p : n.parents){
        s += p.id + " ";
      }
      s += "Children: ";
      for(TraceNode c : n.children){
        s += c.id + " ";
      }
      System.out.println("# Node ID : " + n.id + " " +s);
       
    }
    
    

  } // end of buildTree()
  
  private ArrayList<TraceNode> cloneTree(NodeGroup g){
    
    ArrayList<TraceNode> ret = new ArrayList<TraceNode>();
    
    for(TraceNode n : g.nodeList){
      TraceNode clone = n.shallowClone(g.numNodes);
      ret.add(clone);
      
      for(TraceNode p : n.parents){
        p.children.add(clone);
      }
      
      for(TraceNode c : n.children){
        c.parents.add(clone);
      }
    }
    return ret;
    
    
    
//    for(TraceNode c : root.children){
//      cloneTree(g, c);
//    }
    
  }

  private class ObjGroup{
    private ArrayList<String> objs;
    private int id;

    public ObjGroup(int id, int size){
      this.id = id;
      objs = new ArrayList<String>(size);
      for (int i=0; i < size; i++){
        objs.add(i, "/" + id + "/" + i);
      }
    }

    public void addObj(String obj){
      objs.add(obj);
    }

    public String getRandomObj(){
      return objs.get(r.nextInt(objs.size()));
    }
  }

  private class NodeGroup{
    final int id;

    private TraceNode trustedServer;
    private ArrayList<TraceNode> nodeList;
    private int numNodes;
    //private ArrayList<String> objList;

    public NodeGroup(int id, int numNodes){
      this.id = id;
      this.numNodes = numNodes;
      numTotalNodes += numNodes;
      nodeList = new ArrayList<TraceNode>(numNodes);
    }

    public void setTrustedServer(TraceNode node){
      trustedServer = node;
    }

    public boolean addNode(TraceNode node){
      return nodeList.add(node);
    }

    public boolean isMember(TraceNode node){
      return nodeList.contains(node);
    }

    public TraceNode getRandomNode(){
      return nodeList.get((r.nextInt(nodeList.size())));
    }

    public NodeGroup getSyncTarget(){
      double d = r.nextDouble();
      Double[] darr = syncTo.get(this.id);
      double sum = 0;
      for(int i=0; i < darr.length; i++){
        sum += darr[i];
        if(d < sum){
          return nodeGroups.get(i);
        }
      }
      System.err.println("Must not reach here! : Configuration seems wrong");
      System.exit(-1);
      return this;
    }

  }

  private class TraceNode{
    final int id;
    final int group;
    private ArrayList<String> interestSet;
    //private ArrayList<ObjGroups> interestSet;
    private List<TraceNode> parents;
    private List<TraceNode> children;
    
    public TraceNode(int id, int group){
      this.id = id;
      this.group = group;
      interestSet = new ArrayList<String>();
      parents = Collections.synchronizedList(new ArrayList<TraceNode>());
      children = Collections.synchronizedList(new ArrayList<TraceNode>());
    }

    public void setIS(ArrayList<String> IS){
      interestSet = IS;
    }

    public String getIS(){
      String ret = "";
      for(String obj : interestSet){
        ret += obj + ":";
      }
      ret = ret.substring(0, ret.length() - 1);
      return ret;
    }
    
    public TraceNode shallowClone(int idOffset){
      TraceNode clone = new TraceNode(id + idOffset, group);
      clone.interestSet = (ArrayList<String>) this.interestSet.clone();
      clone.parents = Collections.synchronizedList((new ArrayList<TraceNode>(this.parents)));
      clone.children = Collections.synchronizedList((new ArrayList<TraceNode>(this.children)));
      return clone;
    }

  }

  abstract class Operation implements Comparable{
    final String opName;
    long num;
    final long startTime;
    final long interval;
    long nextTime;

    public Operation(String opName, long num, long startTime, long interval){
      this.opName = opName;
      this.num = num;
      this.startTime = startTime;
      this.interval = interval;

      nextTime = startTime;
    }

    abstract String action();

    public String perform(){
      if(num <= 0)
        return null;
      debug("perform " + this.opName);
      String ret = action();
      nextTime += interval;
      num--;
      return ret;
    }

    public int compareTo(Object arg0){
      Operation op = (Operation) arg0;

      int ret = this.nextTime < op.nextTime ? -1
          : this.nextTime == op.nextTime ? 0 : 1;
      if(ret == 0){
        return this.opName.compareToIgnoreCase(op.opName);
      }else{
        return ret;
      }
    }

    public boolean equals(Object o){
      if(o instanceof Operation){
        return this.opName.equals(((Operation) o).opName);
      }

      return false;
    }
  }

  class UpdateOperation extends Operation{

    public UpdateOperation(long num, long startTime, long interval){
      super("update", num, startTime, interval);

    }

    @Override
    String action(){
      String cmd = null;

      NodeGroup ng;
      TraceNode node;
      boolean isServer;

      do{
        isServer = false;
        ng = nodeGroups.get(r.nextInt(nodeGroups.size()));
        node = ng.getRandomNode();
        if(node.id == ng.trustedServer.id){
          isServer = true;
        }        
      }while(isServer);// && policy == Policy.SIMPLE);

      int nodeID = node.id;

      while(forkedNodes.containsKey(nodeID)){
        Integer[] children = forkedNodes.get(nodeID);
        int index = r2.nextInt(children.length); // pick a child randomly
        nodeID = children[index]; 
      }

      debug("picked " + nodeID);

      String obj = node.interestSet.get(r.nextInt(node.interestSet.size()));
      cmd = TraceExecutor.UPDATE + " " + nodeID + " " + obj + " "
      + r.nextInt(10);

      for(int i=0; i < numRepeats; i++){
        cmd += "\n"+cmd;
      }

      if(policy == Policy.SIMPLE){
        // tell server to receive data from the updating node
        cmd += "\n" + TraceExecutor.STARTCONNECTION + " " + ng.trustedServer.id
        + " " + node.id;
      }

      return cmd;
    }
  }
  
  class TreeSyncOperation extends SyncOperation {

    public TreeSyncOperation(long num, long startTime,
        long interval){
      super(num, startTime, interval);
    }

    @Override
    String action(){
      String cmd;
      NodeGroup g = nodeGroups.get(r.nextInt(nodeGroups.size()));
      TraceNode src = g.getRandomNode();
      assert policy == Policy.FORKJOIN;
      int srcID = src.id;
      while(forkedNodes.containsKey(srcID)){
        Integer[] children = forkedNodes.get(srcID);
        int index = r2.nextInt(children.length); // pick a child randomly
        srcID = children[index]; 
      }
      TraceNode dst = null;
      assert src.parents.size() + src.children.size() > 0;
      
      if(src.parents.isEmpty()){
        dst = src.children.get(r.nextInt(src.children.size()));
      } else if (src.children.isEmpty()) {
        dst = src.parents.get(r.nextInt(src.parents.size()));
      } else {
        if(r.nextBoolean()){
          dst = src.children.get(r.nextInt(src.children.size()));
        } else {
          dst = src.parents.get(r.nextInt(src.parents.size()));
        }
      }
      assert dst != null;
      
      int dstID = dst.id;

      while(forkedNodes.containsKey(dstID)){
        Integer[] children = forkedNodes.get(dstID);
        int index = r2.nextInt(children.length); // pick a child randomly
        dstID = children[index]; 
      }
      
      cmd = TraceExecutor.STARTCONNECTION + " " + srcID + " " + dstID;
      
      return cmd;
    }
    
  }

  class SyncOperation extends Operation{
    int l;

    public SyncOperation(long num, long startTime, long interval){
      super("sync", num, startTime, interval);
      l = (int)(num/10);
    }

    @Override
    String action(){
      String cmd;
      NodeGroup g = nodeGroups.get(r.nextInt(nodeGroups.size()));
      TraceNode src = g.getRandomNode();

      while(policy == Policy.SIMPLE && src.id == g.trustedServer.id){
        src = g.getRandomNode();
      }

      int srcID = src.id;

      while(forkedNodes.containsKey(srcID)){
        Integer[] children = forkedNodes.get(srcID);
        int index = r2.nextInt(children.length); // pick a child randomly
        srcID = children[index]; 
      }

      NodeGroup dstNodeGroup = g.getSyncTarget();
      TraceNode dst = dstNodeGroup.getRandomNode();

      while(src.id == dst.id || 
          (policy==Policy.SIMPLE && dst.id == dstNodeGroup.trustedServer.id)
          || (policy==Policy.FORKJOIN && forkedNodes.containsKey(src.id)
              && forkedNodes.containsKey(dst.id))){
        if(src.id != dst.id && policy == Policy.FORKJOIN){
          Random tmpR = r;
          r = r2;
          dst = dstNodeGroup.getRandomNode();
          r = tmpR;
        } else {
          dst = dstNodeGroup.getRandomNode();
        }
      }            

      int dstID = dst.id;

      while(forkedNodes.containsKey(dstID)){
        Integer[] children = forkedNodes.get(dstID);
        int index = r2.nextInt(children.length); // pick a child randomly
        dstID = children[index]; 
      }
      
      String optionalCmd = "";
      HashSet<Integer> leafBranches = new HashSet<Integer>();
      if(srcID != src.id){
        getLeaves(src.id, leafBranches);
        Iterator<Integer> iter = leafBranches.iterator();
        
        while(iter.hasNext()){
          int id = iter.next();
          if(id != srcID){
            optionalCmd += "\n"+TraceExecutor.OPTIONALSYNC + " " + id + " " + dstID;
          }
        }

      }      

      cmd = TraceExecutor.STARTCONNECTION + " " + srcID + " " + dstID;
      cmd += optionalCmd;
      

      if( nodeGroups.size() >= 2 && policy == Policy.NONSIMPLE && num % l == 0){
        cmd += "\n"+ TraceExecutor.STARTCONNECTION + " "
        + nodeGroups.get(0).trustedServer.id + " "
        + nodeGroups.get(1).trustedServer.id;
        cmd += "\n"+ TraceExecutor.STARTCONNECTION + " "
        + nodeGroups.get(1).trustedServer.id + " "
        + nodeGroups.get(0).trustedServer.id;
      }

      return cmd;
    }
    
    private void getLeaves(int pId, HashSet<Integer> set){
      assert forkedNodes.containsKey(pId);
      Integer[] children = forkedNodes.get(pId);
      for(int cId : children){
        if(forkedNodes.containsKey(cId)){
          getLeaves(cId, set);
        } else {
          set.add(cId);
        }
      }
      
    }

  }

  class GCOperation extends Operation{
    public GCOperation(long num, long startTime, long interval){
      super("gc", num, startTime, interval);
    }
    @Override
    String action(){
      String cmd;
      NodeGroup g = nodeGroups.get(r2.nextInt(nodeGroups.size()));
      TraceNode node = g.nodeList.get(0);
      assert !forkedNodes.containsKey(node.id);
      cmd = TraceExecutor.GARBAGECOLLECT + " " + node.id;
      return cmd;
    }
    
  }
  
  class ForkOperation extends Operation{
    public ForkOperation(long num, long startTime, long interval){
      super("fork",num, startTime, interval);
    }

    @Override
    String action(){
      String cmd;
      
      // Don't use the Random r for forking so that we can get almost identical traces
      Random tempRandom = r;
      r = r2;
      
      NodeGroup g = nodeGroups.get(r.nextInt(nodeGroups.size()));
      TraceNode src = //g.getRandomNode();
                      g.nodeList.get(g.nodeList.size()-1);
      int srcID = src.id;
      while(forkedNodes.containsKey(srcID)){
        Integer[] children = forkedNodes.get(srcID);
        int index = r.nextInt(children.length); // pick a child randomly
        srcID = children[index]; 
      }

      int numChildren = 2; // 2 by default

      String obj = src.interestSet.get(r.nextInt(src.interestSet.size()));
      cmd = TraceExecutor.UPDATE + " " + srcID + " " + obj + " " + r.nextInt(10) +"\n";
      
      Integer[] newChildren = new Integer[numChildren];
      cmd += TraceExecutor.FORK + " " + srcID + " " + numChildren;
      for(int i=0; i < numChildren; i++){
        newChildren[i] = numTotalNodes++; 
        cmd += " " + newChildren[i];
      }

      forkedNodes.put(srcID, newChildren);
      
      r = tempRandom;
      return cmd;
    }

  }

  void readConfig(File configFile) throws IOException{

    BufferedReader reader = new BufferedReader(new FileReader(configFile));
    int numObjGroup = -1;
    int numNodeGroup = -1;

    for(;;){
      String str = reader.readLine();

      if(str == null){
        debug("EOF from config file");
        return;
      }

      if(str.startsWith("#")){
        outCmd("# " + str);
        continue;
      }

      outCmd("# " + str);


      String[] line = str.split("\\s+");
      if(line.length <= 0) continue;

      if(line[0].equalsIgnoreCase("Policy")){
        if(line[1].equalsIgnoreCase("simple"))
          policy = Policy.SIMPLE;
        else if(line[1].equalsIgnoreCase("forkjoin"))
          policy = Policy.FORKJOIN;
        else
          policy = Policy.NONSIMPLE;
        debug("Setting Policy to " + policy);
      }


      if(line[0].equalsIgnoreCase("singleServer")){

        if ( Integer.parseInt(line[1]) == 1 ) {
          useGlobalServer = true;
          debug("Use global server");
        } else {
          useGlobalServer = false;
          debug("Do not use global server");
        }

      }
      
      if(line[0].equalsIgnoreCase("treeTopology")){
        if ( Integer.parseInt(line[1]) == 1 ) {
          useTreeTopology = true;
          debug("Use Tree topology");
        } else {
          useTreeTopology = false;
          debug("Do not use tree topology");
        }
      }
      
      if(line[0].equalsIgnoreCase("tree.branchFactor")){
        branchFactor = Integer.parseInt(line[1]);
      }
      if(line[0].equalsIgnoreCase("tree.numParents")){
        numParentsPerNode = Integer.parseInt(line[1]);
      }
      if(line[0].equalsIgnoreCase("tree.mirror")){
        if ( Integer.parseInt(line[1]) == 1 ) {
          mirrorImage = true;
          debug("Use Tree topology");
        } else {
          mirrorImage = false;
          debug("Do not use tree topology");
        }
      }

      if(line[0].equalsIgnoreCase("objectGroup")){
        numObjGroup = line.length -1;
        for(int i=0; i < numObjGroup; i++){
          int objGrpId = i;
          int numObjs = Integer.parseInt(line[i+1]);
          objGroups.add(new ObjGroup(objGrpId, numObjs));
        }        
      }


      if(line[0].equalsIgnoreCase("nodeGroup")){
        numNodeGroup = line.length -1;
        for(int i=0; i < numNodeGroup; i++){
          int nodeGrpId = i;
          int numNodes = Integer.parseInt(line[i+1]);
          nodeGroups.add(new NodeGroup(nodeGrpId, numNodes));
        }
        if(mirrorImage){
          numTotalNodes *= 2;
        }
        System.out.println("# Number of nodes : "+ numTotalNodes);
      }
      

      debug("NumNodeGroup " + numNodeGroup);

      if(line[0].equalsIgnoreCase("nodeGrp2objGrp")){
        int idx = 1;
        for(int i=0; i<numNodeGroup; i++){
          Double[] isPortion = new Double[numObjGroup];
          double sum = 0;
          for(int j=0; j < numObjGroup; j++){
            isPortion[j] = Double.parseDouble(line[idx]);
            idx++;
            sum += isPortion[j];
          }
          if(sum != 1){
            System.err.println("Wrong configuration : " + str);
            System.err.println("Sum of ISportion for each node group should be 1.0");
            System.exit(-1);
          }
          ISPortion.put(i, isPortion);
        }
      }

      if(line[0].equalsIgnoreCase("syncTo")){
        int idx = 1;
        for(int i=0; i<numNodeGroup; i++){
          Double[] syncPercen = new Double[numNodeGroup];
          double sum = 0;
          for(int j=0; j < numNodeGroup; j++){
            debug("idx : " + idx + ", " + line[idx]);
            syncPercen[j] = Double.parseDouble(line[idx]);
            idx++;
            sum += syncPercen[j];
          }
          if(sum != 1){
            System.err.println("Wrong configuration : " + str);
            System.err.println("Sum of syncTo for each node group should be 1.0");
            System.exit(-1);
          }
          syncTo.put(i, syncPercen);
        }                
      }

      if(line[0].equalsIgnoreCase("numRepeats")){
        numRepeats = Integer.parseInt(line[1]);
      }


      if(line[0].equalsIgnoreCase("ISPortion")){
        System.err.println("Old configuration file..");
        System.exit(-1);
      }

      if(line[0].equalsIgnoreCase("IntraGroupPortion")){
        System.err.println("Old configuration file..");
        System.exit(-1);
      }

      if(line[0].equalsIgnoreCase("seed")){
        randomSeed = Long.parseLong(line[1]);
        debug("Setting randomSeed to " + randomSeed);
      }

      if(line[0].equalsIgnoreCase("update")){
        debug("Adding update op");
        ops.add(new UpdateOperation(Long.parseLong(line[1]), Long
            .parseLong(line[2]), Long.parseLong(line[3])));
      }

      if(line[0].equalsIgnoreCase("sync")){
        debug("Adding sync op");
        if(useTreeTopology){
          ops.add(new TreeSyncOperation(Long.parseLong(line[1]), Long
              .parseLong(line[2]), Long.parseLong(line[3])));
        } else {
          ops.add(new SyncOperation(Long.parseLong(line[1]), Long
              .parseLong(line[2]), Long.parseLong(line[3])));
        }
      }

      if(line[0].equalsIgnoreCase("fork")){
        debug("Adding fork op");
        ops.add(new ForkOperation(Long.parseLong(line[1]), Long
            .parseLong(line[2]), Long.parseLong(line[3])));
      }
      
      if(line[0].equalsIgnoreCase("gc")){
        debug("Adding gc op");
        ops.add(new GCOperation(Long.parseLong(line[1]), Long
            .parseLong(line[2]), Long.parseLong(line[3])));
      }
      

    }

  }

  private void outCmd(String cmd){
    System.out.println(cmd);
  }

  public void generateTrace(){
    String cmd;

    // Initialize each node
    if(policy != Policy.FORKJOIN){
      for(NodeGroup ng : nodeGroups){
        for(TraceNode node : ng.nodeList){
          debug("");
          cmd = TraceExecutor.SETTRUSTEDNODE + " " + node.id + " "
          + ng.trustedServer.id;
          outCmd(cmd);

          cmd = TraceExecutor.SETLIVENESSFILTER + " " + node.id + " ";
          if(policy == Policy.SIMPLE){
            cmd += MapManager.LIVENESS_TRACE_FILTER;
          } else {
            cmd += MapManager.LIVENESS_NORMAL_FILTER;
          }
          outCmd(cmd);

          cmd = TraceExecutor.STARTNODE + " " + node.id;
          outCmd(cmd);

          cmd = TraceExecutor.SETINTERESTSET + " " + node.id + " " + node.getIS();
          outCmd(cmd);
        }
      }
    }

    while(ops.size() > 0){
      Operation op = ops.first();
      ops.remove(op);
      cmd = op.perform();
      if(cmd != null){
        outCmd(cmd);
        ops.add(op);
      }
    }

  }

  public static void main(String[] args){

    TraceGenerator2 gen = new TraceGenerator2();

    try{
      gen.readConfig(new File(args[0]));
    }catch(IOException e){
      System.err.println(e.getLocalizedMessage());
      System.exit(-1);
    }

    gen.init();
    debug("init done");
    gen.generateTrace();

  }

}

/*
 * Example of config file
 * 
 * 
 * # Policy (forkjoin, simple or non-simple) 
 * # all other value will be considered non-simple
 * Policy simple
 * 
 * 
 * # use single global server (single server regardless # of groups)
 * # 1 : yes , otherwise server per each group (default : no)
 * singleServer 1
 * 
 * # use tree topology
 * # 1 : yes
 * treeTopology 0
 * 
 * # branchFactor (assuming single parent) 
 * # default 2
 * tree.branchFactor 3
 * 
 * # number of parents per node (for fat tree)
 * # default 1
 * tree.numParents 1
 * 
 * # create a mirror image of the created tree
 * # default 0
 * tree.mirror 1
 * 
 * # Object Group info 
 * # the following example indicates three groups
 * # first group consists of 100 objs and the other two consist of
 * # 50 objs each 
 * objectGroup 100 50 50
 * 
 * # Node Group info ( # of nodes for each node group )
 * nodeGroup 10 10
 * 
 * # specifying to which obj group and with what proportion 
 * # each node group is interested in
 * nodeGrp2objGrp 0.5 0 0.5 1.0 0 0 
 *  
 * # proportion with which a node group will sync to other node group
 * (first node group will sync to itself with 50% of chance and to second group with 50%,
 * and second node group will sync to first one with 20% and to itself with 80%)  
 * syncTo 0.5 0.5 0.2 0.8
 * 
 * # how many time each update will repeat
 * # n means n extra identical updates for each update
 * numRepeats 3
 * 
 * # random seed (if not given, use system time) 
 * seed 7
 * 
 * #opname      N_Ops   startTime       interval 
 * update       10         0               3 
 * sync         5          200             15
 * fork         2          5               5
 * gc           5          2               2          
 */
