package code.security.holesync;

import java.util.Hashtable;
import java.util.TreeSet;
import java.util.Collection;
import java.util.LinkedList;
import java.util.SortedSet;
import java.util.Iterator;
import java.io.Serializable;

import code.CounterVV;
import code.NoSuchEntryException;
import code.NodeId;
import code.PreciseInv;
import code.ImpreciseInv;
import code.AcceptVV;
import code.VVIterator;
import code.AcceptStamp;
import code.VV;
import code.security.SangminConfig;

import code.security.holesync.filter.*;
import code.security.ahs.AHSEntry;

/**
 * knowledge relevant to a filter
 * Contains a list of holes and a cvv
 * holes can only be removed when completely overlapping collection of updates (precise or imprecise) is received
 * So, holes can't be added for ranges smaller than cvv
 * @author princem
 *
 */
public class FilterKnowledge implements Serializable{

  Filter myFilter;
  
  /**
   * This represents the missing information about updates that are relevant to this filter. 
   * If the holes are empty, then the filter is assumed to be current.
   */
  Hashtable<NodeId, TreeSet<Range>> holes;
  CounterVV cvv;
  
  public FilterKnowledge(Filter f){
    this.myFilter = f;
    holes = new Hashtable<NodeId, TreeSet<Range>>();
    cvv = new CounterVV(); 
  }
  
  public FilterKnowledge(VV vv, Filter f){
    this.myFilter = f;
    holes = new Hashtable<NodeId, TreeSet<Range>>();
    cvv = new CounterVV(vv);
  }
  /**
   * function to replace range r with (possibly empty) replacement
   * replacement occurs only when perfect match occurs
   * @param nId
   * @param r
   * @param replacement
   */
  private void replaceRange(NodeId nId, Range r, Collection<Range> replacement){
    TreeSet<Range> nodeHoles = holes.get(nId);
    if(nodeHoles != null && nodeHoles.size() > 0){
       if(nodeHoles.first().compareTo(r) > 0 || nodeHoles.last().compareTo(r) < 0){
        // if inval is older than the oldest hole
      }else if(nodeHoles.contains(r)){
        nodeHoles.remove(r);
        if(replacement!=null){
          nodeHoles.addAll(replacement);
        }
      }
    }
  }
  
  public final Filter getFilter(){
    return myFilter; 
  }
  
  public final Hashtable<NodeId, TreeSet<Range>> getHoles(){
    return holes;
  }
  
  public final AcceptVV getCVV(){
    return new AcceptVV(cvv);
  }
  
  public void apply(PreciseInv pi){
    if(!cvv.includes(pi.getAcceptStamp())){
      cvv.advanceTimestamps(pi.getAcceptStamp());
    }else{
      Range invalRange = new Range(pi.getStart(), pi.getEnd());
      replaceRange(pi.getNodeId(), invalRange, null);
    }
  }
  
  public void apply(ImpreciseInv ii){
    assert ii.getEndVV().size() == 1: ii;
    VVIterator vvi = ii.getEndVV().getIterator();
    NodeId nid = vvi.getNext();
    long endTS = ii.getEndVV().getStampByIteratorToken(nid);
    long startTS = ii.getStartVV().getStampByIteratorToken(nid);
    if(!cvv.includes(ii.getEndVV())){
      if(!cvv.includes(ii.getStartVV())){
        cvv.advanceTimestamps(ii.getEndVV());
        Range invalRange = new Range(startTS, endTS);

      //assert that the ii spans only one node
        TreeSet<Range> nodeHoles;

        if(holes.containsKey(nid)){
          nodeHoles = holes.get(nid);
          // find the set of holes that are equal to or larger than the new hole that we are adding
          SortedSet<Range> holes = nodeHoles.tailSet(new Range(invalRange.getStart(), invalRange.getStart()));
          nodeHoles.removeAll(holes);
        }else{
          nodeHoles = new TreeSet<Range>();
          holes.put(nid, nodeHoles);
        }
        
        if(myFilter.isPresent(ii)){
          nodeHoles.add(invalRange);
        }
      }else{
        assert false;
        System.out.println("currently knowledge class doesn't expect partially overlapping imprecise invals");
      }
    }
  }

  public void apply(NodeId nodeId, AHSEntry ahsEntry){
    if(!cvv.includes(new AcceptStamp(ahsEntry.getEndTS(), nodeId))){
      cvv.advanceTimestamps(new AcceptStamp(ahsEntry.getEndTS(), nodeId));
      //assert that the ii spans only one node
      Range invalRange = null;
      invalRange = new Range(ahsEntry.getStartTS(), ahsEntry.getEndTS());
   // remove any holes that are also covered by the new hole
      TreeSet<Range> nodeHoles;

      if(holes.containsKey(nodeId)){
        nodeHoles = holes.get(nodeId);
        // find the set of holes that are equal to or larger than the new hole that we are adding
        Collection<Range> holes = new LinkedList<Range>();
        holes.addAll(nodeHoles.tailSet(new Range(invalRange.getStart(), invalRange.getStart())));
        nodeHoles.removeAll(holes);
      }else{
        nodeHoles = new TreeSet<Range>();
        holes.put(nodeId, nodeHoles);
      }
      if(myFilter.isPresent(nodeId, ahsEntry)){
        nodeHoles.add(invalRange);
      }
    }
  }
  
  private void applyHelper(Hole h, Collection<Range> hf){
    //expect that hf completely covers the range r 
    if(holes.containsKey(h.getNodeId()) && holes.get(h.getNodeId()).contains(h.getRange())){
      holes.get(h.getNodeId()).remove(h.getRange());
      holes.get(h.getNodeId()).addAll(hf);
    }
  }
  
  public void apply(Hole h, Collection hf){
    // HoleFiller is a contiguous collection of precise and imprecise AHS tuples
    if(hf.size() > 0){
      long start = Long.MAX_VALUE;
      long end = Long.MIN_VALUE;
      LinkedList<Range> list = new LinkedList<Range>();
      for(Object o: hf){
        if(o instanceof PreciseInv){
          PreciseInv pi = (PreciseInv)o;
          assert pi.getNodeId().equals(h.getNodeId());
        }else{
          assert o instanceof AHSEntry:o;
          AHSEntry ahsEntry = (AHSEntry)o;
          if(myFilter.isPresent(h.getNodeId(), ahsEntry)){
            list.add(new Range(ahsEntry.getStartTS(), ahsEntry.getEndTS()));
          }
        }
      }
      
      this.applyHelper(h, list);
    }
    
  }
  
  
  
  public Object getKnowledgeFragmentContainingRange(NodeId nid, Range r){
    if(!cvv.includes(new AcceptStamp(r.getStart(), nid))){
      return new AcceptVV(cvv);
    }else{ 
      if(holes.containsKey(nid)){
        // search for the relevant hole
        for(Range range: holes.get(nid)){
          if(range.contains(r)){
            return new Hole(nid, range);
          }
        }
      }
      return null;
    }
  }
  
  public void dropNode(NodeId nId){
    cvv.dropNode(nId);
    holes.remove(nId);
  }
  
  public FilterKnowledge clone(){
    FilterKnowledge fk = new FilterKnowledge(cvv, myFilter);
    for(NodeId n: holes.keySet()){
      TreeSet<Range> ts = holes.get(n);
      fk.holes.put(n, (TreeSet<Range>)ts.clone());
    }
    return fk;
    
  }
  
  public boolean equals(Object o){
    if(!(o instanceof FilterKnowledge)){
      return false;
    }else{
      FilterKnowledge f = (FilterKnowledge)o;
      return this.includes(f) && f.includes(this);
    }
  }
  
  public String toString(){
    String str =  myFilter.toString() + ":" + cvv.toString() ;
    for(NodeId n: holes.keySet()){
      str += "\n" + n + ":";
      for(Range r: holes.get(n)){
        str += " " + r + " " ;
      }
    }
    return str;
  }
  
  public AcceptVV getLPVV(){
    CounterVV ret = new CounterVV(cvv);
    for(NodeId n: holes.keySet()){
      if(!holes.get(n).isEmpty()){
        ret.setStampByNodeId(n, holes.get(n).first().getStart());
      }
    }
    return new AcceptVV(ret);
  }
  
  public boolean includes(FilterKnowledge f){
    if(!cvv.includes(f.cvv)){
      return false;
    }else{
      for(NodeId nId: holes.keySet()){
        TreeSet<Range> ts = holes.get(nId);
        if(ts.size() != 0){
          if(!f.holes.containsKey(nId)){
            return false;
          }
          // find out the sortedSet preceding the CVV of f
          try{
            long endTS = f.cvv.getStampByServer(nId);
            Range endRange = new Range(endTS+1, endTS+1);
            SortedSet<Range> mySet = ts.headSet(endRange);
            SortedSet<Range> otherSet = f.holes.get(nId).headSet(endRange);
            
            Range myRange = null, otherRange = null; 
            
            Iterator<Range> mySetIterator = mySet.iterator();
            Iterator<Range> otherSetIterator = otherSet.iterator();
            if(!mySetIterator.hasNext()){
              continue;
            }else if(!otherSetIterator.hasNext()){
              return false;
            }else{
              otherRange = otherSetIterator.next();
            }
            
            while(mySetIterator.hasNext()){
              myRange = mySetIterator.next();
              
              while(!otherRange.contains(myRange)){
                if((otherRange.compareTo(myRange) > 0) || !otherSetIterator.hasNext()){
                  return false;
                }else{
                  otherRange = otherSetIterator.next();
                }
              }
            }
            
          }catch(NoSuchEntryException e){
            // TODO Auto-generated catch block
            return false;
          }
          
        }
      }
    }
    return true;
  }
  
  public int size(){
    int s = (SangminConfig.compressNodeId?1:4); // for holes.size
    for(NodeId n: holes.keySet()){
      TreeSet<Range> ts = holes.get(n);
      if(ts.size() > 0){
        s += (SangminConfig.compressNodeId?1:8); // nodeId
        for(Range r: ts){
          s+=(((r.getEnd() - r.getStart())<127)?9:16); // for each range: start, power of 2 for size
        }
      }
      
    }
    return s;
  }
}
