package code.security.holesync;

import java.util.*;

import code.CounterVV;
import code.ImpreciseInv;
import code.NodeId;
import code.PreciseInv;
import code.security.SecurePreciseInv;
import code.security.ahs.AHSEntry;
import code.security.holesync.filter.*;
import code.AcceptStamp;
import code.AcceptVV;

/**
 * A node's knowledge is the collection of FilterKnowledge for each filter
 * @author princem
 *
 */
public class Knowledge{
  CounterVV currentVV;
  
  Hashtable<Filter, FilterKnowledge> filterKnowledge;
  
  public Knowledge(){
    currentVV = new CounterVV();
    filterKnowledge = new Hashtable<Filter, FilterKnowledge>();
  }
  
  public void addFilter(Filter f){
    if(!filterKnowledge.containsKey(f)){
      filterKnowledge.put(f, new FilterKnowledge(f));
    }
  }
  
  public void setFilter(Collection<Filter> f){
    filterKnowledge.clear();
    for(Filter filter: f){
      filterKnowledge.put(filter, new FilterKnowledge(filter));
    }
  }
  
  public Knowledge(Collection<Filter> filters){
    currentVV = new CounterVV();
    filterKnowledge = new Hashtable<Filter, FilterKnowledge>();
    for(Filter f: filters){
      filterKnowledge.put(f, new FilterKnowledge(f));
    }
  }
  
  public AcceptVV getCurrentVV(){
    return new AcceptVV(currentVV);
  }
  
  public void apply(Hole h, Collection hf){
    for(Filter f: filterKnowledge.keySet()){
      filterKnowledge.get(f).apply(h, hf);
    }
  }
  
  public void apply(NodeId nodeId, AHSEntry ahsEntry){
    currentVV.advanceTimestamps(new AcceptStamp(ahsEntry.getEndTS(), nodeId));
    for(Filter f: filterKnowledge.keySet()){
      filterKnowledge.get(f).apply(nodeId, ahsEntry);
    }
  }
  
  public void apply(PreciseInv pi){
    currentVV.advanceTimestamps(pi.getAcceptStamp());
    for(Filter f: filterKnowledge.keySet()){
      filterKnowledge.get(f).apply(pi);
    }
  }
  
  public void apply(ImpreciseInv ii){
    currentVV.advanceTimestamps(ii.getEndVV());
    for(Filter f: filterKnowledge.keySet()){
      filterKnowledge.get(f).apply(ii);
    }
  }
  
  public Knowledge clone(){
    Knowledge k = new Knowledge();
    for(Filter f: filterKnowledge.keySet()){
      k.filterKnowledge.put(f, filterKnowledge.get(f).clone());
    }
    k.currentVV = new CounterVV(currentVV);
    return k;
  }
  
  public FilterKnowledge getFilterKnowledge(Filter f){
    if(filterKnowledge.containsKey(f)){
      return filterKnowledge.get(f);
    }else{
      return null;
    }
  }
  
  public final Collection<Filter> getFilters(){
    return filterKnowledge.keySet();
  }
  
  public void dropNode(NodeId nId){
    currentVV.dropNode(nId);
    for(Filter f : filterKnowledge.keySet()){
      filterKnowledge.get(f).dropNode(nId);
    }
  }
  
  public String toString(){
    String ret = "CVV: " + currentVV + "\n ";
    for(Filter f:filterKnowledge.keySet()){
      ret += "\n" + filterKnowledge.get(f);
    }
    return ret;
  }
  
  
}
