package code.security.holesync.unit;

import code.security.DataHash;
import code.security.holesync.*;
import code.security.holesync.filter.*;
import code.CounterVV;
import code.HierInvalTarget;
import code.ImpreciseInv;
import code.ObjId;
import code.ObjInvalTarget;
import code.PreciseInv;
import code.SubscriptionSet;
import code.AcceptStamp;
import code.AcceptVV;
import code.NodeId;
import code.security.ahs.AHSEntry;

import java.util.TreeSet;
import java.util.Hashtable;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;
import junit.textui.TestRunner;

public class KnowledgeUnit extends TestCase{
  //public static final String TEST_ALL_TEST_TYPE = "UNIT";

  //create SecureCore as URANode constructor did
  boolean filterOn = true;
  boolean cleanDb = true;  // start from empty

  byte[] buffer, buffer2; 
  
  
  /*
   * Fixtures are run before and after each test case
   * to set up environment in which tests must run.
   */
  protected void setUp() throws Exception{
    super.setUp();
    buffer = new byte[]{0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19};
    buffer2 = new byte[]{10,11,12,13,14,15,16,17,18,19,10,11,12,13,14,15,16,17,18,19};

    Common.makePractiConfig(0, 2);
  }

  protected void tearDown() throws Exception{
    /* 
     * TBD: Insert other fixture cleanup code here
     */
    super.tearDown();

  }

  public void testIncludes(){
    System.out.println("running testIncludes");
    Filter f = new SubscriptionSetFilter(SubscriptionSet.makeSubscriptionSet("/*"));
    NodeId nId = new NodeId(0);
    
   
    //------------------------------------------------------------------
    // case 1: when CVV1 is not included in CVV2
    
    CounterVV cvv1 = new CounterVV();
    cvv1.setStampByNodeId(nId, 5);
    CounterVV cvv2 = new CounterVV();
    cvv2.setStampByNodeId(nId, 4);
    FilterKnowledge fk1 = new FilterKnowledge(cvv1, f);
    FilterKnowledge fk2 = new FilterKnowledge(cvv2, f);
    
    assert fk1.includes(fk2);
    assert !fk2.includes(fk1);
    assert !fk1.equals(fk2);
    
    //------------------------------------------------------------------
    // case 2: when CVV are included and clear inclusion in terms of holes

    Hashtable<NodeId, TreeSet<Range>> holes1 = fk1.getHoles();
    Hashtable<NodeId, TreeSet<Range>> holes2 = fk2.getHoles();
    
    holes1.put(nId, new TreeSet<Range>());
    holes2.put(nId, new TreeSet<Range>());
    
    holes1.get(nId).add(new Range (1,1));
    
    assert !fk1.includes(fk2);
    assert !fk2.includes(fk1);
    assert !fk1.equals(fk2);
    
    holes2.get(nId).add(new Range (1,1));
    
    assert fk1.includes(fk2);
    assert !fk2.includes(fk1);
    assert !fk1.equals(fk2);
    
    //------------------------------------------------------------------
    // case 3: when CVV included but not clear included (one range contains the other)
    
    holes1.put(nId, new TreeSet<Range>());
    holes2.put(nId, new TreeSet<Range>());
    
    holes1.get(nId).add(new Range (1,1));
    
    assert !fk1.includes(fk2);
    assert !fk2.includes(fk1);
    assert !fk1.equals(fk2);
    
    holes2.get(nId).add(new Range (1,2));
    
    assert fk1.includes(fk2);
    assert !fk2.includes(fk1);
    assert !fk1.equals(fk2);
    
    //------------------------------------------------------------------
    // case 4: when CVV included but overlap of range
    
    holes1.put(nId, new TreeSet<Range>());
    holes2.put(nId, new TreeSet<Range>());
    
    holes1.get(nId).add(new Range (1,1));
    
    assert !fk1.includes(fk2);
    assert !fk2.includes(fk1);
    assert !fk1.equals(fk2);
    
    holes2.get(nId).add(new Range (2,3));
    
    assert !fk1.includes(fk2);
    assert !fk2.includes(fk1);
    assert !fk1.equals(fk2);
    
    holes2.get(nId).add(new Range (1,1));
    
    assert fk1.includes(fk2);
    assert !fk2.includes(fk1);
    assert !fk1.equals(fk2);
    
    holes1.get(nId).add(new Range (2,2));
    
    assert fk1.includes(fk2);

    holes1.get(nId).add(new Range (3,4));
    
    assert !fk1.includes(fk2);
    
  }
  
  
  public void testKnowledge()
  throws Exception{
    System.out.println("running testKnowledge");

    
    Knowledge k = new Knowledge();
    Filter f = new SubscriptionSetFilter(SubscriptionSet.makeSubscriptionSet("/A/*"));
    k.addFilter(f);
    
    FilterKnowledge fk = k.getFilterKnowledge(f);
    assert fk.getCVV().equalsIgnoreNegatives(AcceptVV.makeVVAllNegatives());
    assert fk.getHoles().isEmpty();
    
    NodeId nId = new NodeId(0);
    
    ObjId inoid = new ObjId("/A/1");

    ObjId outoid = new ObjId("/B/1");
    
    
    //------------------------------------------------------------------
    // (1) test if knowledge advances when precise and imprecise invals are applied and appropriate holes are created
    //------------------------------------------------------------------

    //------------------------------------------------------------------
    //apply precise
    PreciseInv pi = new PreciseInv(new ObjInvalTarget(inoid, 
        0, 
        1),
        new AcceptStamp(0, nId));
    
    k.apply(pi);
    
    assert fk.getCVV().includes(pi.getAcceptStamp());
    assert fk.getHoles().isEmpty();
    
    //------------------------------------------------------------------
    //apply imprecise not covering any relevant objects
    
    AcceptStamp[] as1 = new AcceptStamp[1];
    AcceptStamp[] as2 = new AcceptStamp[1];
    
    as1[0] = new AcceptStamp( 1, nId);
    as2[0] = new AcceptStamp( 3, nId);
    ImpreciseInv ii = new ImpreciseInv(HierInvalTarget.makeHierInvalTarget("/B/*"),
        new AcceptVV(as1),
         new AcceptVV(as2));
    
    k.apply(ii);
    
    Hashtable<NodeId, TreeSet<Range>> holes  = fk.getHoles();
    assert fk.getCVV().includes(ii.getEndVV());
    for(NodeId n: holes.keySet()){
      assert holes.get(n).isEmpty();
    }
    
    //------------------------------------------------------------------
    // apply precise not covering any relevant objects
    
    pi = new PreciseInv(new ObjInvalTarget(outoid, 
        0, 
        1),
        new AcceptStamp(4, nId));
    
    k.apply(pi);
    
    assert fk.getCVV().includes(pi.getAcceptStamp());
    holes  = fk.getHoles();
    for(NodeId n: holes.keySet()){
      assert holes.get(n).isEmpty();
    }
    
    //------------------------------------------------------------------
    // apply imprecise covering relevant objects
    
    
    as1[0] = new AcceptStamp( 5, nId);
    as2[0] = new AcceptStamp( 7, nId);
    ii = new ImpreciseInv(HierInvalTarget.makeHierInvalTarget("/A/*"),
        new AcceptVV(as1),
         new AcceptVV(as2));
    
    k.apply(ii);
    
    holes = fk.getHoles();
    assert fk.getCVV().includes(ii.getEndVV());
    assert !holes.isEmpty();
    
    Range r = new Range(as1[0].getLocalClock(), as2[0].getLocalClock());
    
    assert holes.containsKey(nId);
    assert holes.get(nId).contains(r);
    
    //------------------------------------------------------------------
    // apply AHSEntry not covering any relevant objects
        
    ObjInvalTarget it = new ObjInvalTarget(outoid, 5, 10);
    AHSEntry ahsEntry = new AHSEntry(2, buffer, 8, 11, it, null, buffer2);

    k.apply(nId, ahsEntry);
    
    holes = fk.getHoles();
    assert fk.getCVV().includes(new AcceptStamp(ahsEntry.getEndTS(), nId));
    assert !holes.isEmpty();
    
    r = new Range(ahsEntry.getStartTS(), ahsEntry.getEndTS());
    
    assert holes.containsKey(nId);
    assert !holes.get(nId).contains(r);
    assert holes.get(nId).size()==1;
    
    //------------------------------------------------------------------
    // apply AHSEntry covering relevant objects
    
    it = new ObjInvalTarget(inoid, 5, 10);
    ahsEntry = new AHSEntry(2, buffer, 12, 16, it, null, buffer2);
    k.apply(nId, ahsEntry);
    
    holes = fk.getHoles();
    assert fk.getCVV().includes(new AcceptStamp(ahsEntry.getEndTS(), nId));
    assert !holes.isEmpty();
    
    r = new Range(ahsEntry.getStartTS(), ahsEntry.getEndTS());
    
    assert holes.containsKey(nId);
    assert holes.get(nId).contains(r);
    assert holes.get(nId).size()==2;
    
    //------------------------------------------------------------------
    // (2) test if holes are correctly filled when precise invals and holefillers covering holes are applied
    //------------------------------------------------------------------

    //------------------------------------------------------------------
    // test that when holes in previously holefree range are applied, no change occurs

    FilterKnowledge oldfk = fk.clone();
    assert oldfk.equals(fk);
    
    it = new ObjInvalTarget(inoid, 5, 10);
    ahsEntry = new AHSEntry(2, buffer, 13, 16, it, null, buffer2);
    k.apply(nId, ahsEntry);
    
    pi = new PreciseInv(new ObjInvalTarget(outoid, 
        0, 
        1),
        new AcceptStamp(3, nId));
    
    k.apply(pi);
    
    HoleFiller hf = new HoleFiller();
    
    ahsEntry = new AHSEntry(1, buffer, 12, 14, it, null, buffer2);

    hf.add(ahsEntry);
    pi = new PreciseInv(new ObjInvalTarget(outoid, 
        0, 
        1),
        new AcceptStamp(15, nId));
    
    hf.add(pi);
    
    fk.apply(new Hole(nId, new Range(12,15)), hf.getInvals());
    
    assert oldfk.equals(fk);
    
    //------------------------------------------------------------------
    // apply AHSEntry covering relevant objects
    
    ahsEntry = new AHSEntry(1, buffer, 16, 16, it, null, buffer2);

    hf.add(ahsEntry);
    
    fk.apply(new Hole(nId, new Range(12,16)), hf.getInvals());
    
    assert !oldfk.includes(fk):"oldfk " + oldfk + "\n fk" + fk;
    assert fk.includes(oldfk);
    assert fk.getHoles().get(nId).size()==3;
    assert !fk.getHoles().get(nId).contains(new Range(12, 16));
    assert fk.getHoles().get(nId).contains(new Range(12, 14));
    assert fk.getHoles().get(nId).contains(new Range(16, 16));
    assert oldfk.getHoles().get(nId).contains(new Range(12, 16));
    assert fk.getCVV().equals(oldfk.getCVV());

    hf = new HoleFiller();
    
    pi = new PreciseInv(new ObjInvalTarget(outoid, 
        0, 
        1),
        new AcceptStamp(5, nId));
    
    hf.add(pi);
    
    pi = new PreciseInv(new ObjInvalTarget(outoid, 
        0, 
        1),
        new AcceptStamp(6, nId));
    
    hf.add(pi);
    
    pi = new PreciseInv(new ObjInvalTarget(outoid, 
        0, 
        1),
        new AcceptStamp(7, nId));
    
    hf.add(pi);
    
    fk.apply(new Hole(nId, new Range(5,7)), hf.getInvals());

    assert fk.getHoles().get(nId).size()==2:fk;
    assert !fk.getHoles().get(nId).contains(new Range(5, 7));

    System.out.println(fk);
    
    //------------------------------------------------------------------
    // (3) test if partially overlappign holes lead to advancement of the CVV and replacement of the hole with a larger hole
    //------------------------------------------------------------------

    ahsEntry = new AHSEntry(1, buffer, 16, 17, it, null, buffer2);
    
    k.apply(nId, ahsEntry);
    assert fk.getHoles().get(nId).size()==2:fk;
    assert !fk.getHoles().get(nId).contains(new Range(16, 16));
    assert fk.getHoles().get(nId).contains(new Range(16, 17));
    assert fk.getCVV().includes(new AcceptStamp(ahsEntry.getEndTS(), nId));

    System.out.println(fk);

  }

  
  /*
   * "new TestSuite(Class c)" constructs a test suite
   * containg every method whose name begins with "test"
   * 
   * TBD: update class name
   */
  public static Test suite(){
    TestSuite suite = new TestSuite(KnowledgeUnit.class);
    return suite;
  }


  /*
   * main() lets us run just this set of unit tests
   * from the comand line (you can also invoke 
   * the testrunner on this class and it will find
   * the suite())
   *
   * TBD: update class name
   */
  public static void main(String s[]) {
    String name = "KnowledgeUnit";
    System.err.print(name + " self test begins...");
    TestRunner tr = new TestRunner();
    tr.doRun(suite());
    System.err.println(name + " self test succeeds");
  }

}

