package code.security.liveness;

import java.io.IOException;

import code.*;
import code.security.*;
import code.security.ahs.*;
import java.util.TimerTask;
import java.util.Timer;

public class TrustedServerCertificateAuthority implements CertificateAuthority{

  NodeId myCA;
  CounterVV currentVV;

  AcceptVV lastCertifiedVV;
  SecurityFilter securityFilter;
  private boolean initialized = false;

  /**
   * delay for temporal batching
   */
  public final static long timerDelay = 10000;

  private boolean isTimerScheduled;
  
  private Timer timer;

  public TrustedServerCertificateAuthority(SecurityFilter sf){
    this.securityFilter = sf;
    initialized = true;
    isTimerScheduled = false;
    timer = new Timer();
    currentVV = new CounterVV();
    lastCertifiedVV = new AcceptVV(currentVV);
  }

  public TrustedServerCertificateAuthority(){
    initialized = true;
    isTimerScheduled = false;
    timer = new Timer();
    currentVV = new CounterVV();
    lastCertifiedVV = new AcceptVV(currentVV);
  }

  public void initialize(SecurityFilter sf){
    securityFilter = sf;
    initialized = true;
    isTimerScheduled = false;
  }
  /**
   * notify the certificate authority that a new update has been received 
   * @param currentVV
   */
  public void notifyUpdateProcessed(SecurePreciseInv pi){
    /**
     * check if the update's accept stamp is greater than currentVV 
     * If not, do nothing
     * If yes, find the largest linear precise chain after currentVV and advance currentVV
     * Also advance currentVV to include the certified currentVV that we receive from our trusted servers
     */
    assert initialized;

    if(currentVV.includes(pi.getEndVV())){
      return;
    }else{
      DependencyVV dvv = pi.getDVV();
      try{
        NodeId sender= pi.getCreator();
        if(!dvv.containsNodeId(sender) || 
            (currentVV.containsNodeId(sender) && 
                dvv.getStampByServer(sender) == currentVV.getStampByServer(sender))){
          // found useful update
          TreeNode tn = securityFilter.getAhsMap().getLeafNodeTS(sender, pi.getEnd());
          System.out.println("tn " + tn + "\n " + pi ); 
          assert tn.getEndTS() == pi.getEnd() && tn.isPrecise() && (tn.getInv() instanceof SecurePreciseInv): tn + " " + pi;

          currentVV.advanceTimestamps(new AcceptStamp(tn.getEndTS(), sender));
          // keep advancing until an imprecise tree node is found
          while(tn.getNext() != null){
            tn = tn.getNext();
            if(!tn.isPrecise()){
              break;
            }
            assert tn.getLevel() == 0 && tn.getInv() instanceof PreciseInv;
            currentVV.advanceTimestamps(new AcceptStamp(tn.getEndTS(), sender));
          }
        }
      }catch(NoSuchEntryException e){
        // TODO Auto-generated catch block
        e.printStackTrace();
        assert false;
      }
    }
  }

  /**
   * see if the currentVV > lastCertifiedVV and create a certificate if it is indeed the case
   */
  synchronized public void createNewCertificate(){
    assert initialized;
    if(!lastCertifiedVV.includes(currentVV) && !isTimerScheduled){
      isTimerScheduled = true;
      timer.schedule(new TimerTask(){
        public void run(){
          timerHandler();
        }
      }, timerDelay);
    }
  }
  
  synchronized private void cancelTimer(){
    isTimerScheduled = false;
  }

  private void timerHandler(){
    System.out.println("Timer fired");
    try{
      lastCertifiedVV = new AcceptVV(currentVV);
      ((SecureCore)securityFilter.core).certify(lastCertifiedVV);
      cancelTimer();
    }catch(IOException e){
      // TODO Auto-generated catch block
      e.printStackTrace();
      assert false;
    }
  }
  

  public void processLivenessCertificate(LivenessCertificate lc){
    assert initialized;
    currentVV.advanceTimestamps(lc.getCertifiedVV());

  }

  public AcceptVV lastCertifiedVV(){
    assert initialized;
    return lastCertifiedVV;
  }


}
