package code.security.liveness;

import code.branchDetecting.BranchID;
import code.security.*;
import code.*;
import code.security.ahs.*;
import code.security.holesync.filter.Filter.Attribute;

import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.lang.reflect.AccessibleObject;
import java.lang.reflect.Field;
import java.util.LinkedList;
import java.util.Timer;
import java.util.List;
/**
 * This class implements the trusted liveness server policy of accepting updates only if they are precise or if they are 
 * covered by a certificate signed by the liveness server
 * @author princem
 *
 */
public class TrustedServerLivenessFilter implements LivenessFilter{
  NodeId trustedServer;
  CounterVV largestVV; 
  SplitManager splitManager;
  CertificateAuthority certificateAuthority;
  SecurityFilter securityFilter;
  private boolean initialized =false;



  public TrustedServerLivenessFilter(NodeId trusted, SecurityFilter securityFilter){
    this.trustedServer = trusted;
    largestVV = new CounterVV();
    this.securityFilter = securityFilter;
    this.splitManager = securityFilter.splitManager;
    certificateAuthority = new TrustedServerCertificateAuthority(securityFilter);
    initialized = true;
  }

  public TrustedServerLivenessFilter(NodeId trusted, CertificateAuthority ca){
    this.trustedServer = trusted;
    largestVV = new CounterVV();
    certificateAuthority = ca;

  }

  public TrustedServerLivenessFilter(){
    trustedServer = null;
  }
  
  public LinkedList<Attribute> getAttribute(){
    LinkedList<Attribute> attr = new LinkedList<Attribute>();
    attr.add(Attribute.LivenessFilter);
    return attr;
  }

  public void initialize(SecurityFilter sf){
    this.securityFilter = sf;
    this.splitManager = sf.splitManager;
    certificateAuthority.initialize(sf);
    initialized = true;
  }

  /**
   *  A receiver will accept an invalidate as long as one of the two conditions is true: 
   *  1) the update is all precise (so can only contain level 1 updates) or 
   *  2) the update's endVV is included in largestVV received from the server
   */
  public boolean shouldAccept(SecureInv si){
    boolean ret = true;
    try{
      if(si instanceof SecureCheckpoint){
        // scan the update for any liveness certificates and use them before applying them
        SecureCheckpoint sc = (SecureCheckpoint)si;
        boolean found = false;
        for(Object o: sc.getOrderedUpdateList()){
          if(o instanceof LivenessCertificate){
            processLivenessCertificate((LivenessCertificate)o);
            found = true;
          }
          if(!(o instanceof PreciseInv)){
            NodeAHSTuple nat = (NodeAHSTuple)o;
            List<AHSEntry> ahsEntries = nat.getAHS().getSortedListOfEntries();
            ret &= largestVV.includes(new AcceptStamp(ahsEntries.get(ahsEntries.size()-1).getEndTS(), nat.getNodeId()));
          }
        }

        if(found && !ret){
          ret = shouldAccept(si);
//          for(Object o: sc.getOrderedUpdateList()){
//            if(!(o instanceof PreciseInv)){
//              NodeAHSTuple nat = (NodeAHSTuple)o;
//              List<AHSEntry> ahsEntries = nat.getAHS().getSortedListOfEntries();
//              ret &= largestVV.includes(new AcceptStamp(ahsEntries.get(ahsEntries.size()-1).getEndTS(), nat.getNodeId()));
//            }
//          }
        }

	System.out.println("shouldAccept returning " + ret + " found " + found);
        return ret;
      }else if(si instanceof PreciseInv){
        if(si instanceof LivenessCertificate){
          processLivenessCertificate((LivenessCertificate)si);
        }
        return true;
      }else{
        ret = largestVV.includes(si.getEndVV());
	System.out.println("shouldAccept returning " + ret + " for summarized update ");
        return ret;
      }
    }finally{
      assert ret: largestVV;
    }
  }

  public boolean shouldSummarize(VV endVV){
    return largestVV.includes(endVV);
  }
  
  public boolean isPresent(ImpreciseInv ii){
    return shouldSummarize(ii.getEndVV());
  }
  
  public boolean isPresent(PreciseInv pi){
    return shouldSummarize(pi.getEndVV());
  }
  
  public boolean isPresent(NodeId nodeId, AHSEntry ahsEntry){
    CounterVV acceptVV = new CounterVV();
    acceptVV.advanceTimestamps(new AcceptStamp(ahsEntry.getEndTS(), nodeId));
    return shouldSummarize(new AcceptVV(acceptVV));
  }

  public void union(LivenessFilter lf){
    if(lf instanceof TrustedServerLivenessFilter){
      TrustedServerLivenessFilter tlf = (TrustedServerLivenessFilter)lf;
      if(tlf.trustedServer.equals(trustedServer)){
        largestVV.addMaxVV(tlf.largestVV);
      }
    }
  }

  public AcceptVV getLargestVV(){
    return new AcceptVV(largestVV);
  }

  protected void processLivenessCertificate(LivenessCertificate lc){
    if(lc.getCreator().equals(trustedServer) && securityFilter.verifySignature(lc)){
      largestVV.advanceTimestamps(lc.getCertifiedVV());
      certificateAuthority.processLivenessCertificate(lc);
    }
  }

  public boolean ensureVerifiability(VV splitVV, AHSMap ahsMap, NodeId trusted){
    // in this policy we, ensure that our AHS is always linear; therefore even for ensureVerfiability, we ensure that the 
    // AHS is split at the splitVV instead of 
    assert initialized;
    VVIterator vvi = splitVV.getIterator();
    boolean verifiable = true;
    while(vvi.hasMoreElements() && verifiable){
      NodeId n = vvi.getNext();
      long ts = splitVV.getStampByIteratorToken(n);
      TreeNode tn = ahsMap.getTreeNodeTS(n, ts);
      if(tn == null || tn.getEndTS() != ts){
        verifiable = false;
      }
    }
    if(!verifiable){
      return splitManager.tryAndSplit(splitVV, trusted);
    }else{
      return true;
    }
    //return false;
  }

  public boolean ensureCompatibility(VV splitVV, AHSMap ahsMap, NodeId trusted){
    assert initialized;
    return splitManager.tryAndSplit(splitVV, trusted);
    //return false;
  }

  public CertificateAuthority getCA(){
    assert initialized;
    return certificateAuthority;
  }

  public void writeExternal(ObjectOutput out) throws IOException{
    if(SangminConfig.forkjoin){
      out.writeObject(trustedServer);
    } else {
      out.writeLong(trustedServer.getIDint());
    }
    new AcceptVV(largestVV).writeExternal(out);    
  }

  public void readExternal(ObjectInput in) 
  throws IOException, ClassNotFoundException{
    //long  serverIdInt = in.readLong();
    NodeId serverId;
    if(SangminConfig.forkjoin){
      serverId = (BranchID)in.readObject();
    } else {
      serverId = new NodeId(in.readLong());
    }
    AcceptVV largestVV_ = new AcceptVV();
    largestVV_.readExternal(in);

    Field[] f = new Field[6];

    try{

      f[0] = TrustedServerLivenessFilter.class.getDeclaredField("certificateAuthority");
      f[1] = TrustedServerLivenessFilter.class.getDeclaredField("initialized");
      f[2] = TrustedServerLivenessFilter.class.getDeclaredField("largestVV");
      f[3] = TrustedServerLivenessFilter.class.getDeclaredField("securityFilter");
      f[4] = TrustedServerLivenessFilter.class.getDeclaredField("splitManager");
      f[5] = TrustedServerLivenessFilter.class.getDeclaredField("trustedServer");
    }catch(NoSuchFieldException ne){
      System.err.println(ne.toString());
      ne.printStackTrace();
      System.exit(-1);

    }
    try{
      AccessibleObject.setAccessible(f, true);
    } catch (SecurityException se){
      System.err.println(se.toString());
      se.printStackTrace();
      System.exit(-1);
    }
    try{
      f[0].set(this, null);
      f[1].setBoolean(this, false);
      f[2].set(this, new CounterVV(largestVV_));
      f[3].set(this, null);
      f[4].set(this, null);
      f[5].set(this, serverId);

    }catch(IllegalArgumentException ie){
      System.err.println(ie.toString());
      ie.printStackTrace();
      System.exit(-1);
    }catch(IllegalAccessException iae){
      System.err.println(iae.toString());
      iae.printStackTrace();
      System.exit(-1);
    }

    try{
      AccessibleObject.setAccessible(f, false);
    } catch (SecurityException se){
      System.err.println(se.toString());
      se.printStackTrace();
      System.exit(-1);
    }

    assert(in.available() == 0);
  }

  public String toString(){
    return "TrustedServerTraceLivenessFilter: " + ((trustedServer!=null)?trustedServer:"NULL") + " largestVV "  +largestVV;
  }

  public LivenessFilter clone(){
   TrustedServerLivenessFilter tslf = new TrustedServerLivenessFilter();
   tslf.largestVV = new CounterVV(largestVV);
   tslf.trustedServer = (NodeId)trustedServer.clone();
   return tslf;
  }
  
  public NodeId getTrustedServer(){
    return trustedServer;
  }
}
