package code.security.liveness;

import java.io.IOException;
import java.io.ObjectOutput;
import java.io.ObjectInput;
import java.lang.reflect.AccessibleObject;
import java.lang.reflect.Field;
import java.security.PublicKey;
import java.util.Enumeration;
import java.util.LinkedList;
import java.util.List;

import code.branchDetecting.BranchID;
import code.security.*;
import code.*;
import code.security.ahs.*;
import code.security.holesync.filter.Filter.Attribute;


/**
 * This class implements the trusted liveness server policy of accepting updates only if they are precise or if they are 
 * covered by a certificate signed by the liveness server
 * @author princem
 *
 */
public class TrustedServerTraceLivenessFilter implements LivenessFilter{
  NodeId trustedServer;
  NodeId myId;
  SplitManager splitManager;
  CertificateAuthority ca;
  SecurityFilter securityFilter;
  boolean initialized = false;
  
//  public TrustedServerTraceLivenessFilter(NodeId trusted, SecurityFilter securityFilter){
//    this.trustedServer = trusted;
//    largestVV = new CounterVV();
//    this.securityFilter = securityFilter;
//    this.splitManager = securityFilter.splitManager;
//    ca = new NoopCertificateAuthority();
//  }
//  
  
  public TrustedServerTraceLivenessFilter(NodeId trusted, NodeId myId, CertificateAuthority ca){
    this.trustedServer = trusted;
    this.ca = ca;
    this.myId = myId;
  }
  
  public TrustedServerTraceLivenessFilter(NodeId trusted){
    this.trustedServer = trusted;
    ca = new NoopCertificateAuthority();
  }
  
  public TrustedServerTraceLivenessFilter(){
    ca = new NoopCertificateAuthority();
  }
  
  public void initialize(SecurityFilter sf){
    this.securityFilter = sf;
    this.splitManager = sf.splitManager;
    ca.initialize(sf);
    myId = sf.core.getMyNodeId();
    initialized = true;
  }
  
  public LinkedList<Attribute> getAttribute(){
    LinkedList<Attribute> attr = new LinkedList<Attribute>();
    attr.add(Attribute.LivenessFilter);
    return attr;
  }
  
  public boolean shouldAccept(SecureInv si){
    // the following assert says that an initialized trusted server should not get an imprecise inval that is newer than cur VV
    if(initialized && (trustedServer != null) && myId.equals(trustedServer) && !securityFilter.core.getCurrentVV().includes(si.getEndVV()) && !(si instanceof PreciseInv)){
      assert si instanceof SecureCheckpoint;
      SecureCheckpoint sc = (SecureCheckpoint)si;
      for(Object o: sc.getOrderedUpdateList()){
        assert (o instanceof PreciseInv): si;
      }
    }
    return true;
  }

  public boolean ensureVerifiability(VV splitVV, AHSMap ahsMap, NodeId sender){
    assert initialized;
    // in this policy we, ensure that our AHS is always linear; therefore even for ensureVerfiability, we ensure that the 
    // AHS is split at the splitVV instead of 
    NodeId server = null;
    if(trustedServer != null){
      server = trustedServer;
    }else{
      server= sender;
    }
    
    VVIterator vvi = splitVV.getIterator();
    boolean verifiable = true;
    while(vvi.hasMoreElements() && verifiable){
      NodeId n = vvi.getNext();
      long ts = splitVV.getStampByIteratorToken(n);
      TreeNode tn = ahsMap.getTreeNodeTS(n, ts);
      if(tn == null || tn.getEndTS() != ts){
        verifiable = false;
      }
    }
    if(!verifiable){
      return splitManager.tryAndSplit(splitVV, server);
    }else{
      return true;
    }
    //return false;
  }
  
  public boolean ensureCompatibility(VV splitVV, AHSMap ahsMap, NodeId sender){
    assert initialized;
    if(trustedServer != null){
      return splitManager.tryAndSplit(splitVV, trustedServer);
    }else{
      return splitManager.tryAndSplit(splitVV, sender);
    }
    //return false;
  }
  
  public boolean shouldSummarize(VV endVV){
    if(trustedServer == null){
      return true;
    }
    else if(myId.equals(trustedServer)){
      return false;
    }else{
      return true;
    }
  }
  
  public boolean isPresent(ImpreciseInv ii){
    return shouldSummarize(ii.getEndVV());
  }
  
  public boolean isPresent(NodeId nodeId, AHSEntry ahsEntry){
    CounterVV acceptVV = new CounterVV();
    acceptVV.advanceTimestamps(new AcceptStamp(ahsEntry.getEndTS(), nodeId));
    return shouldSummarize(new AcceptVV(acceptVV));
  }
  
  public boolean isPresent(PreciseInv pi){
    return shouldSummarize(pi.getEndVV());
  }
  
  public void union(LivenessFilter lf){
    
  }
  
  public CertificateAuthority getCA(){
    return ca;
  }
  
  public LivenessFilter clone(){
    TrustedServerTraceLivenessFilter tslf = new TrustedServerTraceLivenessFilter();
    tslf.myId = (NodeId)myId.clone();
    tslf.trustedServer = (trustedServer!=null?(NodeId)trustedServer.clone():null);
    return tslf;
   }
  
  public void writeExternal(ObjectOutput out) throws IOException{
    //assert initialized;
    
    if(SangminConfig.forkjoin){
      out.writeObject(myId);
      if(trustedServer != null){
        out.writeBoolean(true);
        out.writeObject(trustedServer);
      } else {
        out.writeBoolean(false);
      }
    } else {
      out.writeLong(myId.getIDint());
      if(trustedServer != null){
        out.writeLong(trustedServer.getIDint());
      }else{
        out.writeLong(-1);
      }
    }
  }
  
  public void readExternal(ObjectInput in) 
  throws IOException, ClassNotFoundException{
    NodeId myId = null;
    NodeId serverId = null;
    if(SangminConfig.forkjoin){
      myId = (BranchID)in.readObject();
      if(in.readBoolean()){
        serverId = (BranchID)in.readObject();
      }
    } else {
      long  myIdInt = in.readLong();
      long  serverIdInt = in.readLong();
      myId = new NodeId(myIdInt);
      if(serverIdInt>=0){
        serverId = new NodeId(serverIdInt);
      }
    }
    
    
    Field[] f = new Field[6];
    
    try{

      f[0] = TrustedServerTraceLivenessFilter.class.getDeclaredField("ca");
      f[1] = TrustedServerTraceLivenessFilter.class.getDeclaredField("initialized");
      f[2] = TrustedServerTraceLivenessFilter.class.getDeclaredField("myId");
      f[3] = TrustedServerTraceLivenessFilter.class.getDeclaredField("securityFilter");
      f[4] = TrustedServerTraceLivenessFilter.class.getDeclaredField("splitManager");
      f[5] = TrustedServerTraceLivenessFilter.class.getDeclaredField("trustedServer");
    }catch(NoSuchFieldException ne){
      System.err.println(ne.toString());
      ne.printStackTrace();
      System.exit(-1);

    }
    try{
      AccessibleObject.setAccessible(f, true);
    } catch (SecurityException se){
      System.err.println(se.toString());
      se.printStackTrace();
      System.exit(-1);
    }
    try{
      f[0].set(this, null);
      f[1].setBoolean(this, false);
      f[2].set(this, myId);
      f[3].set(this, null);
      f[4].set(this, null);
      f[5].set(this, serverId);

      
    }catch(IllegalArgumentException ie){
      System.err.println(ie.toString());
      ie.printStackTrace();
      System.exit(-1);
    }catch(IllegalAccessException iae){
      System.err.println(iae.toString());
      iae.printStackTrace();
      System.exit(-1);
    }

    try{
      AccessibleObject.setAccessible(f, false);
    } catch (SecurityException se){
      System.err.println(se.toString());
      se.printStackTrace();
      System.exit(-1);
    }
    
    assert(in.available() == 0);
  }
  
  public String toString(){
    return "TrustedServerTraceLivenessFilter: " + ((trustedServer!=null)?trustedServer:"NULL") + " myId " + myId;
  }

  public NodeId getTrustedServer(){
    return trustedServer;
  }

  public NodeId getMyId(){
    return myId;
  }
}
