package code.security.liveness.unit;

import code.StreamId;
import code.security.ahs.*;

import java.security.PrivateKey;
import java.security.PublicKey;

import code.LocalController;
import code.AcceptStamp;
import code.AcceptVV;
import code.Config;
import code.GeneralInv;
import code.ImmutableBytes;
import code.NodeId;
import code.ObjId;
import code.ObjInvalTarget;
import code.PreciseInv;
import code.SubscriptionSet;
import code.SummaryHash;
import code.security.*;
import code.security.ahs.DependencyVV;
import junit.framework.TestCase;
import code.security.liveness.*;
import code.security.holesync.filter.*;
import code.security.unit.TestSecureRMIClient;
import code.security.holesync.*;
import junit.textui.TestRunner;
import junit.framework.*;
import java.util.*;
import java.io.*;
public class CertificateUnit extends TestCase{

  public static final String TEST_ALL_TEST_TYPE = "UNIT";

  int totalNodes = 10;
  int nodeCount = 0;
  private static final boolean dbg = true; 
  //create SecureCore as URANode constructor did
  boolean filterOn = true;
  boolean cleanDb = true;  // start from empty



  /*
   * Fixtures are run before and after each test case
   * to set up environment in which tests must run.
   */
  protected void setUp() throws Exception{
    super.setUp();
    /* 
     * TBD: Insert other fixture code here
     * e.g., this.objectUnderTest = new ClassToBeTested(...);
     */
    makePractiConfig(0, totalNodes);//generate nodes: 0, 1, 2, 3, 4

    Config.readKeys();




    //SangminConfig.securityLevel = SangminConfig.SIGNATURE;
  }


  private static void makePractiConfig(long first, int total){
    Config.createEmptyConfig();

    long NODE_1_ID = first;
    int port = 9921;
    String NODE_1_IP = "localhost";

    for(int i = 0; i < total; i++){

      Config.addOneNodeConfig(new NodeId(NODE_1_ID++),
          NODE_1_IP,
          port++,
          port++,
          port++,
          port++,
          port++,
          "test" + File.separatorChar + "local-" + 
          NODE_1_ID + ".db",
          "/*",
          -1L,
          NODE_1_IP,
          port++,
          port++,
          -1,
          Config.CACHE_SIZE_BYTES_DEFAULT,
          Config.MAX_LOG_DISK_SIZE_BYTES,
          Config.MAX_LOG_MEM_SIZE_BYTES);
    }

    Config.readKeys();//needed for secure practi

  }


  protected void tearDown() throws Exception{
    /* 
     * TBD: Insert other fixture cleanup code here
     */
    super.tearDown();
  }

  /**
   *  test whether the liveness filter is correctly processing these certificates
   */
  public void testLivenessFilter(){
    // 3 nodes: A:1 trusted server, B:  writer (writes to /a/* 1-4 and /b/* 5-8); also syncs both ways with A after writes to /a/*
    // C: syncs with B after it has synced with A
    // expected: C receives summarized information for /a/* while precise information for /b/*)
    int [] nodeId = new int[]{nodeCount++, nodeCount++, nodeCount++};

    try{
      SecureCore[] SecureCores = new SecureCore[totalNodes];
      SecurityFilter[] filters = new SecurityFilter[totalNodes];
      SecureRMIClient myRMIClient = new SecureRMIClient();   
      SecureInMemLogIterator[] iters = new SecureInMemLogIterator[totalNodes]; 
      NodeId trusted = new NodeId(nodeId[0]);

      for(int nId = nodeId[0]; nId<=nodeId[nodeId.length-1]; nId++){

        CertificateAuthority ca;
        if(nId == nodeId[0]){
          ca = new TrustedServerCertificateAuthority();
        }else{
          ca = new NoopCertificateAuthority();
        }
        LinkedList<Filter> l = new LinkedList<Filter>();
        l.add(new SubscriptionSetFilter(SubscriptionSet.makeSubscriptionSet("/*")));
        SecureCores[nId] = new SecureCore(myRMIClient, filterOn, cleanDb, new NodeId(nId), false, 
            new code.security.liveness.TrustedServerLivenessFilter(trusted, ca), l);
        SecureCores[nId].testSkipRecoverLocalState();
        filters[nId] = SecureCores[nId].getSecurityFilter();
      }


      //
      //populate 1 write to node 0
      //
      ObjId oid = new ObjId("/foo");
      byte[] bdy = new byte[5];
      bdy[0] = 10;
      bdy[1] = 9;
      bdy[2] = 8;
      bdy[3] = 7;
      bdy[4] = 6;
      ImmutableBytes body = new ImmutableBytes(bdy);
      long offset = 0;
      long len = bdy.length;



      //
      // populate 10 write to node 0
      //
      PreciseInv pi1 = null;
      int stage = 0;
      int writeNum = 4;
      for(int i = 0; i < writeNum; i ++ ){
        oid = new ObjId("/A/" + stage + "/" + i);
        SecureCores[nodeId[1]].write(oid, offset, len, 
            SecureCore.DEFAULT_PRIORITY, 
            body, 
            false, 
            Long.MAX_VALUE);
      }
      
      this.checkpointSyncNodes(SecureCores[nodeId[1]], SecureCores[nodeId[0]], SubscriptionSet.makeSubscriptionSet("/*"));
      try{
        Thread.sleep(15000);
      }catch(InterruptedException e){
        // TODO Auto-generated catch block
        e.printStackTrace();
      }
      AcceptVV cert = SecureCores[nodeId[0]].securityFilter.getLastCertificate();
      assert cert.includes(new AcceptStamp(writeNum-1, new NodeId(nodeId[1]))): cert;
      this.checkpointSyncNodes(SecureCores[nodeId[0]], SecureCores[nodeId[1]], SubscriptionSet.makeSubscriptionSet("/*"));
      cert = ((TrustedServerLivenessFilter)SecureCores[nodeId[1]].securityFilter.getLivenessFilter()).getLargestVV();
      assert cert.includes(new AcceptStamp(writeNum-1, new NodeId(nodeId[1]))): cert;
      
      //by now node 1 has the certificate
      //lets do a few more writes at node 1
      
      for(int i = 0; i < 1; i ++ ){
        oid = new ObjId("/A/" + stage + "/" + i);
        SecureCores[nodeId[1]].write(oid, offset, len, 
            SecureCore.DEFAULT_PRIORITY, 
            body, 
            false, 
            Long.MAX_VALUE);
      }
      for(int i = 0; i < 1; i ++ ){
        oid = new ObjId("/B/" + stage + "/" + i);
        SecureCores[nodeId[1]].write(oid, offset, len, 
            SecureCore.DEFAULT_PRIORITY, 
            body, 
            false, 
            Long.MAX_VALUE);
      }
      SecureCheckpoint chkPt = 
        this.checkpointSyncNodes(SecureCores[nodeId[1]], SecureCores[nodeId[2]], SubscriptionSet.makeSubscriptionSet("/B/*"));
      cert = ((TrustedServerLivenessFilter)SecureCores[nodeId[2]].securityFilter.getLivenessFilter()).getLargestVV();
      assert cert.includes(new AcceptStamp(writeNum-1, new NodeId(nodeId[1]))): cert;
      
      // assert that the checkpoint contains no precise updates for /A/* and only precise updates afterwards
      for(Object o: chkPt.getOrderedUpdateList()){
        if(o instanceof SecurePreciseInv){
          if(!(o instanceof LivenessCertificate)){
            assert !cert.includes(((SecurePreciseInv)o).getAcceptStamp()):o;
          }
        }else{
          assert o instanceof NodeAHSTuple;
          
          NodeAHSTuple nat = (NodeAHSTuple)o;
          List<AHSEntry> ahsEntries = nat.getAHS().getSortedListOfEntries();
          
          assert cert.includes(new AcceptStamp(ahsEntries.get(ahsEntries.size()-1).getEndTS(), nat.getNodeId())): "cert " + cert + " NodeAHSTuple " + nat;
        }
      }
      
    }
    catch(IOException e){
      System.out.println(e);
      assert false;
    }
    
    
  }

  /**
   * test whether the certificates are propagating correctly or not? 
   */
  public void testCertificateTransfer() throws Exception{
    nonlocalCheckpointTest(new int[]{nodeCount++, nodeCount++});
  }

  /**
   * test whether the appropriate certificates are being generated under different conditions?
   */
  public void testCertificateAuthority() throws Exception{

    // initialization
    // test1: simple local precise update test
//  localPreciseTest(new int[]{nodeCount++});

    // test2: simple non-local precise update test
//  nonlocalPreciseTest(new int[]{nodeCount++, nodeCount++});

  } 

  //first test: localPreciseTest new precise updates
  public void localPreciseTest(int[] nodeId) throws Exception{
    SecureCore[] SecureCores = new SecureCore[totalNodes];
    SecurityFilter[] filters = new SecurityFilter[totalNodes];
    SecureRMIClient myRMIClient = new SecureRMIClient();   
    SecureInMemLogIterator[] iters = new SecureInMemLogIterator[totalNodes]; 
    NodeId trusted = new NodeId(nodeId[0]);

    for(int nId = nodeId[0]; nId<=nodeId[nodeId.length-1]; nId++){

      TrustedServerCertificateAuthority ca = new TrustedServerCertificateAuthority();
      LinkedList<Filter> l = new LinkedList<Filter>();
      l.add(new SubscriptionSetFilter(SubscriptionSet.makeSubscriptionSet("/*")));
      SecureCores[nId] = new SecureCore(myRMIClient, filterOn, cleanDb, new NodeId(nId), false, 
          new code.security.liveness.TrustedServerLivenessFilter(trusted, ca), l);
      SecureCores[nId].testSkipRecoverLocalState();
      filters[nId] = SecureCores[nId].getSecurityFilter();
    }

    TestSecureRMIClient rmiClientNode0 = new TestSecureRMIClient(SecureCores);
    LinkedList<Filter> l = new LinkedList<Filter>();
    l.add(new SubscriptionSetFilter(SubscriptionSet.makeSubscriptionSet("/*")));
    SecureCore destSecureCore = new SecureCore(rmiClientNode0, filterOn, cleanDb, new NodeId(nodeId[nodeId.length-1]+1), 
        false, new code.security.liveness.TrustedServerTraceLivenessFilter(), l);
    SecurityFilter destSecurityFilter = destSecureCore.getSecurityFilter();

    //
    //populate 1 write to node 0
    //
    ObjId oid = new ObjId("/foo");
    byte[] bdy = new byte[5];
    bdy[0] = 10;
    bdy[1] = 9;
    bdy[2] = 8;
    bdy[3] = 7;
    bdy[4] = 6;
    ImmutableBytes body = new ImmutableBytes(bdy);
    long offset = 0;
    long len = bdy.length;



    //
    // populate 10 write to node 0
    //
    PreciseInv pi1 = null;
    int stage = 0;
    int writeNum = 10;
    for(int i = 0; i < writeNum; i ++ ){
      oid = new ObjId("/"+ nodeId[0] + "/" + stage + "/" + i);
      SecureCores[nodeId[0]].write(oid, offset, len, 
          SecureCore.DEFAULT_PRIORITY, 
          body, 
          false, 
          Long.MAX_VALUE);
    }

    Thread.sleep(TrustedServerCertificateAuthority.timerDelay + 1000);
    AcceptVV cert = SecureCores[nodeId[0]].securityFilter.getLastCertificate();
    assert cert.includes(new AcceptStamp(writeNum-1, new NodeId(nodeId[0]))): cert;

  }

//first test: localPreciseTest new precise updates
  public void nonlocalPreciseTest(int[] nodeId) throws Exception{
    SecureCore[] SecureCores = new SecureCore[totalNodes];
    SecurityFilter[] filters = new SecurityFilter[totalNodes];
    SecureRMIClient myRMIClient = new SecureRMIClient();   
    SecureInMemLogIterator[] iters = new SecureInMemLogIterator[totalNodes]; 
    NodeId trusted = new NodeId(nodeId[0]);

    for(int nId = nodeId[0]; nId<=nodeId[nodeId.length-1]; nId++){

      CertificateAuthority ca;
      if(nId == nodeId[0]){
        ca = new TrustedServerCertificateAuthority();
      }else{
        ca = new NoopCertificateAuthority();
      }
      LinkedList<Filter> l = new LinkedList<Filter>();
      l.add(new SubscriptionSetFilter(SubscriptionSet.makeSubscriptionSet("/*")));
      SecureCores[nId] = new SecureCore(myRMIClient, filterOn, cleanDb, new NodeId(nId), false, 
          new code.security.liveness.TrustedServerLivenessFilter(trusted, ca), l);
      SecureCores[nId].testSkipRecoverLocalState();
      filters[nId] = SecureCores[nId].getSecurityFilter();
    }


    //
    //populate 1 write to node 0
    //
    ObjId oid = new ObjId("/foo");
    byte[] bdy = new byte[5];
    bdy[0] = 10;
    bdy[1] = 9;
    bdy[2] = 8;
    bdy[3] = 7;
    bdy[4] = 6;
    ImmutableBytes body = new ImmutableBytes(bdy);
    long offset = 0;
    long len = bdy.length;



    //
    // populate 10 write to node 0
    //
    PreciseInv pi1 = null;
    int stage = 0;
    int writeNum = 10;
    for(int i = 0; i < writeNum; i ++ ){
      oid = new ObjId("/"+ nodeId[0] + "/" + stage + "/" + i);
      SecureCores[nodeId[1]].write(oid, offset, len, 
          SecureCore.DEFAULT_PRIORITY, 
          body, 
          false, 
          Long.MAX_VALUE);
    }

    syncNodes(SecureCores[nodeId[1]], SecureCores[nodeId[0]]);
    Thread.sleep(TrustedServerCertificateAuthority.timerDelay + 1000);
    AcceptVV cert = SecureCores[nodeId[0]].securityFilter.getLastCertificate();
    assert cert.includes(new AcceptStamp(writeNum-1, new NodeId(nodeId[1]))): cert;

    // the following check ensures the propagation of writes in a precise stream
    syncNodes(SecureCores[nodeId[0]], SecureCores[nodeId[1]]);
    cert = ((TrustedServerLivenessFilter)SecureCores[nodeId[1]].securityFilter.getLivenessFilter()).getLargestVV();
    assert cert.includes(new AcceptStamp(writeNum-1, new NodeId(nodeId[1]))): cert;

  }


  public void nonlocalCheckpointTest(int[] nodeId) throws Exception{
    SecureCore[] SecureCores = new SecureCore[totalNodes];
    SecurityFilter[] filters = new SecurityFilter[totalNodes];
    SecureRMIClient myRMIClient = new SecureRMIClient();   
    SecureInMemLogIterator[] iters = new SecureInMemLogIterator[totalNodes]; 
    NodeId trusted = new NodeId(nodeId[0]);

    for(int nId = nodeId[0]; nId<=nodeId[nodeId.length-1]; nId++){

      CertificateAuthority ca;
      if(nId == nodeId[0]){
        ca = new TrustedServerCertificateAuthority();
      }else{
        ca = new NoopCertificateAuthority();
      }
      LinkedList<Filter> l = new LinkedList<Filter>();
      l.add(new SubscriptionSetFilter(SubscriptionSet.makeSubscriptionSet("/*")));
      SecureCores[nId] = new SecureCore(myRMIClient, filterOn, cleanDb, new NodeId(nId), false, 
          new code.security.liveness.TrustedServerLivenessFilter(trusted, ca), l);
      SecureCores[nId].testSkipRecoverLocalState();
      filters[nId] = SecureCores[nId].getSecurityFilter();
    }

    TestSecureRMIClient rmiClientNode0 = new TestSecureRMIClient(SecureCores);
    LinkedList<Filter> l = new LinkedList<Filter>();
    l.add(new SubscriptionSetFilter(SubscriptionSet.makeSubscriptionSet("/*")));
    SecureCore destSecureCore = new SecureCore(rmiClientNode0, filterOn, cleanDb, new NodeId(nodeId[nodeId.length-1]+1), false, new code.security.liveness.TrustedServerTraceLivenessFilter(), l);
    SecurityFilter destSecurityFilter = destSecureCore.getSecurityFilter();

    //
    //populate 1 write to node 0
    //
    ObjId oid = new ObjId("/foo");
    byte[] bdy = new byte[5];
    bdy[0] = 10;
    bdy[1] = 9;
    bdy[2] = 8;
    bdy[3] = 7;
    bdy[4] = 6;
    ImmutableBytes body = new ImmutableBytes(bdy);
    long offset = 0;
    long len = bdy.length;



    //
    // populate 10 write to node 0
    //
    PreciseInv pi1 = null;
    int stage = 0;
    int writeNum = 10;
    for(int i = 0; i < writeNum; i ++ ){
      oid = new ObjId("/"+ nodeId[0] + "/" + stage + "/");
      SecureCores[nodeId[0]].write(oid, offset, len, 
          SecureCore.DEFAULT_PRIORITY, 
          body, 
          false, 
          Long.MAX_VALUE);
    }

    Thread.sleep(TrustedServerCertificateAuthority.timerDelay + 1000);
    AcceptVV cert = SecureCores[nodeId[0]].securityFilter.getLastCertificate();
    assert cert.includes(new AcceptStamp(writeNum-1, new NodeId(nodeId[0]))): cert;

    // the following check ensures the propagation of writes in a precise stream
    SubscriptionSet ss = SubscriptionSet.makeSubscriptionSet("/*");
    this.checkpointSyncNodes(SecureCores[nodeId[0]], SecureCores[nodeId[1]], ss/*, false*/);

    cert = ((TrustedServerLivenessFilter)SecureCores[nodeId[1]].securityFilter.getLivenessFilter()).getLargestVV();
    assert cert.includes(new AcceptStamp(writeNum-1, new NodeId(nodeId[0]))): cert;
    assert SecureCores[nodeId[1]].getCurrentVV().equalsIgnoreNegatives(SecureCores[nodeId[0]].getCurrentVV()):"node 0 " + SecureCores[nodeId[0]].getCurrentVV() + " \n node 1: " + SecureCores[nodeId[1]].getCurrentVV() ;
  }




  public void testWriteExternal(){
    try{
      PrivateKey privKey;
      PublicKey pubKey;
      int nodeId = 0;

      privKey = (PrivateKey)Config.privateKeys.get(new Long(nodeId));
      pubKey = (PublicKey)Config.publicKeys.get(new Long(nodeId));

      byte[] buffer = new byte[]{0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19};
      byte[] buffer2 = new byte[]{10,11,12,13,14,15,16,17,18,19,10,11,12,13,14,15,16,17,18,19};

      SummaryHash sh = new SummaryHash(buffer, true);    
      DataHash dh = new DataHash(buffer,true);

      DependencyVV dvv = new DependencyVV();
      for(int i=0; i < 1; i++){
        dvv.put(new NodeId(i), i);
      }

      DependencyVV cvv = new DependencyVV();
      for(int i=0; i < 1; i++){
        cvv.put(new NodeId(i), 10-i);
      }

      LivenessCertificate spi = new LivenessCertificate(new ObjInvalTarget(new ObjId("/rand1"), 5, 10),
          new AcceptStamp(1, new NodeId(nodeId)),
          new AcceptStamp(2, new NodeId(nodeId)),
          dvv,
          sh,
          new AcceptVV(cvv), 
          privKey);
//    Create a stream for writing.
      FileOutputStream fos = new FileOutputStream( "tmpz" );

      //  Next, create an object that can write to that file.
      ObjectOutputStream outStream = 
        new ObjectOutputStream( fos );

      spi.writeExternal(outStream);

      outStream.flush();


//    Create a stream for reading.
      FileInputStream fis = new FileInputStream( "tmpz" );

      //  Next, create an object that can read from that file.
      ObjectInputStream inStream = new ObjectInputStream( fis );

      LivenessCertificate spi2 = new LivenessCertificate();
      spi2.readExternal(inStream);

      if( !Arrays.equals(spi.getSH().getValue(), spi2.getSH().getValue())){
        System.out.println("spi1 : " + spi);
        System.out.println("spi2 : " + spi2);
        fail("serialization: SummaryHash does not match1");
      }

      if(!spi.getSH().equals(spi2.getSH())){
        System.out.println("spi1 : " + spi);
        System.out.println("spi2 : " + spi2);
        fail("serialization: SummaryHash does not match2");
      }

      if(!spi.getDH().equals(spi2.getDH())){
        fail("serialization: DataHash does not match");
      }

      if(!(spi.getDVV().includes(spi2.getDVV()) && spi2.getDVV().includes(spi.getDVV()))){
        fail("serialization: DVV does not match");
      }

      if(!(spi.getCertifiedVV().includes(spi2.getCertifiedVV()) && spi2.getCertifiedVV().includes(spi.getCertifiedVV()))){
        fail("serialization: CertifiedVV does not match");
      }

    }catch(Exception e){
      System.out.println("Got Exception: "+ e.toString());
      fail("exception");
    }
  }

  public void testWriteExternalTraceLivenessFilter(){
    try{

      NodeId trusted = new NodeId(0);
      NodeId myId = new NodeId(1);
      TrustedServerTraceLivenessFilter tlf = new TrustedServerTraceLivenessFilter(trusted, myId, new NoopCertificateAuthority());

      //    Create a stream for writing.
      FileOutputStream fos = new FileOutputStream( "tmpz" );

      //  Next, create an object that can write to that file.
      ObjectOutputStream outStream = 
        new ObjectOutputStream( fos );

      tlf.writeExternal(outStream);

      outStream.flush();


//    Create a stream for reading.
      FileInputStream fis = new FileInputStream( "tmpz" );

      //  Next, create an object that can read from that file.
      ObjectInputStream inStream = new ObjectInputStream( fis );

      TrustedServerTraceLivenessFilter tlf1 = new TrustedServerTraceLivenessFilter();
      tlf1.readExternal(inStream);

      if( !tlf.getMyId().equals(tlf1.getMyId())){
        System.out.println("filter1 : " + tlf);
        System.out.println("filter2 : " + tlf1);
        fail("serialization: myId does not match1");
      }

      if( !tlf.getTrustedServer().equals(tlf1.getTrustedServer())){
        System.out.println("filter1 : " + tlf);
        System.out.println("filter2 : " + tlf1);
        fail("serialization: trusted does not match1");
      }

    }catch(Exception e){
      System.out.println("Got Exception: "+ e.toString());
      fail("exception");
    }
  }


  public void testWriteExternalLivenessFilter(){
    try{

      NodeId trusted = new NodeId(0);
      TrustedServerLivenessFilter tslf = new TrustedServerLivenessFilter(trusted, new NoopCertificateAuthority());

      //    Create a stream for writing.
      FileOutputStream fos = new FileOutputStream( "tmpz" );

      //  Next, create an object that can write to that file.
      ObjectOutputStream outStream = 
        new ObjectOutputStream( fos );

      tslf.writeExternal(outStream);

      outStream.flush();


//    Create a stream for reading.
      FileInputStream fis = new FileInputStream( "tmpz" );

      //  Next, create an object that can read from that file.
      ObjectInputStream inStream = new ObjectInputStream( fis );


      TrustedServerLivenessFilter tslf1 = new TrustedServerLivenessFilter();
      tslf1.readExternal(inStream);

      if( !tslf.getLargestVV().equalsIgnoreNegatives(tslf1.getLargestVV())){
        System.out.println("filter1 : " + tslf);
        System.out.println("filter2 : " + tslf1);
        fail("serialization: largestVV does not match1");
      }

      if( !trusted.equals(tslf.getTrustedServer())){
        System.out.println("filter1 : " + tslf);
        System.out.println("filter2 : " + tslf1);
        fail("serialization: trusted does not match1");
      }

    }catch(Exception e){
      System.out.println("Got Exception: "+ e.toString());
      fail("exception");
    }
  }


  public void testWriteExternalBlockingLivenessFilter(){
    try{

      BlockingLivenessFilter blf = new BlockingLivenessFilter();
      //    Create a stream for writing.
      FileOutputStream fos = new FileOutputStream( "tmpz" );

      //  Next, create an object that can write to that file.
      ObjectOutputStream outStream = 
        new ObjectOutputStream( fos );

      blf.writeExternal(outStream);

      outStream.flush();


//    Create a stream for reading.
      FileInputStream fis = new FileInputStream( "tmpz" );

      //  Next, create an object that can read from that file.
      ObjectInputStream inStream = new ObjectInputStream( fis );

      BlockingLivenessFilter blf1 = new BlockingLivenessFilter();
      blf1.readExternal(inStream);

      NodeId trusted = new NodeId(0);
      NodeId myId = new NodeId(1);
      TrustedServerTraceLivenessFilter tlf = new TrustedServerTraceLivenessFilter(trusted, myId, new NoopCertificateAuthority());
      //    Create a stream for writing.
      fos = new FileOutputStream( "tmpz1" );

      //  Next, create an object that can write to that file.
      outStream = 
        new ObjectOutputStream( fos );

      tlf.writeExternal(outStream);

      outStream.flush();


//    Create a stream for reading.
      fis = new FileInputStream( "tmpz1" );

      //  Next, create an object that can read from that file.
      inStream = new ObjectInputStream( fis );

      TrustedServerTraceLivenessFilter tlf1 = new TrustedServerTraceLivenessFilter();
      tlf1.readExternal(inStream);

      if( !myId.equals(tlf1.getMyId())){
        System.out.println("filter1 : " + tlf);
        System.out.println("filter2 : " + tlf1);
        fail("serialization: myId does not match1");
      }

      if( !trusted.equals(tlf1.getTrustedServer())){
        System.out.println("filter1 : " + tlf);
        System.out.println("filter2 : " + tlf1);
        fail("serialization: trusted does not match1");
      }


    }catch(Exception e){
      System.out.println("Got Exception: "+ e.toString());
      fail("exception");
    }
  }

  private boolean verifyAndApplyToNode(SecureCore SecureCore, GeneralInv gi){
    SecurityFilter filter = SecureCore.getSecurityFilter();
    boolean ret = filter.verifyAndApply(gi, SecureCore.getMyNodeId());
    SecureCore.applyInval(gi, null);
    return ret;
  }

  private void syncNodes(SecureCore srcSecureCore, SecureCore destSecureCore){
    SecurityFilter srcFilter = srcSecureCore.getSecurityFilter();
    SecurityFilter destFilter = destSecureCore.getSecurityFilter();
    SecureInMemLogIterator srcIter = srcFilter.makeSecureInMemLogIterator(AcceptVV.makeVVAllNegatives());
    SecureInMemLogIterator destIter = destFilter.makeSecureInMemLogIterator(AcceptVV.makeVVAllNegatives());
    while(srcIter.hasNext()){
      GeneralInv gi = srcIter.next();
      assert this.verifyAndApplyToNode(destSecureCore, gi):"can't apply" + gi.toString();
    }
    //reset source
    srcIter = srcFilter.makeSecureInMemLogIterator(AcceptVV.makeVVAllNegatives());
    while(srcIter.hasNext() || destIter.hasNext()){
      assert destIter.hasNext();
      assert srcIter.hasNext();

      GeneralInv spiDest = destIter.next();
      GeneralInv spiSrc = srcIter.next();
      assert spiDest instanceof SecurePreciseInv;
      assert spiSrc instanceof SecurePreciseInv;
      if(dbg){
        if(!spiDest.equals(spiSrc))System.out.println("Mismatching values from iterator: Src:" + spiSrc + " Dest: " + spiDest);
        else System.out.println("Matching values from iterator: Src:" + spiSrc + " Dest: " + spiDest);
      }     

      assert spiDest.equals(spiSrc): "Mismatching values from iterator: Src:" + spiSrc + " Dest: " + spiDest;
    }
  }

  private SecureCheckpoint checkpointSyncNodes(SecureCore srcSecureCore, SecureCore destSecureCore, SubscriptionSet ss/*, boolean includeAll*/){
    SecurityFilter srcFilter = srcSecureCore.getSecurityFilter();
    SecurityFilter destFilter = destSecureCore.getSecurityFilter();

    LivenessFilter lf = destFilter.getLivenessFilter().clone();
    lf.union(srcFilter.getLivenessFilter());
    boolean withBody = false;

    SecureCheckpoint chkPt = null;
    
    SecureNewIncomingInvalConnection sic = new SecureNewIncomingInvalConnection(StreamId.makeNewStreamId(), ss, destSecureCore.getCurrentVV(), destSecureCore, new LocalController());
    AcceptVV endVV = srcSecureCore.getCurrentVV().getDiff(destSecureCore.getCurrentVV());
    AcceptVV startVV  = destSecureCore.getCurrentVV().project(endVV);
    if(!startVV.includes(endVV)){
      Filter f = new SubscriptionSetFilter(ss);
      chkPt = srcFilter.getSecureCheckpoint(AcceptVV.incrementAll(startVV), withBody, endVV/*, includeAll*/, lf, f, new FilterKnowledge(startVV, f));    
      chkPt.initExternalDVV();
      destFilter.apply(chkPt, sic, srcSecureCore.getMyNodeId());
    }

    return chkPt;
    
  }
  /*
   * "new TestSuite(Class c)" constructs a test suite
   * containg every method whose name begins with "test"
   * 
   * TBD: update class name
   */
  public static Test suite(){
    TestSuite suite = new TestSuite(CertificateUnit.class);
    return suite;
  }


  /*
   * main() lets us run just this set of unit tests
   * from the comand line (you can also invoke 
   * the testrunner on this class and it will find
   * the suite())
   *
   * TBD: update class name
   */
  public static void main(String s[]) {
    String name = "CertificateUnit";
    System.err.print(name + " self test begins...");
    TestRunner tr = new TestRunner();
    tr.doRun(suite());
    System.err.println(name + " self test succeeds");
  }


}
