package code.serialization;

import java.io.*;

import code.*;
import code.branchDetecting.BranchID;
import code.security.SangminConfig;
import code.security.ahs.DependencyVV;
import code.simulator.Hash;
import code.simulator.HashedVV;
import code.simulator.ProofOfMisbehavior;
import code.simulator.SyncRequest;

import java.util.*;
import java.util.Map.Entry;

public class SerializationHelper{
  ByteArrayOutputStream bs;
  ObjectOutput oos;

  public SerializationHelper() throws Exception{
    bs = new ByteArrayOutputStream(1024);
    oos = new ObjectOutputStream(bs);
  }

  public SerializationHelper(ObjectOutput oos){
    bs = null;
    this.oos = oos;
  }

  public void close(){
    try{
      if(bs != null){
        bs.close();
      }
      oos.close();
    }catch(IOException e){
      // TODO Auto-generated catch block
      e.printStackTrace();
    }
  }

  public byte[] toBytes(){
    try{
      oos.flush();
    }catch(IOException e){
      // TODO Auto-generated catch block
      e.printStackTrace();
    }
    return bs.toByteArray();
  }

  public ObjectOutput getObjectOutputStream(){
    return oos;
  }

  public void writeVV(VV vv) throws IOException{
    List<NodeId> l = new LinkedList<NodeId>(vv.getNodes());
    Collections.sort(l);
    ListIterator<NodeId> iter;
    try{
      for(NodeId nodeId: l){
        oos.writeLong(nodeId.getIDint());
        oos.writeLong(vv.getStampByServer(nodeId));                                
      }
    }catch(NoSuchEntryException e){
      e.printStackTrace();
      System.exit(1);
    }
  }


  public void writeHDVV(HashedVV vv) throws IOException{
    Collection<Entry<NodeId, Hash>> sortedList = vv.getHashes();
    oos.writeInt(sortedList.size());
    try{
      for(Entry<NodeId, Hash> n: sortedList){
        oos.writeLong(n.getKey().getIDint());
        oos.writeLong(vv.getStampByServer(n.getKey()));
        oos.write(n.getValue().getHashVal());
      }
    }catch(NoSuchEntryException e){
      // TODO Auto-generated catch block
      e.printStackTrace();
      assert false;
    }
  }

  /**
   * Takes a serializable object and returns a byte representation
   * @param o
   * @return
   * @throws Exception 
   */
  public static byte[] serialize(Object o){
    byte[] ret = null;
    try{
      ByteArrayOutputStream bs = new ByteArrayOutputStream();
      ObjectOutputStream oos = new ObjectOutputStream(bs);
      oos.writeObject(o);
      oos.flush();
      ret = bs.toByteArray();
      bs.close();
      oos.close();
      return ret;
    } catch (Exception e){
      e.printStackTrace();
    }
    assert false;
    return null;
  }

  /**
   * takes a byte stream and returns an object--expected that the serializaed 
   * object was produced by the invocation of getNonDeterministicBytes method
   * @param bytes
   * @return
   * @throws IOException 
   * @throws ClassNotFoundException 
   */
  public static Object deserilize(byte[] bytes){
    try{
      ObjectInputStream ois = new ObjectInputStream(new ByteArrayInputStream(bytes));
      Object ret = ois.readObject();
      ois.close();
      return ret;
    } catch (Exception e){
      e.printStackTrace();
    }
    assert false;
    return null;
  }

  public static void writeShortString(String str, ObjectOutput out) throws IOException{
    if(str == null){
      out.writeShort(0);
      return;
    }
    byte[] bytes = str.getBytes();
    out.writeShort(bytes.length);
    if(bytes.length > 0){
      out.write(bytes);
    }
  }

  public static String readShortString(ObjectInput in) throws IOException{
    short len = in.readShort();
    if(len > 0){
      byte[] bytes = new byte[len]; 
      in.readFully(bytes);
      return new String(bytes);
    }else{
      return null;
    }
  }

  
  public static void writeShortString(String str, OutputStream out) throws IOException{
    if(str == null){
      writeShort(out, (short)0);
      return;
    }
    byte[] bytes = str.getBytes();
    writeShort(out, (short)bytes.length);
    if(bytes.length > 0){
      SerializationHelper.write(out, bytes);
    }
  }

  public static String readShortString(InputStream in) throws IOException{
    short len = readShort(in);
    if(len > 0){
      byte[] bytes = new byte[len]; 
      SerializationHelper.readFully(in, bytes);
      return new String(bytes);
    }else{
      return null;
    }
  }

  public static void writeVV(ObjectOutput oos, AcceptVV vv) throws IOException{
    oos.writeShort(vv.getSize());
    for(AcceptStamp as: vv.getAllStamps()){
      as.writeExternal(oos);
    }
  }
  
  public static void writeVV(IrisOutputStream oos, AcceptVV vv) throws IOException{
    oos.writeShort(vv.getSize());
    if(!SangminConfig.compressVV){
      for(AcceptStamp as: vv.getAllStamps()){
        as.writeToStream(oos, false);
      }
    }else{
      if(vv.size() > 0){
        long minTS = Long.MAX_VALUE;
        Collection<AcceptStamp> stamps = vv.getAllStamps();
        for(AcceptStamp as: stamps){
          minTS = (minTS > as.getLocalClock())?as.getLocalClock():minTS;
        }
        for(AcceptStamp as: stamps){
          AcceptStamp newAS = new AcceptStamp(as.getLocalClock()-minTS, as.getNodeId());
          newAS.writeToStream(oos, false);
        }
        oos.writeLong(minTS);
      }
    }
  }

  

  public static AcceptVV readVV(ObjectInput ois) throws IOException, ClassNotFoundException{
    short size = ois.readShort();
    AcceptStamp[] stamps = new AcceptStamp[size];
    for(int i = 0; i < size; i++){
      stamps[i] = new AcceptStamp();
      stamps[i].readExternal(ois);
    }
    return new AcceptVV(stamps);
  }
  
  public static AcceptVV readVV(IrisInputStream ois) throws IOException{
    short size = ois.readShort();
    if(size == 0){
      return new AcceptVV();
    }
    assert size > 0:size;
    AcceptStamp[] stamps = new AcceptStamp[size];
    for(int i = 0; i < size; i++){
      stamps[i] = new AcceptStamp();
      stamps[i].readFromStream(ois, false);
    }
    if(SangminConfig.compressVV){
      long minTS = ois.readLong();
      for(int i = 0; i < stamps.length; i++){
        AcceptStamp newAS = new AcceptStamp(stamps[i].getLocalClock()+minTS, stamps[i].getNodeId());
      }
    }
    return new AcceptVV(stamps);
  }


  public static void putBoolean(byte[] b, int off, boolean val) {
    b[off] = (byte) (val ? 1 : 0);
  }
  
  public static int writeBoolean(OutputStream os, boolean val) throws IOException {
    os.write((byte) (val ? 1 : 0));
    return 1;
  }

  public static void putChar(byte[] b, int off, char val) {
    b[off + 1] = (byte) (val >>> 0);
    b[off + 0] = (byte) (val >>> 8);
  }

  public static void putShort(byte[] b, int off, short val) {
    b[off + 1] = (byte) (val >>> 0);
    b[off + 0] = (byte) (val >>> 8);
  }
  
  public static int writeShort(OutputStream os, short val) throws IOException {
    os.write((byte) (val >>> 0));
    os.write((byte) (val >>> 8));
    return 2;
  }

  public static void putInt(byte[] b, int off, int val) {
    b[off + 3] = (byte) (val >>> 0);
    b[off + 2] = (byte) (val >>> 8);
    b[off + 1] = (byte) (val >>> 16);
    b[off + 0] = (byte) (val >>> 24);
  }
  
  public static int writeInt(OutputStream os, int val) throws IOException {
    os.write((byte) (val >>> 0));
    os.write((byte) (val >>> 8));
    os.write((byte) (val >>> 16));
    os.write((byte) (val >>> 24));
    return 4;
  }
  
  public static byte[] getBytesFromInt(int i){
    byte[] b = new byte[4];
    putInt(b, 0, i);
    return b;
  }

  public static void putFloat(byte[] b, int off, float val) {
    int i = Float.floatToIntBits(val);
    b[off + 3] = (byte) (i >>> 0);
    b[off + 2] = (byte) (i >>> 8);
    b[off + 1] = (byte) (i >>> 16);
    b[off + 0] = (byte) (i >>> 24);
  }

  public static void putLong(byte[] b, int off, long val) {
    b[off + 7] = (byte) (val >>> 0);
    b[off + 6] = (byte) (val >>> 8);
    b[off + 5] = (byte) (val >>> 16);
    b[off + 4] = (byte) (val >>> 24);
    b[off + 3] = (byte) (val >>> 32);
    b[off + 2] = (byte) (val >>> 40);
    b[off + 1] = (byte) (val >>> 48);
    b[off + 0] = (byte) (val >>> 56);
  }
  
  public static int writeLong(OutputStream os, long val) throws IOException {
    os.write((byte) (val >>> 0));
    os.write((byte) (val >>> 8));
    os.write((byte) (val >>> 16));
    os.write((byte) (val >>> 24));
    os.write((byte) (val >>> 32));
    os.write((byte) (val >>> 40));
    os.write((byte) (val >>> 48));
    os.write((byte) (val >>> 56));
    return 8;
  }

  public static void putDouble(byte[] b, int off, double val) {
    long j = Double.doubleToLongBits(val);
    b[off + 7] = (byte) (j >>> 0);
    b[off + 6] = (byte) (j >>> 8);
    b[off + 5] = (byte) (j >>> 16);
    b[off + 4] = (byte) (j >>> 24);
    b[off + 3] = (byte) (j >>> 32);
    b[off + 2] = (byte) (j >>> 40);
    b[off + 1] = (byte) (j >>> 48);
    b[off + 0] = (byte) (j >>> 56);
  }

  public static short getShort(byte[] b, int off) {
    return (short) (((b[off + 1] & 0xFF) << 0) + 
        ((b[off + 0]) << 8));
  }
  
  public static short readShort(InputStream is) throws IOException {
    return (short) (((is.read() & 0xFF) << 0) + 
        ((is.read()) << 8));
  }
  
  public static short getShort(InputStream is) throws IOException {
    return (short) (((is.read() & 0xFF) << 0) + 
        ((is.read()) << 8));
  }

  public static int getInt(byte[] b, int off) {
    return ((b[off + 3] & 0xFF) << 0) +
    ((b[off + 2] & 0xFF) << 8) +
    ((b[off + 1] & 0xFF) << 16) +
    ((b[off + 0]) << 24);
  }
  
  public static int readInt(InputStream is) throws IOException {
    return ((is.read() & 0xFF) << 0) +
    ((is.read() & 0xFF) << 8) +
    ((is.read() & 0xFF) << 16) +
    ((is.read()) << 24);
  }

  public static float getFloat(byte[] b, int off) {
    int i = ((b[off + 3] & 0xFF) << 0) +
    ((b[off + 2] & 0xFF) << 8) +
    ((b[off + 1] & 0xFF) << 16) +
    ((b[off + 0]) << 24);
    return Float.intBitsToFloat(i);
  }

  public static long getLong(byte[] b, int off) {
    return ((b[off + 7] & 0xFFL) << 0) +
    ((b[off + 6] & 0xFFL) << 8) +
    ((b[off + 5] & 0xFFL) << 16) +
    ((b[off + 4] & 0xFFL) << 24) +
    ((b[off + 3] & 0xFFL) << 32) +
    ((b[off + 2] & 0xFFL) << 40) +
    ((b[off + 1] & 0xFFL) << 48) +
    (((long) b[off + 0]) << 56);
  }
  
  public static long readLong(InputStream is) throws IOException{
    return ((is.read() & 0xFFL) << 0) +
    ((is.read() & 0xFFL) << 8) +
    ((is.read() & 0xFFL) << 16) +
    ((is.read() & 0xFFL) << 24) +
    ((is.read() & 0xFFL) << 32) +
    ((is.read() & 0xFFL) << 40) +
    ((is.read() & 0xFFL) << 48) +
    (((long) is.read()) << 56);
  }

  public static double getDouble(byte[] b, int off) {
    long j = ((b[off + 7] & 0xFFL) << 0) +
    ((b[off + 6] & 0xFFL) << 8) +
    ((b[off + 5] & 0xFFL) << 16) +
    ((b[off + 4] & 0xFFL) << 24) +
    ((b[off + 3] & 0xFFL) << 32) +
    ((b[off + 2] & 0xFFL) << 40) +
    ((b[off + 1] & 0xFFL) << 48) +
    (((long) b[off + 0]) << 56);
    return Double.longBitsToDouble(j);
  }
  public static boolean getBoolean(byte[] b, int off) {
    return b[off] != 0;
  }
  
  public static boolean readBoolean(InputStream is) throws IOException{
    return is.read() != 0;
  }

  public static char getChar(byte[] b, int off) {
    return (char) (((b[off + 1] & 0xFF) << 0) + 
        ((b[off + 0]) << 8));
  }
  
  public static void readFully(InputStream is, byte[] buf) throws IOException{
    int nRead = 0;
    while(nRead < buf.length){
      nRead += is.read(buf, nRead, (buf.length-nRead));
    }
  }
  
  public static int write(OutputStream is, byte[] buf) throws IOException{
    is.write(buf);
    return buf.length;
  }

  public static void main(String[] args) throws IOException{

    DependencyVV dvv = new DependencyVV();
    dvv.put(new BranchID(5), 3);
    dvv.put(new BranchID(6), 4);
    AcceptVV avv = new AcceptVV(dvv); 

    ByteArrayOutputStream bs;
    ObjectOutputStream oos;

    bs = new ByteArrayOutputStream();
    oos = new IrisObjectOutputStream(bs);
    System.out.println("Serialise it");
    //    oos.writeObject(sspi);
    avv.writeExternal(oos);

    oos.flush();
    //bs.toByteArray();
    System.out.println("read it " + bs.toByteArray().length + "\n" + Arrays.toString(bs.toByteArray()));
    AcceptVV avv1 = null;
    ObjectInputStream ois = new IrisObjectInputStream(new ByteArrayInputStream(bs.toByteArray()));
    try{
      //      sspi2 = (SecureSimPreciseInv)ois.readObject();
      avv1 = new AcceptVV();
      avv1.readExternal(ois);
    }catch(ClassNotFoundException e){
      e.printStackTrace();
    }

    if(avv.equals(avv1)){
      System.out.println("Success");
    } else {
      System.out.println("Fail : \n" + avv + "\n" + avv1);

    }

    //    sspi.replaceIrisObject(new IrisHashObject(sspi.getData().getHash()));
    //    assert sspi.generateMyHash().equals(sspi2.generateMyHash());

  }

  public static byte readByte(InputStream is) throws IOException{
    return (byte)is.read();
  }
  
  public static int writeByte(OutputStream os, byte b) throws IOException{
    os.write(b);
    return 1;
  }

  public static byte[] getBufWithLength(IrisSerializable is) throws IOException {
	  ByteArrayOutputStream bs = new ByteArrayOutputStream(40);
	  IrisOutputStream ios = new IrisOutputStream(bs);
	  is.writeToStream(ios, true);
	  ios.flush();
	  return SerializationHelper.getBufWithLength(bs.toByteArray());
  }
  
  public static byte[] getBufWithLength(byte[] byteArray) {
	  byte[] bigbuf = new byte[4+byteArray.length];
	  System.arraycopy(SerializationHelper.getBytesFromInt(byteArray.length), 0, bigbuf, 0, 4);
	  System.arraycopy(byteArray, 0, bigbuf, 4, byteArray.length);
	  return bigbuf;
  }
  
  public static byte[] getBufWithoutLength(InputStream is) throws IOException{
	  byte[] intA = new byte[4];
      SerializationHelper.readFully(is, intA);
      int size = SerializationHelper.getInt(intA, 0);
      byte []buf = new byte[size];
      SerializationHelper.readFully(is, buf);
      return buf;
  }
}
