package code.simulator;
import java.io.*;
import java.lang.reflect.AccessibleObject;
import java.lang.reflect.Field;
import java.security.*;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Random;

import code.AcceptVV;
import code.Immutable;
import code.NodeId;
import code.ResultStats;
import code.branchDetecting.BranchID;
import code.lasr.db.DbException;
import code.security.SangminConfig;
import code.security.ahs.DependencyVV;
import code.serialization.IrisOutputStream;
import code.serialization.IrisSerializable;
import code.serialization.SerializationHelper;
import code.simulator.persistentLog.PersistentStore;
/**
 * Reside in an InvalListItem and used to calculate a summary hash
 **/
public class Hash implements Externalizable, Comparable<Hash>, Immutable, Obj2Bytes{
  public final static Hash NullHash;
  public final static String ALGORITHM = "SHA-256";
  public static final int Length = 32;
  public static final boolean trackHashCost = false;

  static{
    byte[] _nullHash = new byte[Length];
    for(int i=0; i < Length; i++){
      _nullHash[i] = 0;
    }
    NullHash = new Hash(_nullHash, false);
  }

  private final byte [] digest;

  public Hash(){
    digest = null;
  }

  public static Hash createHash(IrisSerializable spi){
    ByteArrayOutputStream buffer = new ByteArrayOutputStream(100);

    assert(spi != null);
    byte[] buf = null;
    byte[] d = null;
    
    long start;
    if(trackHashCost)
	start= System.nanoTime();
    try {
      IrisOutputStream ios = new IrisOutputStream(buffer);
      spi.writeToStream(ios, true);
      ios.flush();
      buf = buffer.toByteArray();
      d = getHashHelper(buf);
    } catch (Exception e){
      System.out.println(e.toString());
      e.printStackTrace();
      System.exit(1);
      assert false;
    }
    if(trackHashCost){
	long end = System.nanoTime();
	System.out.println(" creating Hash of size " + buf.length + "   took " + ((double)(end-start))/1000000);
    }
        
    return new Hash(d, false);
  }
  
  private static byte[] getHashHelper(byte[] buf) throws NoSuchAlgorithmException{
    byte []d = null;
    if(!SangminConfig.useHash){
      // create a byte array after paddding/truncating the 
      // 
      d = new byte[Hash.Length];
      if(buf.length > Hash.Length){
        System.arraycopy(buf, 0, d, 0, Hash.Length);
      }else{
        System.arraycopy(buf, 0, d, 0, buf.length);
        for(int i = buf.length; i < Hash.Length; i++){
          d[i]=(byte)i;
        }
      }
    }else{
      MessageDigest md = MessageDigest.getInstance(ALGORITHM);
      d = md.digest(buf);
    }
    return d;
  }

  public Hash(Obj2Bytes spi){
    assert(spi != null);
    long start;
    if(trackHashCost)
	start= System.nanoTime();
    

    byte[] d = null;
    byte[] buf = spi.obj2Bytes();
    try {
      d = getHashHelper(buf);
    } catch (Exception e){
      System.out.println(e.toString());
      e.printStackTrace();
      System.exit(1);
      assert false;
    }
    this.digest = d;
    if(trackHashCost){
	long end = System.nanoTime();
	System.out.println(" creating Hash of size " + buf.length + "   took " + ((double)(end-start))/1000000);
    }
  }

  public Hash(byte[] data, boolean compute){
    byte[] d = null;
    long start;
    if(trackHashCost)
	start= System.nanoTime();
    
    if(compute){
      try {
        d = getHashHelper(data);
      } catch (Exception e){
        System.out.println(e.toString());
        e.printStackTrace();
        System.exit(1);
        assert false;
      }
    }else{
      d = new byte[Length];
      assert data.length == Length;
      for(int i = 0; i < Length; i++){
        d[i] = data[i];
      }
    }
    this.digest = d;
    if(trackHashCost){
	long end = System.nanoTime();
	System.out.println(" creating Hash of size " + data.length + "   took " + ((double)(end-start))/1000000);
    }
  }

  public final byte[] getHashVal(){
    return this.digest;
  }

//  public int hashCode(){
//    {
//      int code = 0;
//      final int len = digest.length;
//      for ( int i=0; i<len; i++ )
//      {
//        code ^= digest[ i ];
//      }
//      return code;
//    }
//  }

//  public boolean equals(Object o){
//    if(o instanceof Hash && MessageDigest.isEqual(this.digest, ((Hash)o).getHashVal())){
//      return true;
//    }
//    return false;
//  }

  public String toString(){
    //    return "Hash" + Arrays.toString(this.digest);
    return toShortString();
  }

  public String toLongString(){
    return "Hash" + Arrays.toString(this.digest);
    //  return toShortString();
  }

  public String toShortString(){
    byte[] small = new byte[3];
    small[0] = this.digest[0];
    small[1] = this.digest[1];
    small[2] = this.digest[2];
    return "Hash" + Arrays.toString(small).replaceAll("\\s","");
  }

  public int compareTo(Hash o) {
    Hash h = (Hash)o;
    final int len = digest.length;
    for ( int i=0; i<len; i++ )
    {
      if(this.digest[i] < h.digest[i]){
        return -1;
      }else if(this.digest[i] > h.digest[i]){
        return 1;
      }
    }
    return 0;
  }

  /** 
   *  Serialization -- optimization to reduce the pikles in the serialization  
   *                   and improve the performance -- removed from  
   *                   TaggedOutputStream to here 
   **/ 
  public void writeExternal(ObjectOutput out)
  throws IOException{
    out.write(this.getHashVal());
  }


  /** 
   *  Serialization --  optimization to reduce the pikles in the serialization  
   *                   and improve the performance --originated from  
   *                   TaggedInputStream 
   **/ 
  public void readExternal(ObjectInput in)
  throws IOException{

    //AcceptStamp[] vvTime = new AcceptStamp[avvSize];
    byte [] hashVal  = new byte[Length];
    in.readFully(hashVal);

    Field[] f = new Field[1];      
    try{

      f[0] = this.getClass().getDeclaredField("digest");

    }catch(NoSuchFieldException ne){
      ne.printStackTrace();
      assert false:ne;
    }
    try{
      AccessibleObject.setAccessible(f, true);
    } catch (SecurityException se){
      assert false;
    }
    try{
      f[0].set(this, hashVal);

    }catch(IllegalArgumentException ie){
      ie.printStackTrace();
      assert false;
    }catch(IllegalAccessException iae){
      assert false;
    }

    try{
      AccessibleObject.setAccessible(f, false);
    } catch (SecurityException se){
      assert false;
    }
  }

  public byte[] obj2Bytes(){
    // TODO Auto-generated method stub
    return this.getHashVal();
  }

  public static void main(String[] args) throws IOException, DbException{
    int size = -1;
    if(args.length > 0){
      size = Integer.parseInt(args[0]);
    }
    System.out.println(" size: "  + size);
    byte[] testD = null;
    if(size != -1){
      testD = new byte[size];
      for(int c = 0; c < size; c++){
        testD[c] = (byte)(c%128);
      }
    }

    ResultStats stats = new ResultStats();

    long stime = System.currentTimeMillis();
    for(int i = 0; i < 1000; i++){
      long stime1 = System.currentTimeMillis();

      testD[0]=(byte)i;
      testD[1]=(byte)(i>>1);
      testD[2]=(byte)(i>>1);
      Hash h = new Hash(testD, true);
      long etime1 = System.currentTimeMillis();
      stats.enter(etime1-stime1);
    }

    System.out.println("###### Signning time!!");
    System.out.println("mean0" + stats.getMean(0));
    System.out.println("mean1" + stats.getMean(1));
    System.out.println(stats.getAverage90());
    System.out.println("std0" + stats.getStandardDeviation(0));
    System.out.println("std1" +stats.getStandardDeviation(1));


    long etime = System.currentTimeMillis();
    System.out.println("1000 hashes took " + (etime-stime));
  }

  /* (non-Javadoc)
   * @see java.lang.Object#hashCode()
   */
  @Override
  public int hashCode(){
    final int prime = 31;
    int result = 1;
    result = prime * result + Arrays.hashCode(digest);
    return result;
  }

  /* (non-Javadoc)
   * @see java.lang.Object#equals(java.lang.Object)
   */
  @Override
  public boolean equals(Object obj){
    if(this == obj){
      return true;
    }
    if(obj == null){
      return false;
    }
    if(!(obj instanceof Hash)){
      return false;
    }
    Hash other = (Hash) obj;
    if(!Arrays.equals(digest, other.digest)){
      return false;
    }
    return true;
  }

}
